"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrEksCluster = exports.EmrVersion = exports.Autoscaler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const aws_emrcontainers_1 = require("aws-cdk-lib/aws-emrcontainers");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const SimpleBase = require("simple-base");
const ara_bucket_1 = require("../ara-bucket");
const context_options_1 = require("../common/context-options");
const tracked_construct_1 = require("../common/tracked-construct");
const singleton_kms_key_1 = require("../singleton-kms-key");
const emr_managed_endpoint_1 = require("./emr-managed-endpoint");
const CriticalDefaultConfig = require("./resources/k8s/emr-eks-config/critical.json");
const NotebookDefaultConfig = require("./resources/k8s/emr-eks-config/notebook-pod-template-ready.json");
const SharedDefaultConfig = require("./resources/k8s/emr-eks-config/shared.json");
const K8sRoleBinding = require("./resources/k8s/rbac/emr-containers-role-binding.json");
const K8sRole = require("./resources/k8s/rbac/emr-containers-role.json");
const emr_eks_cluster_helpers_1 = require("./emr-eks-cluster-helpers");
const singleton_launch_template_1 = require("../singleton-launch-template");
const emr_eks_nodegroup_asg_tag_1 = require("./emr-eks-nodegroup-asg-tag");
const emr_eks_job_template_1 = require("./emr-eks-job-template");
/**
 * The different autoscaler available with EmrEksCluster
 */
var Autoscaler;
(function (Autoscaler) {
    Autoscaler["KARPENTER"] = "KARPENTER";
    Autoscaler["CLUSTER_AUTOSCALER"] = "CLUSTER_AUTOSCALER";
})(Autoscaler = exports.Autoscaler || (exports.Autoscaler = {}));
/**
 * The different EMR versions available on EKS
 */
var EmrVersion;
(function (EmrVersion) {
    EmrVersion["V_10"] = "emr-6.10.0-latest";
    EmrVersion["V6_9"] = "emr-6.9.0-latest";
    EmrVersion["V6_8"] = "emr-6.8.0-latest";
    EmrVersion["V6_7"] = "emr-6.7.0-latest";
    EmrVersion["V6_6"] = "emr-6.6.0-latest";
    EmrVersion["V6_5"] = "emr-6.5.0-latest";
    EmrVersion["V6_4"] = "emr-6.4.0-latest";
    EmrVersion["V6_3"] = "emr-6.3.0-latest";
    EmrVersion["V6_2"] = "emr-6.2.0-latest";
    EmrVersion["V5_33"] = "emr-5.33.0-latest";
    EmrVersion["V5_32"] = "emr-5.32.0-latest";
})(EmrVersion = exports.EmrVersion || (exports.EmrVersion = {}));
/**
 * EmrEksCluster Construct packaging all the resources and configuration required to run Amazon EMR on EKS.
 * It deploys:
 * * An EKS cluster (VPC configuration can be customized)
 * * A tooling nodegroup to run tools including the Kubedashboard and the Cluster Autoscaler
 * * Optionally multiple nodegroups (one per AZ) for critical/shared/notebook EMR workloads
 * * Additional nodegroups can be configured
 *
 * The construct will upload on S3 the Pod templates required to run EMR jobs on the default nodegroups.
 * It will also parse and store the configuration of EMR on EKS jobs for each default nodegroup in object parameters
 *
 * Methods are available to add EMR Virtual Clusters to the EKS cluster and to create execution roles for the virtual clusters.
 *
 * Usage example:
 *
 * ```typescript
 * const emrEks: EmrEksCluster = EmrEksCluster.getOrCreate(stack, {
 *   eksAdminRoleArn: <ROLE_ARN>,
 *   eksClusterName: <CLUSTER_NAME>,
 * });
 *
 * const virtualCluster = emrEks.addEmrVirtualCluster(stack, {
 *   name: <Virtual_Cluster_Name>,
 *   createNamespace: <TRUE OR FALSE>,
 *   eksNamespace: <K8S_namespace>,
 * });
 *
 * const role = emrEks.createExecutionRole(stack, 'ExecRole',{
 *   policy: <POLICY>,
 * })
 *
 * // EMR on EKS virtual cluster ID
 * cdk.CfnOutput(self, 'VirtualClusterId',value = virtualCluster.attr_id)
 * // Job config for each nodegroup
 * cdk.CfnOutput(self, "CriticalConfig", value = emrEks.criticalDefaultConfig)
 * cdk.CfnOutput(self, "SharedConfig", value = emrEks.sharedDefaultConfig)
 * // Execution role arn
 * cdk.CfnOutput(self,'ExecRoleArn', value = role.roleArn)
 * ```
 *
 */
class EmrEksCluster extends tracked_construct_1.TrackedConstruct {
    /**
     * Constructs a new instance of the EmrEksCluster construct.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps}
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingCode: context_options_1.ContextOptions.EMR_EKS_TRACKING_ID,
        };
        super(scope, id, trackedConstructProps);
        this.clusterName = props.eksClusterName ?? EmrEksCluster.DEFAULT_CLUSTER_NAME;
        //Define EKS cluster logging
        const eksClusterLogging = [
            aws_eks_1.ClusterLoggingTypes.API,
            aws_eks_1.ClusterLoggingTypes.AUTHENTICATOR,
            aws_eks_1.ClusterLoggingTypes.SCHEDULER,
            aws_eks_1.ClusterLoggingTypes.CONTROLLER_MANAGER,
            aws_eks_1.ClusterLoggingTypes.AUDIT,
        ];
        //Set the autoscaler mechanism flag
        this.isKarpenter = props.autoscaling == Autoscaler.KARPENTER ? true : false;
        this.defaultNodes = props.defaultNodes == undefined ? true : props.defaultNodes;
        // Create a role to be used as instance profile for nodegroups
        this.ec2InstanceNodeGroupRole = new aws_iam_1.Role(this, 'ec2InstanceNodeGroupRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
        });
        //attach policies to the role to be used by the nodegroups
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        // Create the custom resource provider for tagging the EC2 Auto Scaling groups
        this.nodegroupAsgTagsProviderServiceToken = new emr_eks_nodegroup_asg_tag_1.EmrEksNodegroupAsgTagProvider(this, 'AsgTagProvider', {
            eksClusterName: this.clusterName,
        }).provider.serviceToken;
        // Create the custom resource provider for tagging the EC2 Auto Scaling groups
        this.jobTemplateProviderToken = new emr_eks_job_template_1.EmrEksJobTemplateProvider(this, 'jobTemplateProvider').provider.serviceToken;
        // create an Amazon EKS CLuster with default parameters if not provided in the properties
        if (props.eksCluster == undefined) {
            this.eksCluster = new aws_eks_1.Cluster(scope, `${this.clusterName}Cluster`, {
                defaultCapacity: 0,
                clusterName: this.clusterName,
                version: props.kubernetesVersion || EmrEksCluster.DEFAULT_EKS_VERSION,
                clusterLogging: eksClusterLogging,
                kubectlLayer: props.kubectlLambdaLayer ?? undefined,
            });
            //Create VPC flow log for the EKS VPC
            let eksVpcFlowLogLogGroup = new aws_logs_1.LogGroup(this, 'eksVpcFlowLogLogGroup', {
                logGroupName: `/aws/emr-eks-vpc-flow/${this.clusterName}`,
                encryptionKey: singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey'),
                retention: aws_logs_1.RetentionDays.ONE_WEEK,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            });
            //Allow vpc flowlog to access KMS key to encrypt logs
            singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey').addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                principals: [new aws_iam_1.ServicePrincipal(`logs.${aws_cdk_lib_1.Stack.of(this).region}.amazonaws.com`)],
                actions: [
                    'kms:Encrypt*',
                    'kms:Decrypt*',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:Describe*',
                ],
                conditions: {
                    ArnLike: {
                        'kms:EncryptionContext:aws:logs:arn': `arn:aws:logs:${aws_cdk_lib_1.Stack.of(this).region}:${aws_cdk_lib_1.Stack.of(this).account}:*`,
                    },
                },
                resources: ['*'],
            }));
            //Setup the VPC flow logs
            const iamRoleforFlowLog = new aws_iam_1.Role(this, 'iamRoleforFlowLog', {
                assumedBy: new aws_iam_1.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
            this.eksCluster.vpc.addFlowLog('eksVpcFlowLog', {
                destination: aws_ec2_1.FlowLogDestination.toCloudWatchLogs(eksVpcFlowLogLogGroup, iamRoleforFlowLog),
            });
            //Setting up the cluster with the required controller
            emr_eks_cluster_helpers_1.eksClusterSetup(this, this, props.eksAdminRoleArn);
            //Deploy the right autoscaler using the flag set earlier 
            if (this.isKarpenter) {
                this.karpenterChart = emr_eks_cluster_helpers_1.karpenterSetup(this.eksCluster, this.clusterName, this, props.karpenterVersion || EmrEksCluster.DEFAULT_KARPENTER_VERSION);
            }
            else {
                const kubernetesVersion = props.kubernetesVersion ?? EmrEksCluster.DEFAULT_EKS_VERSION;
                emr_eks_cluster_helpers_1.clusterAutoscalerSetup(this.eksCluster, this.clusterName, this, kubernetesVersion);
            }
        }
        else {
            //Initialize with the provided EKS Cluster
            this.eksCluster = props.eksCluster;
        }
        //Check if the user want to use the default nodegroup and
        //Add the default nodegroup configured and optimized to run Spark workloads
        if (this.defaultNodes && props.autoscaling == Autoscaler.CLUSTER_AUTOSCALER) {
            emr_eks_cluster_helpers_1.setDefaultManagedNodeGroups(this);
        }
        //Check if there user want to use the default Karpenter provisioners and
        //Add the defaults pre-configured and optimized to run Spark workloads
        if (this.defaultNodes && props.autoscaling == Autoscaler.KARPENTER) {
            emr_eks_cluster_helpers_1.setDefaultKarpenterProvisioners(this);
        }
        ara_bucket_1.AraBucket.getOrCreate(this, { bucketName: 's3-access-logs' });
        // Tags the Amazon VPC and Subnets of the Amazon EKS Cluster
        aws_cdk_lib_1.Tags.of(this.eksCluster.vpc).add('for-use-with-amazon-emr-managed-policies', 'true');
        this.eksCluster.vpc.privateSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('for-use-with-amazon-emr-managed-policies', 'true'));
        this.eksCluster.vpc.publicSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('for-use-with-amazon-emr-managed-policies', 'true'));
        // Create Amazon IAM ServiceLinkedRole for Amazon EMR and add to kubernetes configmap
        // required to add a dependency on the Amazon EMR virtual cluster
        this.emrServiceRole = new aws_iam_1.CfnServiceLinkedRole(this, 'EmrServiceRole', {
            awsServiceName: 'emr-containers.amazonaws.com',
        });
        this.eksCluster.awsAuth.addRoleMapping(aws_iam_1.Role.fromRoleArn(this, 'ServiceRoleForAmazonEMRContainers', `arn:aws:iam::${aws_cdk_lib_1.Stack.of(this).account}:role/AWSServiceRoleForAmazonEMRContainers`), {
            username: 'emr-containers',
            groups: ['']
        });
        // Create an Amazon S3 Bucket for default podTemplate assets
        this.assetBucket = ara_bucket_1.AraBucket.getOrCreate(this, { bucketName: `${this.clusterName.toLowerCase()}-emr-eks-assets`, encryption: aws_s3_1.BucketEncryption.KMS_MANAGED });
        // Configure the podTemplate location
        this.podTemplateLocation = {
            bucketName: this.assetBucket.bucketName,
            objectKey: `${this.clusterName}/pod-template`,
        };
        aws_cdk_lib_1.Tags.of(this.assetBucket).add('for-use-with', 'cdk-analytics-reference-architecture');
        let s3DeploymentLambdaPolicyStatement = [];
        s3DeploymentLambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        //Policy to allow lambda access to cloudwatch logs
        const lambdaExecutionRolePolicy = new aws_iam_1.ManagedPolicy(this, 's3BucketDeploymentPolicy', {
            statements: s3DeploymentLambdaPolicyStatement,
            description: 'Policy used by S3 deployment cdk construct',
        });
        //Create an execution role for the lambda and attach to it a policy formed from user input
        this.assetUploadBucketRole = new aws_iam_1.Role(this, 's3BucketDeploymentRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            description: 'Role used by S3 deployment cdk construct',
            managedPolicies: [lambdaExecutionRolePolicy],
        });
        // Upload the default podTemplate to the Amazon S3 asset bucket
        this.uploadPodTemplate('defaultPodTemplates', path_1.join(__dirname, 'resources/k8s/pod-template'));
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the notebook default config
        NotebookDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-driver.yaml`);
        NotebookDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-executor.yaml`);
        this.notebookDefaultConfig = JSON.parse(JSON.stringify(NotebookDefaultConfig));
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the critical default config
        CriticalDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-driver.yaml`);
        CriticalDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-executor.yaml`);
        this.criticalDefaultConfig = JSON.stringify(CriticalDefaultConfig);
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the shared default config
        SharedDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-driver.yaml`);
        SharedDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-executor.yaml`);
        this.sharedDefaultConfig = JSON.stringify(SharedDefaultConfig);
        // Set the custom resource provider service token here to avoid circular dependencies
        this.managedEndpointProviderServiceToken = new emr_managed_endpoint_1.EmrManagedEndpointProvider(this, 'ManagedEndpointProvider', {
            assetBucket: this.assetBucket,
        }).provider.serviceToken;
        // Provide the podTemplate location on Amazon S3
        new aws_cdk_lib_1.CfnOutput(this, 'podTemplateLocation', {
            description: 'Use podTemplates in Amazon EMR jobs from this Amazon S3 Location',
            value: this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}`),
        });
    }
    /**
     * Get an existing EmrEksCluster based on the cluster name property or create a new one
     * only one EKS cluster can exist per stack
     * @param {Construct} scope the CDK scope used to search or create the cluster
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps} if created
     */
    static getOrCreate(scope, props) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = props.eksClusterName || EmrEksCluster.DEFAULT_CLUSTER_NAME;
        let emrEksCluster;
        if (stack.node.tryFindChild(id) == undefined) {
            emrEksCluster = new EmrEksCluster(stack, id, props);
        }
        return stack.node.tryFindChild(id) || emrEksCluster;
    }
    /**
     * Add a new Amazon EMR Virtual Cluster linked to Amazon EKS Cluster.
     * @param {Construct} scope of the stack where virtual cluster is deployed
     * @param {EmrVirtualClusterOptions} options the EmrVirtualClusterProps [properties]{@link EmrVirtualClusterProps}
     */
    addEmrVirtualCluster(scope, options) {
        const eksNamespace = options.eksNamespace ?? 'default';
        const regex = /^[a-z0-9]+$/g;
        if (!eksNamespace.match(regex)) {
            throw new Error(`Namespace provided violates the constraints of Namespace naming ${eksNamespace}`);
        }
        const ns = options.createNamespace
            ? this.eksCluster.addManifest(`${options.name}Namespace`, {
                apiVersion: 'v1',
                kind: 'Namespace',
                metadata: { name: eksNamespace },
            })
            : null;
        // deep clone the Role template object and replace the namespace
        const k8sRole = JSON.parse(JSON.stringify(K8sRole));
        k8sRole.metadata.namespace = eksNamespace;
        const role = this.eksCluster.addManifest(`${options.name}Role`, k8sRole);
        role.node.addDependency(this.emrServiceRole);
        if (ns)
            role.node.addDependency(ns);
        // deep clone the Role Binding template object and replace the namespace
        const k8sRoleBinding = JSON.parse(JSON.stringify(K8sRoleBinding));
        k8sRoleBinding.metadata.namespace = eksNamespace;
        const roleBinding = this.eksCluster.addManifest(`${options.name}RoleBinding`, k8sRoleBinding);
        roleBinding.node.addDependency(role);
        const virtCluster = new aws_emrcontainers_1.CfnVirtualCluster(scope, `${options.name}VirtualCluster`, {
            name: options.name,
            containerProvider: {
                id: this.clusterName,
                type: 'EKS',
                info: { eksInfo: { namespace: options.eksNamespace || 'default' } },
            },
            tags: [{
                    key: 'for-use-with',
                    value: 'cdk-analytics-reference-architecture',
                }],
        });
        virtCluster.node.addDependency(roleBinding);
        virtCluster.node.addDependency(this.emrServiceRole);
        if (ns)
            virtCluster.node.addDependency(ns);
        aws_cdk_lib_1.Tags.of(virtCluster).add('for-use-with', 'cdk-analytics-reference-architecture');
        return virtCluster;
    }
    /**
     * Creates a new Amazon EMR managed endpoint to be used with Amazon EMR Virtual Cluster .
     * CfnOutput can be customized.
     * @param {Construct} scope the scope of the stack where managed endpoint is deployed
     * @param {string} id the CDK id for endpoint
     * @param {EmrManagedEndpointOptions} options the EmrManagedEndpointOptions to configure the Amazon EMR managed endpoint
     */
    addManagedEndpoint(scope, id, options) {
        if (options.managedEndpointName.length > 64) {
            throw new Error(`error managed endpoint name length is greater than 64 ${id}`);
        }
        if (this.notebookDefaultConfig == undefined) {
            throw new Error('error empty configuration override is not supported on non-default nodegroups');
        }
        let jsonConfigurationOverrides;
        // TODO this need to be broadended to all possible emr configuration
        // try {
        //   //Check if the configOverride provided by user is valid
        //   let isConfigOverrideValid: boolean = validateSchema(JSON.stringify(configOverrideSchema), options.configurationOverrides);
        //   jsonConfigurationOverrides = isConfigOverrideValid ? options.configurationOverrides : this.notebookDefaultConfig;
        // } catch (error) {
        //   throw new Error(`The configuration override is not valid JSON : ${options.configurationOverrides}`);
        // }
        jsonConfigurationOverrides = options.configurationOverrides ? options.configurationOverrides : this.notebookDefaultConfig;
        // Create custom resource with async waiter until the Amazon EMR Managed Endpoint is created
        const cr = new aws_cdk_lib_1.CustomResource(scope, id, {
            serviceToken: this.managedEndpointProviderServiceToken,
            properties: {
                clusterId: options.virtualClusterId,
                executionRoleArn: options.executionRole.roleArn,
                endpointName: options.managedEndpointName,
                releaseLabel: options.emrOnEksVersion || EmrEksCluster.DEFAULT_EMR_VERSION,
                configurationOverrides: jsonConfigurationOverrides,
            },
        });
        cr.node.addDependency(this.eksCluster);
        return cr;
    }
    /**
   * Add new nodegroups to the cluster for Amazon EMR on EKS. This method overrides Amazon EKS nodegroup options then create the nodegroup.
   * If no subnet is provided, it creates one nodegroup per private subnet in the Amazon EKS Cluster.
   * If NVME local storage is used, the user_data is modified.
   * @param {string} id the CDK ID of the resource
   * @param {EmrEksNodegroupOptions} props the EmrEksNodegroupOptions [properties]{@link EmrEksNodegroupOptions}
   */
    addEmrEksNodegroup(id, props) {
        if (this.isKarpenter) {
            throw new Error(`You can\'t use this method when the autoscaler is set to ${Autoscaler.KARPENTER}`);
        }
        // Get the subnet from Properties or one private subnet for each AZ
        const subnetList = props.subnet ? [props.subnet] : this.eksCluster.vpc.selectSubnets({
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
        }).subnets;
        // Add Amazon SSM agent to the user data
        var userData = [
            'yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm',
            'systemctl enable amazon-ssm-agent',
            'systemctl start amazon-ssm-agent',
        ];
        var launchTemplateName = `EmrEksLaunch-${this.clusterName}`;
        // If the Nodegroup uses NVMe, add user data to configure them
        if (props.mountNvme) {
            userData = userData.concat([
                'INSTANCE_TYPE=$(ec2-metadata -t)',
                'if [[ $INSTANCE_TYPE == *"2xlarge"* ]]; then',
                'DEVICE="/dev/nvme1n1"',
                'mkfs.ext4 $DEVICE',
                'else',
                'yum install -y mdadm',
                'SSD_NVME_DEVICE_LIST=("/dev/nvme1n1" "/dev/nvme2n1")',
                'SSD_NVME_DEVICE_COUNT=${#SSD_NVME_DEVICE_LIST[@]}',
                'RAID_DEVICE=${RAID_DEVICE:-/dev/md0}',
                'RAID_CHUNK_SIZE=${RAID_CHUNK_SIZE:-512}  # Kilo Bytes',
                'FILESYSTEM_BLOCK_SIZE=${FILESYSTEM_BLOCK_SIZE:-4096}  # Bytes',
                'STRIDE=$((RAID_CHUNK_SIZE * 1024 / FILESYSTEM_BLOCK_SIZE))',
                'STRIPE_WIDTH=$((SSD_NVME_DEVICE_COUNT * STRIDE))',
                'mdadm --create --verbose "$RAID_DEVICE" --level=0 -c "${RAID_CHUNK_SIZE}" --raid-devices=${#SSD_NVME_DEVICE_LIST[@]} "${SSD_NVME_DEVICE_LIST[@]}"',
                'while [ -n "$(mdadm --detail "$RAID_DEVICE" | grep -ioE \'State :.*resyncing\')" ]; do',
                'echo "Raid is resyncing.."',
                'sleep 1',
                'done',
                'echo "Raid0 device $RAID_DEVICE has been created with disks ${SSD_NVME_DEVICE_LIST[*]}"',
                'mkfs.ext4 -m 0 -b "$FILESYSTEM_BLOCK_SIZE" -E "stride=$STRIDE,stripe-width=$STRIPE_WIDTH" "$RAID_DEVICE"',
                'DEVICE=$RAID_DEVICE',
                'fi',
                'systemctl stop docker',
                'mkdir -p /var/lib/kubelet/pods',
                'mount $DEVICE /var/lib/kubelet/pods',
                'chmod 750 /var/lib/docker',
                'systemctl start docker',
            ]);
            launchTemplateName = `EmrEksNvmeLaunch-${this.clusterName}`;
        }
        // Add headers and footers to user data
        const userDataMime = aws_cdk_lib_1.Fn.base64(`MIME-Version: 1.0
Content-Type: multipart/mixed; boundary="==MYBOUNDARY=="

--==MYBOUNDARY==
Content-Type: text/x-shellscript; charset="us-ascii"

#!/bin/bash
${userData.join('\r\n')}

--==MYBOUNDARY==--\\
`);
        // Create a new LaunchTemplate or reuse existing one
        const lt = singleton_launch_template_1.SingletonCfnLaunchTemplate.getOrCreate(this, launchTemplateName, userDataMime);
        // Create one Amazon EKS Nodegroup per subnet
        subnetList.forEach((subnet, index) => {
            // Make the ID unique across AZ using the index of subnet in the subnet list
            const resourceId = `${id}-${index}`;
            const nodegroupName = props.nodegroupName ? `${props.nodegroupName}-${index}` : resourceId;
            // Add the user data to the NodegroupOptions
            const nodeGroupParameters = {
                ...props,
                ...{
                    launchTemplateSpec: {
                        id: lt.ref,
                        version: lt.attrLatestVersionNumber,
                    },
                    subnets: {
                        subnets: [subnet],
                    },
                    nodegroupName: nodegroupName,
                },
            };
            // Create the Amazon EKS Nodegroup
            this.addNodegroupCapacity(resourceId, nodeGroupParameters);
        });
    }
    /**
     * Add a new Amazon EKS Nodegroup to the cluster.
     * This method is used to add a nodegroup to the Amazon EKS cluster and automatically set tags based on labels and taints
     *  so it can be used for the cluster autoscaler.
     * @param {string} nodegroupId the ID of the nodegroup
     * @param {EmrEksNodegroupOptions} options the EmrEksNodegroup [properties]{@link EmrEksNodegroupOptions}
     */
    addNodegroupCapacity(nodegroupId, options) {
        const nodegroup = this.eksCluster.addNodegroupCapacity(nodegroupId, options);
        // Adding the Amazon SSM policy
        nodegroup.role.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // Add tags for the Cluster Autoscaler IAM scoping
        aws_cdk_lib_1.Tags.of(nodegroup).add('eks:cluster-name', `${this.clusterName}`);
        // Add tags for the Cluster Autoscaler management
        aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/enabled', 'true', {
            applyToLaunchedInstances: true,
        });
        // Add tag for the AZ
        if (options.subnets && options.subnets.subnets) {
            aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/node-template/label/topology.kubernetes.io/zone', options.subnets.subnets[0].availabilityZone, {
                applyToLaunchedInstances: true,
            });
        }
        // Add tag for the lifecycle type (spot or on-demand)
        aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/node-template/label/eks.amazonaws.com/capacityType', (options.capacityType == aws_eks_1.CapacityType.SPOT) ? 'SPOT' : 'ON_DEMAND', {
            applyToLaunchedInstances: true,
        });
        // Iterate over labels and add appropriate tags
        if (options.labels) {
            for (const [key, value] of Object.entries(options.labels)) {
                aws_cdk_lib_1.Tags.of(nodegroup).add(`k8s.io/cluster-autoscaler/node-template/label/${key}`, value, {
                    applyToLaunchedInstances: true,
                });
                new aws_cdk_lib_1.CustomResource(this, `${nodegroupId}Label${key}`, {
                    serviceToken: this.nodegroupAsgTagsProviderServiceToken,
                    properties: {
                        nodegroupName: options.nodegroupName,
                        tagKey: `k8s.io/cluster-autoscaler/node-template/label/${key}`,
                        tagValue: value,
                    },
                }).node.addDependency(nodegroup);
            }
        }
        // Iterate over taints and add appropriate tags
        if (options.taints) {
            options.taints.forEach((taint) => {
                aws_cdk_lib_1.Tags.of(nodegroup).add(`k8s.io/cluster-autoscaler/node-template/taint/${taint.key}`, `${taint.value}:${taint.effect}`, {
                    applyToLaunchedInstances: true,
                });
                new aws_cdk_lib_1.CustomResource(this, `${nodegroupId}Taint${taint.key}`, {
                    serviceToken: this.nodegroupAsgTagsProviderServiceToken,
                    properties: {
                        nodegroupName: options.nodegroupName,
                        tagKey: `k8s.io/cluster-autoscaler/node-template/taint/${taint.key}`,
                        tagValue: `${taint.value}:${taint.effect}`,
                    },
                }).node.addDependency(nodegroup);
            });
        }
        return nodegroup;
    }
    /**
     * Create and configure a new Amazon IAM Role usable as an execution role.
     * This method makes the created role assumed by the Amazon EKS cluster Open ID Connect provider.
     * @param {Construct} scope of the IAM role
     * @param {string} id of the CDK resource to be created, it should be unique across the stack
     * @param {IManagedPolicy} policy the execution policy to attach to the role
     * @param {string} namespace The namespace from which the role is going to be used. MUST be the same as the namespace of the Virtual Cluster from which the job is submitted
     * @param {string} name Name to use for the role, required and is used to scope the iam role
     */
    createExecutionRole(scope, id, policy, namespace, name) {
        const stack = aws_cdk_lib_1.Stack.of(this);
        let irsaConditionkey = new aws_cdk_lib_1.CfnJson(this, `${id}irsaConditionkey'`, {
            value: {
                [`${this.eksCluster.openIdConnectProvider.openIdConnectProviderIssuer}:sub`]: 'system:serviceaccount:' + namespace + ':emr-containers-sa-*-*-' + aws_cdk_lib_1.Aws.ACCOUNT_ID.toString() + '-' + SimpleBase.base36.encode(name),
            },
        });
        // Create an execution role assumable by EKS OIDC provider
        return new aws_iam_1.Role(scope, `${id}ExecutionRole`, {
            assumedBy: new aws_iam_1.FederatedPrincipal(this.eksCluster.openIdConnectProvider.openIdConnectProviderArn, {
                StringLike: irsaConditionkey,
            }, 'sts:AssumeRoleWithWebIdentity'),
            roleName: name,
            managedPolicies: [policy],
            inlinePolicies: {
                PodTemplateAccess: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                's3:getObject',
                            ],
                            resources: [
                                stack.formatArn({
                                    region: '',
                                    account: '',
                                    service: 's3',
                                    resource: this.podTemplateLocation.bucketName,
                                    resourceName: `${this.podTemplateLocation.objectKey}/*`,
                                }),
                            ],
                        }),
                    ],
                }),
            },
        });
    }
    /**
     * Upload podTemplates to the Amazon S3 location used by the cluster.
     * @param {string} id the unique ID of the CDK resource
     * @param {string} filePath The local path of the yaml podTemplate files to upload
     */
    uploadPodTemplate(id, filePath) {
        new aws_s3_deployment_1.BucketDeployment(this, `${id}AssetDeployment`, {
            destinationBucket: this.assetBucket,
            destinationKeyPrefix: this.podTemplateLocation.objectKey,
            sources: [aws_s3_deployment_1.Source.asset(filePath)],
            role: this.assetUploadBucketRole,
        });
    }
    /**
     * Creates a new Amazon EMR on EKS job template based on the props passed
     * @param {Construct} scope the scope of the stack where job template is created
     * @param {string} id the CDK id for job template resource
     * @param {EmrEksJobTemplateDefinition} options the EmrManagedEndpointOptions to configure the Amazon EMR managed endpoint
     */
    addJobTemplate(scope, id, options) {
        // Create custom resource to execute the create job template boto3 call
        const cr = new aws_cdk_lib_1.CustomResource(scope, id, {
            serviceToken: this.jobTemplateProviderToken,
            properties: {
                name: options.name,
                jobTemplateData: options.jobTemplateData,
            },
        });
        cr.node.addDependency(this.eksCluster);
        return cr;
    }
    /**
     * Apply the provided manifest and add the CDK dependency on EKS cluster
     * @param {string} id the unique ID of the CDK resource
     * @param {any} manifest The manifest to apply.
     * You can use the Utils class that offers method to read yaml file and load it as a manifest
     */
    addKarpenterProvisioner(id, manifest) {
        if (!this.isKarpenter) {
            throw new Error(`You can\'t use this method when the autoscaler is set to ${Autoscaler.KARPENTER}`);
        }
        let manifestApply = this.eksCluster.addManifest(id, ...manifest);
        if (this.karpenterChart) {
            manifestApply.node.addDependency(this.karpenterChart);
        }
        return manifestApply;
    }
}
exports.EmrEksCluster = EmrEksCluster;
_a = JSII_RTTI_SYMBOL_1;
EmrEksCluster[_a] = { fqn: "aws-analytics-reference-architecture.EmrEksCluster", version: "2.9.7" };
EmrEksCluster.DEFAULT_EMR_VERSION = EmrVersion.V6_8;
EmrEksCluster.DEFAULT_EKS_VERSION = aws_eks_1.KubernetesVersion.V1_22;
EmrEksCluster.DEFAULT_CLUSTER_NAME = 'data-platform';
EmrEksCluster.DEFAULT_KARPENTER_VERSION = 'v0.20.0';
//# sourceMappingURL=data:application/json;base64,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