"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataDomain = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const constructs_1 = require("constructs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const central_governance_1 = require("./central-governance");
const data_lake_storage_1 = require("../data-lake-storage");
const data_domain_nrac_workflow_1 = require("./data-domain-nrac-workflow");
const data_domain_tbac_workflow_1 = require("./data-domain-tbac-workflow");
const data_domain_crawler_1 = require("./data-domain-crawler");
const data_mesh_workflow_role_1 = require("./data-mesh-workflow-role");
const lake_formation_1 = require("../lake-formation");
const s3_cross_account_1 = require("../s3-cross-account");
/**
 * This CDK Construct creates all required resources for data mesh in Data Domain account.
 *
 * It creates the following:
 * * A data lake with multiple layers (Raw, Cleaned, Transformed) using {@link DataLakeStorage} construct
 * * An mazon EventBridge Event Bus and Rules to enable Central Governance account to send events to Data Domain account
 * * An AWS Secret Manager secret encrypted via AWS KMS and used to share references with the central governance account
 * * A Data Domain Workflow {@link DataDomainWorkflow} responsible for creating resources in the data domain via a Step Functions state machine
 * * An optional Crawler workflow {@link DataDomainCrawler} responsible for updating the data product schema after registration via a Step Functions state machine
 *
 * Usage example:
 * ```typescript
 * import { App, Stack } from 'aws-cdk-lib';
 * import { Role } from 'aws-cdk-lib/aws-iam';
 * import { DataDomain } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new App();
 * const stack = new Stack(exampleApp, 'DataProductStack');
 *
 * new DataDomain(stack, 'myDataDomain', {
 *  centralAccountId: '1234567891011',
 *  crawlerWorkflow: true,
 *  domainName: 'domainName'
 * });
 * ```
 */
class DataDomain extends constructs_1.Construct {
    /**
     * Construct a new instance of DataDomain.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataDomainProps} props the DataDomainProps properties
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // The data lake used by the Data Domain
        this.dataLake = new data_lake_storage_1.DataLakeStorage(this, 'dataLakeStorage');
        this.centralAccountId = props.centralAccountId;
        // Add CDK execution role to LF admins
        lake_formation_1.LakeFormationAdmin.addCdkExecRole(scope, 'CdkLfAdmin');
        // Using the Bucket object and not the IBucket because CDK needs to change the bucket policy
        // KMS key is automatically discovered from the Bucket object and key policy is updated
        new s3_cross_account_1.S3CrossAccount(this, 'DataProductsPathCrossAccount', {
            accountId: this.centralAccountId,
            s3Bucket: this.dataLake.cleanBucket,
            s3ObjectKey: DataDomain.DATA_PRODUCTS_PREFIX,
        });
        // Workflow role used by state machine workflows
        const workflowRole = new data_mesh_workflow_role_1.DataMeshWorkflowRole(this, 'WorkflowRole');
        // Event Bridge event bus for data domain account
        this.eventBus = new aws_events_1.EventBus(this, 'dataDomainEventBus', {
            eventBusName: DataDomain.DOMAIN_BUS_NAME,
        });
        this.eventBus.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        // Cross-account policy to allow the central account to send events to data domain's bus
        const crossAccountBusPolicy = new aws_events_1.CfnEventBusPolicy(this, 'crossAccountBusPolicy', {
            eventBusName: this.eventBus.eventBusName,
            statementId: 'AllowCentralAccountToPutEvents',
            action: 'events:PutEvents',
            principal: this.centralAccountId,
        });
        crossAccountBusPolicy.node.addDependency(this.eventBus);
        // Create NRAC workflow
        const nracWorkflow = new data_domain_nrac_workflow_1.DataDomainNracWorkflow(this, 'nracWorkflow', {
            workflowRole: workflowRole.role,
            centralAccountId: this.centralAccountId,
            eventBus: this.eventBus,
        });
        // Event Bridge Rule to trigger NRAC worklfow upon event from the central account
        this.addBusRule('NRAC', central_governance_1.LfAccessControlMode.NRAC, nracWorkflow.stateMachine);
        // Create TBAC workflow
        const tbacWorkflow = new data_domain_tbac_workflow_1.DataDomainTbacWorkflow(this, 'tbacWorkflow', {
            domainName: props.domainName,
            workflowRole: workflowRole.role,
            centralAccountId: this.centralAccountId,
            eventBus: this.eventBus,
        });
        // Event Bridge Rule to trigger NRAC worklfow upon event from the central account
        this.addBusRule('TBAC', central_governance_1.LfAccessControlMode.TBAC, tbacWorkflow.stateMachine);
        // Allow the workflow role to send events to data domain event bus
        workflowRole.role.attachInlinePolicy(new aws_iam_1.Policy(this, 'SendEvents', {
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ['events:Put*'],
                    resources: [this.eventBus.eventBusArn],
                }),
            ],
        }));
        // create a workflow to update data products schemas on registration
        if (props.crawlerWorkflow) {
            const crawlerWorkflow = new data_domain_crawler_1.DataDomainCrawler(this, 'DataDomainCrawler', {
                domainName: props.domainName,
                workflowRole: workflowRole.role,
                dataProductsBucket: this.dataLake.cleanBucket,
                dataProductsPrefix: DataDomain.DATA_PRODUCTS_PREFIX,
                eventBus: this.eventBus,
            });
            // AWS Lambda function responsible to grant permissions on AWS Lake Formation tag to crawler role
            const tagPermissionsFn = new pre_bundled_function_1.PreBundledFunction(this, 'TagPermissionsFn', {
                runtime: aws_lambda_1.Runtime.PYTHON_3_9,
                codePath: 'data-mesh/resources/lambdas/crawler-tag-permission',
                handler: 'lambda.handler',
                logRetention: aws_logs_1.RetentionDays.ONE_DAY,
                timeout: aws_cdk_lib_1.Duration.seconds(20),
                role: workflowRole.role,
                environment: {
                    'CRAWLER_ROLE_ARN': crawlerWorkflow.crawlerRole.roleArn,
                    'CENTRAL_CATALOG_ID': props.centralAccountId,
                    'TAG_KEY': central_governance_1.CentralGovernance.DOMAIN_TAG_KEY,
                    'DOMAIN_TAG_VALUE': props.domainName,
                }
            });
            // add a rule to trigger the workflow from the event bus
            const triggerCrawlerRule = new aws_events_1.Rule(this, 'TriggerCrawler', {
                eventBus: this.eventBus,
                targets: [
                    crawlerWorkflow.stateMachine,
                ],
                eventPattern: {
                    source: ['com.central.stepfunction'],
                    detailType: ['triggerCrawler'],
                }
            });
            triggerCrawlerRule.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
            // add a rule to trigger the workflow from the event bus
            const grantCrawlerPermissionRule = new aws_events_1.Rule(this, 'grantCrawlerPermission', {
                eventBus: this.eventBus,
                targets: [
                    new targets.LambdaFunction(tagPermissionsFn),
                ],
                eventPattern: {
                    source: ['com.central.stepfunction'],
                    detailType: ['grantCrawlerPermission'],
                }
            });
            grantCrawlerPermissionRule.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
            // allow grantCrawlerPermissionRule to invoke Lambda fn
            targets.addLambdaPermission(grantCrawlerPermissionRule, tagPermissionsFn);
            // Construct central gov. account bus ARN
            const centralBusArn = `arn:aws:events:${aws_cdk_lib_1.Aws.REGION}:${props.centralAccountId}:event-bus/${central_governance_1.CentralGovernance.CENTRAL_BUS_NAME}`;
            // Event Bridge Rule to send event about crawler state to Central account
            const forwardCrawlerStateRule = new aws_events_1.Rule(this, 'CrawlerStateRule', {
                eventPattern: {
                    source: ['data-domain-state-change'],
                    detailType: ['data-domain-crawler-update'],
                },
                eventBus: this.eventBus,
            });
            forwardCrawlerStateRule.addTarget(new targets.EventBus(aws_events_1.EventBus.fromEventBusArn(this, '`${id}centralBus`', centralBusArn)));
            forwardCrawlerStateRule.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        }
        // create the data domain configuration object (in JSON) to be passed to the central governance account 
        var secretObject = {
            DomainName: aws_cdk_lib_1.SecretValue.unsafePlainText(props.domainName),
            BucketName: aws_cdk_lib_1.SecretValue.unsafePlainText(this.dataLake.cleanBucket.bucketName),
            Prefix: aws_cdk_lib_1.SecretValue.unsafePlainText(DataDomain.DATA_PRODUCTS_PREFIX),
        };
        // if the data product bucket is encrypted, add the key ID
        if (this.dataLake.cleanBucket.encryptionKey) {
            secretObject = {
                ...secretObject,
                ...{ KmsKeyId: aws_cdk_lib_1.SecretValue.unsafePlainText(this.dataLake.cleanBucket.encryptionKey.keyId) }
            };
        }
        const centralGovAccount = new aws_iam_1.AccountPrincipal(props.centralAccountId);
        // create a KMS key for encrypting the secret. It's required for cross account secret access
        const secretKey = new aws_kms_1.Key(this, 'SecretKey', {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        secretKey.grantDecrypt(centralGovAccount);
        // create the secret containing the data domain configuration object
        const domainConfigSecret = new aws_secretsmanager_1.Secret(this, 'DomainBucketSecret', {
            secretObjectValue: secretObject,
            secretName: DataDomain.DOMAIN_CONFIG_SECRET,
            encryptionKey: secretKey,
        });
        domainConfigSecret.grantRead(centralGovAccount);
        // output the full ARN of the secret to be passed when registring the data domain
        new aws_cdk_lib_1.CfnOutput(this, 'DomainSecretArnOutput', {
            value: domainConfigSecret.secretArn,
            exportName: `${aws_cdk_lib_1.Aws.ACCOUNT_ID}SecretArn`,
        });
    }
    addBusRule(id, mode, workflow) {
        // Create a Rule in Data Domain Event Bus
        const rule = new aws_events_1.Rule(this, `${id}Rule`, {
            eventPattern: {
                source: ['com.central.stepfunction'],
                account: [this.centralAccountId],
                detailType: [`${aws_cdk_lib_1.Aws.ACCOUNT_ID}_createResourceLinks`],
                detail: {
                    'lf_access_mode': [mode],
                },
            },
            eventBus: this.eventBus,
        });
        rule.node.addDependency(this.eventBus);
        rule.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        // Add target for this Rule
        rule.addTarget(new targets.SfnStateMachine(workflow));
    }
}
exports.DataDomain = DataDomain;
_a = JSII_RTTI_SYMBOL_1;
DataDomain[_a] = { fqn: "aws-analytics-reference-architecture.DataDomain", version: "2.9.7" };
DataDomain.DATA_PRODUCTS_PREFIX = 'data-products';
DataDomain.DOMAIN_CONFIG_SECRET = 'domain-config';
DataDomain.DOMAIN_BUS_NAME = 'data-mesh-bus';
//# sourceMappingURL=data:application/json;base64,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