"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BatchReplayer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const constructs_1 = require("constructs");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const batch_replayer_helpers_1 = require("./batch-replayer-helpers");
/**
 * Replay the data in the given PartitionedDataset.
 *
 * It will dump files into the target based on the given `frequency`.
 * The computation is in a Step Function with two Lambda steps.
 *
 * 1. resources/lambdas/find-file-paths
 * Read the manifest file and output a list of S3 file paths within that batch time range
 *
 * 2. resources/lambdas/write-in-batch
 * Take a file path, filter only records within given time range, adjust the time with offset to
 * make it looks like just being generated. Then write the output to the target
 *
 * Usage example:
 * ```typescript
 *
 * const myBucket = new Bucket(stack, "MyBucket")
 *
 * let myProps: IS3Sink = {
 *  sinkBucket: myBucket,
 *  sinkObjectKey: 'some-prefix',
 *  outputFileMaxSizeInBytes: 10000000,
 * }
 *
 * new BatchReplayer(stack, "WebSalesReplayer", {
 *   dataset: PreparedDataset.RETAIL_1_GB_WEB_SALE,
 *   s3Props: myProps,
 *   frequency: 120,
 * });
 * ```
 *
 * :warning: **If the Bucket is encrypted with KMS, the Key must be managed by this stack.
 */
class BatchReplayer extends constructs_1.Construct {
    /**
     * Constructs a new instance of the BatchReplayer construct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {BatchReplayerProps} props the BatchReplayer [properties]{@link BatchReplayerProps}
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.dataset = props.dataset;
        this.frequency = props.frequency?.toSeconds() || 60;
        // Properties for S3 target
        if (props.s3Props) {
            this.s3Props = props.s3Props;
            if (!this.s3Props.outputFileMaxSizeInBytes) {
                this.s3Props.outputFileMaxSizeInBytes = 100 * 1024 * 1024; //Default to 100 MB
            }
        }
        const manifestBucketName = this.dataset.manifestLocation.bucketName;
        const manifestObjectKey = this.dataset.manifestLocation.objectKey;
        const dataBucketName = this.dataset.location.bucketName;
        const dataObjectKey = this.dataset.location.objectKey;
        // Properties for DynamoDB target
        this.ddbProps = props.ddbProps ? props.ddbProps : undefined;
        // Properties for Redshift target
        this.redshiftProps = (props.redshiftProps && props.secGroup && props.vpc) ? props.redshiftProps : undefined;
        // Properties for Aurora target
        this.auroraProps = (props.auroraProps && props.secGroup && props.vpc) ? props.auroraProps : undefined;
        // Properties for RDS target
        this.rdsProps = (props.rdsProps && props.secGroup && props.vpc) ? props.rdsProps : undefined;
        const dataWranglerLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'PandasLayer', `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:336392948345:layer:AWSDataWrangler-Python39:1`);
        const findFilePathsFnPolicy = [
            new aws_iam_1.PolicyStatement({
                actions: [
                    's3:GetObject',
                    's3:ListBucket',
                ],
                resources: [
                    `arn:aws:s3:::${dataBucketName}/${dataObjectKey}/*`,
                    `arn:aws:s3:::${dataBucketName}/${dataObjectKey}-manifest.csv`,
                    `arn:aws:s3:::${dataBucketName}`,
                ],
            }),
        ];
        /**
         * Find all paths within the time range from the manifest file
         */
        const findFilePathsFn = new pre_bundled_function_1.PreBundledFunction(this, 'FindFilePath', {
            memorySize: 1024,
            codePath: 'data-generator/resources/lambdas/find-file-paths',
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            handler: 'find-file-paths.handler',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            layers: [dataWranglerLayer],
            lambdaPolicyStatements: findFilePathsFnPolicy,
        });
        const findFilePathsFnTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'FindFilePathFnTask', {
            lambdaFunction: findFilePathsFn,
            payload: aws_stepfunctions_1.TaskInput.fromObject({
                frequency: this.frequency,
                manifestFileBucket: manifestBucketName,
                manifestFileKey: manifestObjectKey,
                triggerTime: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Input.time'),
                offset: this.dataset.offset,
            }),
            // Retry on 500 error on invocation with an interval of 2 sec with back-off rate 2, for 6 times
            retryOnServiceExceptions: true,
            outputPath: '$.Payload',
        });
        const writeInBatchFnPolicy = [];
        let taskInputObj = {
            // Array from the last step to be mapped
            outputFileIndex: aws_stepfunctions_1.JsonPath.stringAt('$.index'),
            filePath: aws_stepfunctions_1.JsonPath.stringAt('$.filePath'),
            // For calculating the start/end time
            frequency: this.frequency,
            triggerTime: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Input.time'),
            offset: this.dataset.offset,
            // For file processing
            dateTimeColumnToFilter: this.dataset.dateTimeColumnToFilter,
            dateTimeColumnsToAdjust: this.dataset.dateTimeColumnsToAdjust,
        };
        // S3 target is selected
        if (this.s3Props) {
            // Used to force S3 bucket auto cleaning after deletion of this
            this.node.addDependency(this.s3Props.sinkBucket);
            this.s3Props.sinkObjectKey = this.s3Props.sinkObjectKey ?
                `${this.s3Props.sinkObjectKey}/${this.dataset.tableName}` : this.dataset.tableName;
            const { policy, taskInputParams } = batch_replayer_helpers_1.prepareS3Target(this.s3Props, dataBucketName, dataObjectKey);
            writeInBatchFnPolicy.push(policy);
            taskInputObj = Object.assign(taskInputObj, taskInputParams);
        }
        // DynamoDB target is selected
        if (this.ddbProps) {
            const { policy, taskInputParams } = batch_replayer_helpers_1.prepareDdbTarget(this.ddbProps);
            writeInBatchFnPolicy.push(policy);
            taskInputObj = Object.assign(taskInputObj, taskInputParams);
        }
        /**
         * Redshift, Aurora and RDS databases require the Lambda to have VPC access.
         */
        if (this.secGroup && this.vpc) {
            // Lambda requires these actions to have access to all resources in order to connect to a VPC
            writeInBatchFnPolicy.push(new aws_iam_1.PolicyStatement({
                actions: [
                    'ec2:CreateNetworkInterface',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:DeleteNetworkInterface',
                ],
                resources: ['*'],
            }));
            // Redshift target is selected
            if (this.redshiftProps) {
                const { policy, taskInputParams } = batch_replayer_helpers_1.prepareRedshiftTarget(this.redshiftProps);
                writeInBatchFnPolicy.push(policy);
                taskInputObj = Object.assign(taskInputObj, taskInputParams);
            }
            // Aurora target is selected
            if (this.auroraProps) {
                const { policy, taskInputParams } = batch_replayer_helpers_1.prepareAuroraTarget(this.auroraProps);
                writeInBatchFnPolicy.push(policy);
                taskInputObj = Object.assign(taskInputObj, taskInputParams);
            }
            // RDS target is selected
            if (this.rdsProps) {
                const { policy, taskInputParams } = batch_replayer_helpers_1.prepareRdsTarget(this.rdsProps);
                writeInBatchFnPolicy.push(policy);
                taskInputObj = Object.assign(taskInputObj, taskInputParams);
            }
        }
        /**
         * Rewrite data
         */
        const writeInBatchFn = new pre_bundled_function_1.PreBundledFunction(this, 'WriteInBatch', {
            memorySize: 3008,
            codePath: 'data-generator/resources/lambdas/write-in-batch',
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            handler: 'write-in-batch.handler',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            layers: [dataWranglerLayer],
            lambdaPolicyStatements: writeInBatchFnPolicy,
            vpc: this.vpc ? this.vpc : undefined,
            vpcSubnets: this.vpc ? this.vpc.selectSubnets({
                subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_NAT,
            }) : undefined,
            securityGroups: this.secGroup ? [this.secGroup] : undefined,
        });
        if (this.s3Props) {
            // grant permissions to write to the bucket and to use the KMS key
            this.s3Props.sinkBucket.grantWrite(writeInBatchFn, `${this.s3Props.sinkObjectKey}/*`);
        }
        const writeInBatchFnTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'WriteInBatchFnTask', {
            lambdaFunction: writeInBatchFn,
            payload: aws_stepfunctions_1.TaskInput.fromObject(taskInputObj),
            // Retry on 500 error on invocation with an interval of 2 sec with back-off rate 2, for 6 times
            retryOnServiceExceptions: true,
            outputPath: '$.Payload',
        });
        // Use "Map" step to write each filePath parallelly
        const writeInBatchMapTask = new aws_stepfunctions_1.Map(this, 'WriteInBatchMapTask', {
            itemsPath: aws_stepfunctions_1.JsonPath.stringAt('$.filePaths'),
            parameters: {
                index: aws_stepfunctions_1.JsonPath.stringAt('$$.Map.Item.Index'),
                filePath: aws_stepfunctions_1.JsonPath.stringAt('$$.Map.Item.Value'),
            },
        });
        writeInBatchMapTask.iterator(writeInBatchFnTask);
        // Overarching Step Function StateMachine
        const batchReplayStepFn = new aws_stepfunctions_1.StateMachine(this, 'BatchReplayStepFn', {
            definition: findFilePathsFnTask.next(writeInBatchMapTask),
            timeout: aws_cdk_lib_1.Duration.minutes(20),
            logs: {
                destination: new aws_logs_1.LogGroup(this, 'LogGroup', {
                    retention: aws_logs_1.RetentionDays.ONE_WEEK,
                    logGroupName: `/aws/batch-replayer/${this.dataset.tableName}`,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                }),
                level: aws_stepfunctions_1.LogLevel.ALL,
            },
        });
        new aws_events_1.Rule(this, 'BatchReplayStepFnTrigger', {
            schedule: aws_events_1.Schedule.cron({ minute: `0/${Math.ceil(this.frequency / 60)}` }),
            targets: [new aws_events_targets_1.SfnStateMachine(batchReplayStepFn, {})],
        });
    }
}
exports.BatchReplayer = BatchReplayer;
_a = JSII_RTTI_SYMBOL_1;
BatchReplayer[_a] = { fqn: "aws-analytics-reference-architecture.BatchReplayer", version: "2.9.7" };
//# sourceMappingURL=data:application/json;base64,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