"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AraBucket = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const singleton_kms_key_1 = require("./singleton-kms-key");
/**
* An Amazon S3 Bucket following best practices for the AWS Analytics Reference Architecture.
* The bucket name is mandatory and is used as the CDK id.
* The bucket name is postfixed with the AWS account ID and the AWS region.
*
* The bucket has the following default properties:
*  * the encryption mode is KMS managed by AWS
*  * if the encryption mode is KMS customer managed, the encryption key is a default and unique KMS key for ARA
*  * the KMS key is used as a bucket key
*  * the SSL is enforced
*  * the objects are automatically deleted when the bucket is deleted
*  * the access are logged in a default and unique S3 bucket for ARA if serverAccessLogsPrefix is provided
*  * the access are not logged if serverAccessLogsPrefix is  not provided
*  * the public access is blocked and no bucket policy or object permission can grant public access
*
* All standard S3 Bucket properties can be provided to not use the defaults.
* Usage example:
* ```typescript
* import * as cdk from 'aws-cdk-lib';
* import { AraBucket } from 'aws-analytics-reference-architecture';
*
* const exampleApp = new cdk.App();
* const stack = new cdk.Stack(exampleApp, 'AraBucketStack');
*
* new AraBucket(stack, {
*  bucketName: 'test-bucket',
*  serverAccessLogsPrefix: 'test-bucket',
* });
* ```
*/
class AraBucket extends aws_s3_1.Bucket {
    /**
    * Constructs a new instance of the AraBucket class
    * @param {Construct} scope the Scope of the CDK Construct
    * @param {AraBucketProps} props the AraBucketProps [properties]{@link AraBucketProps}
    * @access private
    */
    constructor(scope, props) {
        var serverAccessLogsBucket = undefined;
        if (props.serverAccessLogsPrefix) {
            serverAccessLogsBucket = props.serverAccessLogsBucket || AraBucket.getOrCreate(scope, { bucketName: 's3-access-logs', encryption: aws_s3_1.BucketEncryption.S3_MANAGED });
        }
        // If using KMS encryption then use a customer managed key, if not set the key to undefined
        let bucketEncryptionKey = aws_s3_1.BucketEncryption.KMS == props.encryption ? props.encryptionKey || singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey') : undefined;
        // If the bucket is for s3 access logs, we remove the bucketname to ensure uniqueness across stacks
        let bucketName = (props.bucketName == 's3-access-logs') ? undefined : `${props.bucketName}-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`;
        // set the right default parameters in the S3 bucket props
        const bucketProps = {
            ...props,
            ...{
                bucketName: bucketName,
                encryption: props.encryption ? props.encryption : aws_s3_1.BucketEncryption.KMS_MANAGED,
                encryptionKey: bucketEncryptionKey,
                bucketKeyEnabled: aws_s3_1.BucketEncryption.KMS == props.encryption ? true : false,
                enforceSSL: props.enforceSSL || true,
                removalPolicy: props.removalPolicy || aws_cdk_lib_1.RemovalPolicy.DESTROY,
                autoDeleteObjects: props.autoDeleteObjects || true,
                serverAccessLogsBucket: serverAccessLogsBucket,
                serverAccessLogsPrefix: props.serverAccessLogsPrefix,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                lifecycleRules: props.lifecycleRules || [{ abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(1) }],
            },
        };
        // build the S3 bucket
        super(scope, props.bucketName, bucketProps);
        // Used to force the cleaning of the access logs bucket after deletion of this 
        if (serverAccessLogsBucket) {
            this.node.addDependency(serverAccessLogsBucket);
        }
    }
    /**
    * Get the Amazon S3 Bucket from the AWS CDK Stack based on the provided name.
    * If no bucket exists, it creates a new one based on the provided properties.
    */
    static getOrCreate(scope, props) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = `${props.bucketName}`;
        const stackBucket = stack.node.tryFindChild(id) ?? (stack.nestedStackParent ? stack.nestedStackParent.node.tryFindChild(id) : undefined);
        return stackBucket || new AraBucket(stack, props);
    }
}
exports.AraBucket = AraBucket;
_a = JSII_RTTI_SYMBOL_1;
AraBucket[_a] = { fqn: "aws-analytics-reference-architecture.AraBucket", version: "2.9.7" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXJhLWJ1Y2tldC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9hcmEtYnVja2V0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBS0EsK0NBWTRCO0FBQzVCLDZDQUFrRTtBQUVsRSwyREFBbUQ7QUFtSW5EOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztFQTZCRTtBQUNGLE1BQWEsU0FBVSxTQUFRLGVBQU07SUFlbkM7Ozs7O01BS0U7SUFFRixZQUFvQixLQUFnQixFQUFFLEtBQXFCO1FBRXpELElBQUksc0JBQXNCLEdBQUcsU0FBUyxDQUFDO1FBQ3ZDLElBQUksS0FBSyxDQUFDLHNCQUFzQixFQUFFO1lBQ2hDLHNCQUFzQixHQUFHLEtBQUssQ0FBQyxzQkFBc0IsSUFBSSxTQUFTLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRSxFQUFFLFVBQVUsRUFBRSxnQkFBZ0IsRUFBRSxVQUFVLEVBQUUseUJBQWdCLENBQUMsVUFBVSxFQUFFLENBQUMsQ0FBQztTQUNsSztRQUVELDJGQUEyRjtRQUMzRixJQUFJLG1CQUFtQixHQUFxQix5QkFBZ0IsQ0FBQyxHQUFHLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGFBQWEsSUFBSSxnQ0FBWSxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsZUFBZSxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQztRQUUzSyxtR0FBbUc7UUFDbkcsSUFBSSxVQUFVLEdBQUcsQ0FBQyxLQUFLLENBQUMsVUFBVSxJQUFJLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLGlCQUFHLENBQUMsVUFBVSxJQUFJLGlCQUFHLENBQUMsTUFBTSxFQUFFLENBQUM7UUFDNUgsMERBQTBEO1FBQzFELE1BQU0sV0FBVyxHQUFHO1lBQ2xCLEdBQUcsS0FBSztZQUNSLEdBQUc7Z0JBQ0QsVUFBVSxFQUFFLFVBQVU7Z0JBQ3RCLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyx5QkFBZ0IsQ0FBQyxXQUFXO2dCQUM5RSxhQUFhLEVBQUUsbUJBQW1CO2dCQUNsQyxnQkFBZ0IsRUFBRSx5QkFBZ0IsQ0FBQyxHQUFHLElBQUksS0FBSyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxLQUFLO2dCQUN6RSxVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVUsSUFBSSxJQUFJO2dCQUNwQyxhQUFhLEVBQUUsS0FBSyxDQUFDLGFBQWEsSUFBSSwyQkFBYSxDQUFDLE9BQU87Z0JBQzNELGlCQUFpQixFQUFFLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxJQUFJO2dCQUNsRCxzQkFBc0IsRUFBRSxzQkFBc0I7Z0JBQzlDLHNCQUFzQixFQUFFLEtBQUssQ0FBQyxzQkFBc0I7Z0JBQ3BELGlCQUFpQixFQUFFLDBCQUFpQixDQUFDLFNBQVM7Z0JBQzlDLGNBQWMsRUFBRSxLQUFLLENBQUMsY0FBYyxJQUFJLENBQUMsRUFBRSxtQ0FBbUMsRUFBRSxzQkFBUSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2FBQ3BHO1NBQ0YsQ0FBQztRQUNGLHNCQUFzQjtRQUN0QixLQUFLLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxVQUFVLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFFNUMsK0VBQStFO1FBQy9FLElBQUksc0JBQXNCLEVBQUU7WUFDMUIsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsc0JBQXNCLENBQUMsQ0FBQztTQUNqRDtJQUNILENBQUM7SUF4REQ7OztNQUdFO0lBQ0ssTUFBTSxDQUFDLFdBQVcsQ0FBQyxLQUFnQixFQUFFLEtBQXFCO1FBQy9ELE1BQU0sS0FBSyxHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLE1BQU0sRUFBRSxHQUFHLEdBQUcsS0FBSyxDQUFDLFVBQVUsRUFBRSxDQUFDO1FBRWpDLE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBVyxJQUFJLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQVcsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUM7UUFFN0osT0FBTyxXQUFXLElBQUksSUFBSSxTQUFTLENBQUMsS0FBSyxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3BELENBQUM7O0FBYkgsOEJBMkRDIiwic291cmNlc0NvbnRlbnQiOlsiLy8gQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4vLyBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogTUlULTBcblxuaW1wb3J0IHsgSVJvbGUgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCB7IElLZXkgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mta21zJztcbmltcG9ydCB7XG4gIEJsb2NrUHVibGljQWNjZXNzLFxuICBCdWNrZXQsXG4gIEJ1Y2tldEFjY2Vzc0NvbnRyb2wsXG4gIEJ1Y2tldEVuY3J5cHRpb24sXG4gIEJ1Y2tldE1ldHJpY3MsXG4gIENvcnNSdWxlLFxuICBJQnVja2V0LFxuICBJbnRlbGxpZ2VudFRpZXJpbmdDb25maWd1cmF0aW9uLFxuICBJbnZlbnRvcnksXG4gIExpZmVjeWNsZVJ1bGUsXG4gIE9iamVjdE93bmVyc2hpcCxcbn0gZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzJztcbmltcG9ydCB7IER1cmF0aW9uLCBSZW1vdmFsUG9saWN5LCBTdGFjaywgQXdzIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQgeyBTaW5nbGV0b25LZXkgfSBmcm9tICcuL3NpbmdsZXRvbi1rbXMta2V5JztcblxuZXhwb3J0IGludGVyZmFjZSBBcmFCdWNrZXRQcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIFMzIGJ1Y2tldCBuYW1lLiBUaGUgYnVja2V0IG5hbWUgaXMgcG9zdGZpeGVkIHdpdGggdGhlIEFXUyBhY2NvdW50IElEIGFuZCB0aGUgQVdTIHJlZ2lvblxuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0TmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbG9nIGZpbGUgcHJlZml4IHRvIHVzZSBmb3IgdGhlIGJ1Y2tldCdzIGFjY2VzcyBsb2dzLlxuICAgKiBAZGVmYXVsdCAtIGFjY2VzcyBhcmUgbm90IGxvZ2dlZFxuICAgKi9cbiAgcmVhZG9ubHkgc2VydmVyQWNjZXNzTG9nc1ByZWZpeD86IHN0cmluZztcblxuICAvKipcbiAgICogRGVzdGluYXRpb24gYnVja2V0IGZvciB0aGUgc2VydmVyIGFjY2VzcyBsb2dzLlxuICAgKiBAZGVmYXVsdCAtIGlmIHNlcnZlckFjY2Vzc0xvZ3NQcmVmaXggaXMgZGVmaW5lZCwgdXNlIGEgdW5pcXVlIGJ1Y2tldCBhY3Jvc3MgdGhlIHN0YWNrIGNhbGxlZCBgczMtYWNjZXNzLWxvZ3NgXG4gICAqL1xuICByZWFkb25seSBzZXJ2ZXJBY2Nlc3NMb2dzQnVja2V0PzogSUJ1Y2tldDtcblxuICAvKipcbiAgICogVGhlIGVuY3J5cHRpb24gbW9kZSBmb3IgdGhlIGJ1Y2tldFxuICAgKiBAZGVmYXVsdCAtIFNlcnZlciBzaWRlIGVuY3J5cHRpb24gd2l0aCBBV1MgbWFuYWdlZCBrZXkgKFNTRS1LTVMpXG4gICAqL1xuICByZWFkb25seSBlbmNyeXB0aW9uPzogQnVja2V0RW5jcnlwdGlvbjtcblxuICAvKipcbiAgICogVGhlIEtNUyBrZXkgZm9yIHRoZSBidWNrZXQgZW5jcnlwdGlvblxuICAgKiBAZGVmYXVsdCAtIGlmIGVuY3J5cHRpb24gaXMgS01TLCB1c2UgYSB1bmlxdWUgS01TIGtleSBhY3Jvc3MgdGhlIHN0YWNrIGNhbGxlZCBgQXJhRGVmYXVsdEttc0tleWBcbiAgICovXG4gIHJlYWRvbmx5IGVuY3J5cHRpb25LZXk/OiBJS2V5O1xuXG4gIC8qKlxuICAgKiBFbmZvcmNlcyBTU0wgZm9yIHJlcXVlc3RzLlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBlbmZvcmNlU1NMPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHdoZXRoZXIgQW1hem9uIFMzIHNob3VsZCB1c2UgYW4gUzMgQnVja2V0IEtleSB3aXRoIHNlcnZlci1zaWRlIGVuY3J5cHRpb24gdXNpbmcgS01TIChTU0UtS01TKSBmb3IgbmV3IG9iamVjdHMgaW4gdGhlIGJ1Y2tldC5cbiAgICogQGRlZmF1bHQgdHJ1ZVxuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0S2V5RW5hYmxlZD86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFBvbGljeSB0byBhcHBseSB3aGVuIHRoZSBidWNrZXQgaXMgcmVtb3ZlZCBmcm9tIHRoaXMgc3RhY2suXG4gICAqIEBkZWZhdWx0IC0gZGVzdHJveSB0aGUgYnVja2V0XG4gICAqL1xuICByZWFkb25seSByZW1vdmFsUG9saWN5PzogUmVtb3ZhbFBvbGljeTtcblxuICAvKipcbiAgICogV2hldGhlciBhbGwgb2JqZWN0cyBzaG91bGQgYmUgYXV0b21hdGljYWxseSBkZWxldGVkIHdoZW4gdGhlIGJ1Y2tldCBpcyByZW1vdmVkIGZyb20gdGhlIHN0YWNrIG9yIHdoZW4gdGhlIHN0YWNrIGlzIGRlbGV0ZWQuXG4gICAqIFJlcXVpcmVzIHRoZSBgcmVtb3ZhbFBvbGljeWAgdG8gYmUgc2V0IHRvIGBSZW1vdmFsUG9saWN5LkRFU1RST1lgLlxuICAgKiBAZGVmYXVsdCB0cnVlXG4gICAqL1xuICByZWFkb25seSBhdXRvRGVsZXRlT2JqZWN0cz86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBibG9jayBwdWJsaWMgYWNjZXNzIGNvbmZpZ3VyYXRpb24gb2YgdGhpcyBidWNrZXQuXG4gICAqIEBkZWZhdWx0IC0gQmxvY2sgYWxsIHB1YmxpYyBhY2Nlc3MgYW5kIG5vIEFDTCBvciBidWNrZXQgcG9saWN5IGNhbiBncmFudCBwdWJsaWMgYWNjZXNzLlxuICAgKi9cbiAgcmVhZG9ubHkgYmxvY2tQdWJsaWNBY2Nlc3M/OiBCbG9ja1B1YmxpY0FjY2VzcztcblxuICAvKipcbiAgICogV2hldGhlciB0aGlzIGJ1Y2tldCBzaG91bGQgaGF2ZSB2ZXJzaW9uaW5nIHR1cm5lZCBvbiBvciBub3QuXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSB2ZXJzaW9uZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBSdWxlcyB0aGF0IGRlZmluZSBob3cgQW1hem9uIFMzIG1hbmFnZXMgb2JqZWN0cyBkdXJpbmcgdGhlaXIgbGlmZXRpbWUuXG4gICAqIEBkZWZhdWx0IC0gTm8gbGlmZWN5Y2xlIHJ1bGVzLlxuICAgKi9cbiAgcmVhZG9ubHkgbGlmZWN5Y2xlUnVsZXM/OiBMaWZlY3ljbGVSdWxlW107XG5cbiAgLyoqXG4gICAqIFNwZWNpZmllcyBhIGNhbm5lZCBBQ0wgdGhhdCBncmFudHMgcHJlZGVmaW5lZCBwZXJtaXNzaW9ucyB0byB0aGUgYnVja2V0LlxuICAgKiBAZGVmYXVsdCBCdWNrZXRBY2Nlc3NDb250cm9sLlBSSVZBVEVcbiAgICovXG4gIHJlYWRvbmx5IGFjY2Vzc0NvbnRyb2w/OiBCdWNrZXRBY2Nlc3NDb250cm9sO1xuXG4gIC8qKlxuICAgKiBHcmFudHMgcHVibGljIHJlYWQgYWNjZXNzIHRvIGFsbCBvYmplY3RzIGluIHRoZSBidWNrZXQuXG4gICAqIFNpbWlsYXIgdG8gY2FsbGluZyBgYnVja2V0LmdyYW50UHVibGljQWNjZXNzKClgXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBwdWJsaWNSZWFkQWNjZXNzPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIG1ldHJpY3MgY29uZmlndXJhdGlvbiBvZiB0aGlzIGJ1Y2tldC5cbiAgICogQGRlZmF1bHQgLSBObyBtZXRyaWNzIGNvbmZpZ3VyYXRpb24uXG4gICAqL1xuICByZWFkb25seSBtZXRyaWNzPzogQnVja2V0TWV0cmljc1tdO1xuXG4gIC8qKlxuICAgKiBUaGUgQ09SUyBjb25maWd1cmF0aW9uIG9mIHRoaXMgYnVja2V0LlxuICAgKiBAZGVmYXVsdCAtIE5vIENPUlMgY29uZmlndXJhdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGNvcnM/OiBDb3JzUnVsZVtdO1xuXG4gIC8qKlxuICAgKiBUaGUgaW52ZW50b3J5IGNvbmZpZ3VyYXRpb24gb2YgdGhlIGJ1Y2tldC5cbiAgICogQGRlZmF1bHQgLSBObyBpbnZlbnRvcnkgY29uZmlndXJhdGlvblxuICAqL1xuICByZWFkb25seSBpbnZlbnRvcmllcz86IEludmVudG9yeVtdO1xuXG4gIC8qKlxuICAgKiBUaGUgb2JqZWN0T3duZXJzaGlwIG9mIHRoZSBidWNrZXQuXG4gICAqIEBkZWZhdWx0IC0gTm8gT2JqZWN0T3duZXJzaGlwIGNvbmZpZ3VyYXRpb24sIHVwbG9hZGluZyBhY2NvdW50IHdpbGwgb3duIHRoZSBvYmplY3QuXG4gICAqL1xuICByZWFkb25seSBvYmplY3RPd25lcnNoaXA/OiBPYmplY3RPd25lcnNoaXA7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgdGhpcyBidWNrZXQgc2hvdWxkIGhhdmUgdHJhbnNmZXIgYWNjZWxlcmF0aW9uIHR1cm5lZCBvbiBvciBub3QuXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSB0cmFuc2ZlckFjY2VsZXJhdGlvbj86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSByb2xlIHRvIGJlIHVzZWQgYnkgdGhlIG5vdGlmaWNhdGlvbnMgaGFuZGxlclxuICAgKiBAZGVmYXVsdCAtIGEgbmV3IHJvbGUgd2lsbCBiZSBjcmVhdGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgbm90aWZpY2F0aW9uc0hhbmRsZXJSb2xlPzogSVJvbGU7XG5cbiAgLyoqXG4gICAqIEludGVsaWdlbnQgVGllcmluZyBDb25maWd1cmF0aW9uc1xuICAgKiBAZGVmYXVsdCBObyBJbnRlbGxpZ2VudCBUaWllcmluZyBDb25maWd1cmF0aW9ucy5cbiAgICovXG4gIHJlYWRvbmx5IGludGVsbGlnZW50VGllcmluZ0NvbmZpZ3VyYXRpb25zPzogSW50ZWxsaWdlbnRUaWVyaW5nQ29uZmlndXJhdGlvbltdO1xufVxuXG4vKipcbiogQW4gQW1hem9uIFMzIEJ1Y2tldCBmb2xsb3dpbmcgYmVzdCBwcmFjdGljZXMgZm9yIHRoZSBBV1MgQW5hbHl0aWNzIFJlZmVyZW5jZSBBcmNoaXRlY3R1cmUuXG4qIFRoZSBidWNrZXQgbmFtZSBpcyBtYW5kYXRvcnkgYW5kIGlzIHVzZWQgYXMgdGhlIENESyBpZC5cbiogVGhlIGJ1Y2tldCBuYW1lIGlzIHBvc3RmaXhlZCB3aXRoIHRoZSBBV1MgYWNjb3VudCBJRCBhbmQgdGhlIEFXUyByZWdpb24uXG4qXG4qIFRoZSBidWNrZXQgaGFzIHRoZSBmb2xsb3dpbmcgZGVmYXVsdCBwcm9wZXJ0aWVzOlxuKiAgKiB0aGUgZW5jcnlwdGlvbiBtb2RlIGlzIEtNUyBtYW5hZ2VkIGJ5IEFXU1xuKiAgKiBpZiB0aGUgZW5jcnlwdGlvbiBtb2RlIGlzIEtNUyBjdXN0b21lciBtYW5hZ2VkLCB0aGUgZW5jcnlwdGlvbiBrZXkgaXMgYSBkZWZhdWx0IGFuZCB1bmlxdWUgS01TIGtleSBmb3IgQVJBXG4qICAqIHRoZSBLTVMga2V5IGlzIHVzZWQgYXMgYSBidWNrZXQga2V5XG4qICAqIHRoZSBTU0wgaXMgZW5mb3JjZWRcbiogICogdGhlIG9iamVjdHMgYXJlIGF1dG9tYXRpY2FsbHkgZGVsZXRlZCB3aGVuIHRoZSBidWNrZXQgaXMgZGVsZXRlZFxuKiAgKiB0aGUgYWNjZXNzIGFyZSBsb2dnZWQgaW4gYSBkZWZhdWx0IGFuZCB1bmlxdWUgUzMgYnVja2V0IGZvciBBUkEgaWYgc2VydmVyQWNjZXNzTG9nc1ByZWZpeCBpcyBwcm92aWRlZFxuKiAgKiB0aGUgYWNjZXNzIGFyZSBub3QgbG9nZ2VkIGlmIHNlcnZlckFjY2Vzc0xvZ3NQcmVmaXggaXMgIG5vdCBwcm92aWRlZFxuKiAgKiB0aGUgcHVibGljIGFjY2VzcyBpcyBibG9ja2VkIGFuZCBubyBidWNrZXQgcG9saWN5IG9yIG9iamVjdCBwZXJtaXNzaW9uIGNhbiBncmFudCBwdWJsaWMgYWNjZXNzXG4qXG4qIEFsbCBzdGFuZGFyZCBTMyBCdWNrZXQgcHJvcGVydGllcyBjYW4gYmUgcHJvdmlkZWQgdG8gbm90IHVzZSB0aGUgZGVmYXVsdHMuXG4qIFVzYWdlIGV4YW1wbGU6XG4qIGBgYHR5cGVzY3JpcHRcbiogaW1wb3J0ICogYXMgY2RrIGZyb20gJ2F3cy1jZGstbGliJztcbiogaW1wb3J0IHsgQXJhQnVja2V0IH0gZnJvbSAnYXdzLWFuYWx5dGljcy1yZWZlcmVuY2UtYXJjaGl0ZWN0dXJlJztcbipcbiogY29uc3QgZXhhbXBsZUFwcCA9IG5ldyBjZGsuQXBwKCk7XG4qIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayhleGFtcGxlQXBwLCAnQXJhQnVja2V0U3RhY2snKTtcbipcbiogbmV3IEFyYUJ1Y2tldChzdGFjaywge1xuKiAgYnVja2V0TmFtZTogJ3Rlc3QtYnVja2V0JyxcbiogIHNlcnZlckFjY2Vzc0xvZ3NQcmVmaXg6ICd0ZXN0LWJ1Y2tldCcsXG4qIH0pO1xuKiBgYGBcbiovXG5leHBvcnQgY2xhc3MgQXJhQnVja2V0IGV4dGVuZHMgQnVja2V0IHtcblxuICAvKipcbiAgKiBHZXQgdGhlIEFtYXpvbiBTMyBCdWNrZXQgZnJvbSB0aGUgQVdTIENESyBTdGFjayBiYXNlZCBvbiB0aGUgcHJvdmlkZWQgbmFtZS5cbiAgKiBJZiBubyBidWNrZXQgZXhpc3RzLCBpdCBjcmVhdGVzIGEgbmV3IG9uZSBiYXNlZCBvbiB0aGUgcHJvdmlkZWQgcHJvcGVydGllcy5cbiAgKi9cbiAgcHVibGljIHN0YXRpYyBnZXRPckNyZWF0ZShzY29wZTogQ29uc3RydWN0LCBwcm9wczogQXJhQnVja2V0UHJvcHMpIHtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHNjb3BlKTtcbiAgICBjb25zdCBpZCA9IGAke3Byb3BzLmJ1Y2tldE5hbWV9YDtcblxuICAgIGNvbnN0IHN0YWNrQnVja2V0ID0gc3RhY2subm9kZS50cnlGaW5kQ2hpbGQoaWQpIGFzIEJ1Y2tldCA/PyAoc3RhY2submVzdGVkU3RhY2tQYXJlbnQgPyBzdGFjay5uZXN0ZWRTdGFja1BhcmVudC5ub2RlLnRyeUZpbmRDaGlsZChpZCkgYXMgQnVja2V0IDogdW5kZWZpbmVkKTtcblxuICAgIHJldHVybiBzdGFja0J1Y2tldCB8fCBuZXcgQXJhQnVja2V0KHN0YWNrLCBwcm9wcyk7XG4gIH1cblxuICAvKipcbiAgKiBDb25zdHJ1Y3RzIGEgbmV3IGluc3RhbmNlIG9mIHRoZSBBcmFCdWNrZXQgY2xhc3NcbiAgKiBAcGFyYW0ge0NvbnN0cnVjdH0gc2NvcGUgdGhlIFNjb3BlIG9mIHRoZSBDREsgQ29uc3RydWN0XG4gICogQHBhcmFtIHtBcmFCdWNrZXRQcm9wc30gcHJvcHMgdGhlIEFyYUJ1Y2tldFByb3BzIFtwcm9wZXJ0aWVzXXtAbGluayBBcmFCdWNrZXRQcm9wc31cbiAgKiBAYWNjZXNzIHByaXZhdGVcbiAgKi9cblxuICBwcml2YXRlIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIHByb3BzOiBBcmFCdWNrZXRQcm9wcykge1xuXG4gICAgdmFyIHNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQgPSB1bmRlZmluZWQ7XG4gICAgaWYgKHByb3BzLnNlcnZlckFjY2Vzc0xvZ3NQcmVmaXgpIHtcbiAgICAgIHNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQgPSBwcm9wcy5zZXJ2ZXJBY2Nlc3NMb2dzQnVja2V0IHx8IEFyYUJ1Y2tldC5nZXRPckNyZWF0ZShzY29wZSwgeyBidWNrZXROYW1lOiAnczMtYWNjZXNzLWxvZ3MnLCBlbmNyeXB0aW9uOiBCdWNrZXRFbmNyeXB0aW9uLlMzX01BTkFHRUQgfSk7XG4gICAgfVxuXG4gICAgLy8gSWYgdXNpbmcgS01TIGVuY3J5cHRpb24gdGhlbiB1c2UgYSBjdXN0b21lciBtYW5hZ2VkIGtleSwgaWYgbm90IHNldCB0aGUga2V5IHRvIHVuZGVmaW5lZFxuICAgIGxldCBidWNrZXRFbmNyeXB0aW9uS2V5OiBJS2V5IHwgdW5kZWZpbmVkID0gQnVja2V0RW5jcnlwdGlvbi5LTVMgPT0gcHJvcHMuZW5jcnlwdGlvbiA/IHByb3BzLmVuY3J5cHRpb25LZXkgfHwgU2luZ2xldG9uS2V5LmdldE9yQ3JlYXRlKHNjb3BlLCAnRGVmYXVsdEttc0tleScpIDogdW5kZWZpbmVkO1xuXG4gICAgLy8gSWYgdGhlIGJ1Y2tldCBpcyBmb3IgczMgYWNjZXNzIGxvZ3MsIHdlIHJlbW92ZSB0aGUgYnVja2V0bmFtZSB0byBlbnN1cmUgdW5pcXVlbmVzcyBhY3Jvc3Mgc3RhY2tzXG4gICAgbGV0IGJ1Y2tldE5hbWUgPSAocHJvcHMuYnVja2V0TmFtZSA9PSAnczMtYWNjZXNzLWxvZ3MnKSA/IHVuZGVmaW5lZCA6IGAke3Byb3BzLmJ1Y2tldE5hbWV9LSR7QXdzLkFDQ09VTlRfSUR9LSR7QXdzLlJFR0lPTn1gO1xuICAgIC8vIHNldCB0aGUgcmlnaHQgZGVmYXVsdCBwYXJhbWV0ZXJzIGluIHRoZSBTMyBidWNrZXQgcHJvcHNcbiAgICBjb25zdCBidWNrZXRQcm9wcyA9IHtcbiAgICAgIC4uLnByb3BzLFxuICAgICAgLi4ue1xuICAgICAgICBidWNrZXROYW1lOiBidWNrZXROYW1lLFxuICAgICAgICBlbmNyeXB0aW9uOiBwcm9wcy5lbmNyeXB0aW9uID8gcHJvcHMuZW5jcnlwdGlvbiA6IEJ1Y2tldEVuY3J5cHRpb24uS01TX01BTkFHRUQsXG4gICAgICAgIGVuY3J5cHRpb25LZXk6IGJ1Y2tldEVuY3J5cHRpb25LZXksXG4gICAgICAgIGJ1Y2tldEtleUVuYWJsZWQ6IEJ1Y2tldEVuY3J5cHRpb24uS01TID09IHByb3BzLmVuY3J5cHRpb24gPyB0cnVlIDogZmFsc2UsXG4gICAgICAgIGVuZm9yY2VTU0w6IHByb3BzLmVuZm9yY2VTU0wgfHwgdHJ1ZSxcbiAgICAgICAgcmVtb3ZhbFBvbGljeTogcHJvcHMucmVtb3ZhbFBvbGljeSB8fCBSZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgICAgIGF1dG9EZWxldGVPYmplY3RzOiBwcm9wcy5hdXRvRGVsZXRlT2JqZWN0cyB8fCB0cnVlLFxuICAgICAgICBzZXJ2ZXJBY2Nlc3NMb2dzQnVja2V0OiBzZXJ2ZXJBY2Nlc3NMb2dzQnVja2V0LFxuICAgICAgICBzZXJ2ZXJBY2Nlc3NMb2dzUHJlZml4OiBwcm9wcy5zZXJ2ZXJBY2Nlc3NMb2dzUHJlZml4LFxuICAgICAgICBibG9ja1B1YmxpY0FjY2VzczogQmxvY2tQdWJsaWNBY2Nlc3MuQkxPQ0tfQUxMLFxuICAgICAgICBsaWZlY3ljbGVSdWxlczogcHJvcHMubGlmZWN5Y2xlUnVsZXMgfHwgW3sgYWJvcnRJbmNvbXBsZXRlTXVsdGlwYXJ0VXBsb2FkQWZ0ZXI6IER1cmF0aW9uLmRheXMoMSkgfV0sXG4gICAgICB9LFxuICAgIH07XG4gICAgLy8gYnVpbGQgdGhlIFMzIGJ1Y2tldFxuICAgIHN1cGVyKHNjb3BlLCBwcm9wcy5idWNrZXROYW1lLCBidWNrZXRQcm9wcyk7XG5cbiAgICAvLyBVc2VkIHRvIGZvcmNlIHRoZSBjbGVhbmluZyBvZiB0aGUgYWNjZXNzIGxvZ3MgYnVja2V0IGFmdGVyIGRlbGV0aW9uIG9mIHRoaXMgXG4gICAgaWYgKHNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQpIHtcbiAgICAgIHRoaXMubm9kZS5hZGREZXBlbmRlbmN5KHNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQpO1xuICAgIH1cbiAgfVxufVxuIl19