"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Packaging = exports.DependenciesFile = void 0;
const fs = require("fs");
const path = require("path");
var DependenciesFile;
(function (DependenciesFile) {
    DependenciesFile["PIP"] = "requirements.txt";
    DependenciesFile["POETRY"] = "poetry.lock";
    DependenciesFile["PIPENV"] = "Pipfile.lock";
    DependenciesFile["UV"] = "uv.lock";
    DependenciesFile["NONE"] = "";
})(DependenciesFile || (exports.DependenciesFile = DependenciesFile = {}));
class Packaging {
    /**
     * Standard packaging with `pip`.
     */
    static withPip() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIP,
        });
    }
    /**
     * Packaging with `pipenv`.
     */
    static withPipenv() {
        return new Packaging({
            dependenciesFile: DependenciesFile.PIPENV,
            // By default, pipenv creates a virtualenv in `/.local`, so we force it to create one in the package directory.
            // At the end, we remove the virtualenv to avoid creating a duplicate copy in the Lambda package.
            exportCommand: `PIPENV_VENV_IN_PROJECT=1 pipenv requirements > ${DependenciesFile.PIP} && rm -rf .venv`,
        });
    }
    /**
     * Packaging with `poetry`.
     */
    static withPoetry(props) {
        return new Packaging({
            dependenciesFile: DependenciesFile.POETRY,
            // Export dependencies with credentials available in the bundling image.
            exportCommand: [
                'poetry', 'export',
                ...props?.poetryIncludeHashes ? [] : ['--without-hashes'],
                ...props?.poetryWithoutUrls ? ['--without-urls'] : [],
                '--with-credentials',
                '--format', DependenciesFile.PIP,
                '--output', DependenciesFile.PIP,
            ].join(' '),
        });
    }
    /**
     * Packaging with `uv`.
     */
    static withUv() {
        return new Packaging({
            dependenciesFile: DependenciesFile.UV,
            exportCommand: `uv export --frozen --no-emit-workspace --no-dev --no-editable -o ${DependenciesFile.PIP}`,
        });
    }
    /**
     * No dependencies or packaging.
     */
    static withNoPackaging() {
        return new Packaging({ dependenciesFile: DependenciesFile.NONE });
    }
    static fromEntry(entry, poetryIncludeHashes, poetryWithoutUrls) {
        if (fs.existsSync(path.join(entry, DependenciesFile.PIPENV))) {
            return this.withPipenv();
        }
        if (fs.existsSync(path.join(entry, DependenciesFile.POETRY))) {
            return this.withPoetry({ poetryIncludeHashes, poetryWithoutUrls });
        }
        else if (fs.existsSync(path.join(entry, DependenciesFile.PIP))) {
            return this.withPip();
        }
        else if (fs.existsSync(path.join(entry, DependenciesFile.UV))) {
            return this.withUv();
        }
        else {
            return this.withNoPackaging();
        }
    }
    dependenciesFile;
    exportCommand;
    constructor(props) {
        this.dependenciesFile = props.dependenciesFile;
        this.exportCommand = props.exportCommand;
    }
}
exports.Packaging = Packaging;
//# sourceMappingURL=data:application/json;base64,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