# AUTOGENERATED! DO NOT EDIT! File to edit: ../notebooks/01_ljubljana.ipynb.

# %% auto 0
__all__ = ['LjubljanaDataset', 'Transforms']

# %% ../notebooks/01_ljubljana.ipynb 3
from pathlib import Path

import numpy as np
import torch
from diffdrr.data import read
from diffdrr.pose import RigidTransform
from torchio import ScalarImage, Subject

from .utils import load_file

# %% ../notebooks/01_ljubljana.ipynb 5
class LjubljanaDataset(torch.utils.data.Dataset):
    """
    A `torch.utils.data.Dataset` that stores the imaging data for subjects
    in the `Ljubljana` dataset.
    """

    def __init__(
        self,
        id_number: int,  # Subject ID in {1, ..., 10}
        preprocess: bool = True,  # Convert X-rays from exponentiated to linear form
    ):
        self.f = load_file("ljubljana.h5")
        self.id_number = id_number
        self.subject, self.anatomical2world = parse_volume(self.f, self.id_number)
        self.preprocess = preprocess

        # Miscellaneous transformation matrices for wrangling SE(3) poses
        self.flip_z = RigidTransform(
            torch.tensor(
                [
                    [-1, 0, 0, 0],
                    [0, -1, 0, 0],
                    [0, 0, -1, 0],
                    [0, 0, 0, 1],
                ]
            ).to(torch.float32)
        )
        self.reorient = RigidTransform(self.subject.reorient)

    def __len__(self):
        return 2

    def __iter__(self):
        return iter(self[idx] for idx in range(len(self)))

    def __getitem__(self, idx):
        view = {0: "ap", 1: "lat"}[idx]
        img, world2camera, focal_len, height, width, delx, dely, x0, y0 = parse_proj(
            self.f, self.id_number, view
        )

        if self.preprocess:
            img = img - img.mode().values.mode().values.item()  # Subtract background color
            img = torch.clamp(img, -1, 0) + 1  # Restrict to [0, 1]
            img += 1  # Convert to log-scale
            img = img.max().log() - img.log()

        pose = (
            self.reorient.inverse()
            .compose(self.flip_z)
            .compose(world2camera.inverse())
            .compose(self.anatomical2world)
        )

        return img, pose, focal_len, height, width, delx, dely, x0, y0

# %% ../notebooks/01_ljubljana.ipynb 6
def parse_volume(f, subject_id):
    subject = f[f"subject{subject_id:02d}"]

    # Get the volume
    volume = subject["volume/pixels"][:]
    volume = volume.copy()
    volume = torch.from_numpy(volume).unsqueeze(0).flip(1)
    volume[volume < 1000] = 0.0  # Discard a lot of the background from the 3D DSA

    affine = np.eye((4))
    spacing = subject["volume/spacing"][:]
    affine[0, 0] = spacing[0]
    affine[1, 1] = spacing[1]
    affine[2, 2] = spacing[2]
    affine[:3, 3] = subject["volume/origin"][:]
    affine = torch.from_numpy(affine)

    volume = ScalarImage(tensor=volume, affine=affine)
    fiducials = torch.from_numpy(subject["points"][:]).unsqueeze(0)

    # Move the Subject's isocenter to the origin in world coordinates
    isocenter = volume.get_center()
    anatomical2world = RigidTransform(
        torch.tensor(
            [
                [1.0, 0.0, 0.0, -isocenter[0]],
                [0.0, 1.0, 0.0, -isocenter[1]],
                [0.0, 0.0, 1.0, -isocenter[2]],
                [0.0, 0.0, 0.0, 1.0],
            ],
            dtype=torch.float32,
        )
    )

    # Package the subject
    subject = read(
        volume=volume,
        labelmap=None,
        bone_attenuation_multiplier=1.0,
        fiducials=fiducials,
        orientation="AP",
    )
    
    return subject, anatomical2world

# %% ../notebooks/01_ljubljana.ipynb 7
from diffdrr.detector import parse_intrinsic_matrix


def parse_proj(f, subject_id, view):
    proj = f[f"subject{subject_id:02d}/proj-{view}"]

    img = torch.from_numpy(proj["pixels"][:]).unsqueeze(0).unsqueeze(0)
    num_rows, num_cols = proj["pixels"].shape

    extrinsic = torch.from_numpy(proj["extrinsic"][:])
    world2camera = RigidTransform(extrinsic)

    intrinsic = torch.from_numpy(proj["intrinsic"][:])
    proj_col_spacing = float(proj["col-spacing"][()])
    proj_row_spacing = float(proj["row-spacing"][()])
    focal_len, x0, y0 = parse_intrinsic_matrix(
        intrinsic,
        num_rows,
        num_cols,
        proj_row_spacing,
        proj_col_spacing,
    )

    return (
        img,
        world2camera,
        focal_len,
        int(num_rows),
        int(num_cols),
        proj_row_spacing,
        proj_col_spacing,
        x0,
        y0,
    )

# %% ../notebooks/01_ljubljana.ipynb 8
from torchvision.transforms import Compose, Lambda, Normalize, Resize


class Transforms:
    def __init__(self, height: int, width: int, eps: float = 1e-6):
        """Standardize, resize, and normalize X-rays and DRRs before inputting to a deep learning model."""
        self.transforms = Compose(
            [
                Lambda(lambda x: (x - x.min()) / (x.max() - x.min() + eps)),
                Resize((height, width), antialias=True),
                Normalize(mean=0.0306, std=0.0564),
            ]
        )

    def __call__(self, x):
        return self.transforms(x)
