import { CODE_ASSIST_API_VERSION, CODE_ASSIST_ENDPOINT, OPENAI_CHAT_COMPLETION_OBJECT } from "../utils/constant.js";
import { AutoModelSwitchingHelper } from "./auto-model-switching.js";
import { getLogger } from "../utils/logger.js";
import chalk from "chalk";
/**
 * Custom error class for Gemini API errors with status code information
 */
export class GeminiApiError extends Error {
    statusCode;
    responseText;
    constructor(message, statusCode, responseText) {
        super(message);
        this.statusCode = statusCode;
        this.responseText = responseText;
        this.name = "GeminiApiError";
    }
}
/**
 * Handles communication with Google's Gemini API through the Code Assist endpoint.
 */
export class GeminiApiClient {
    authClient;
    googleCloudProject;
    disableAutoModelSwitch;
    projectId = null;
    firstChunk = true;
    creationTime;
    chatID;
    autoSwitcher;
    logger;
    constructor(authClient, googleCloudProject, disableAutoModelSwitch) {
        this.authClient = authClient;
        this.googleCloudProject = googleCloudProject;
        this.disableAutoModelSwitch = disableAutoModelSwitch;
        this.googleCloudProject = googleCloudProject;
        this.chatID = `chat-${crypto.randomUUID()}`;
        this.creationTime = Math.floor(Date.now() / 1000);
        this.autoSwitcher = AutoModelSwitchingHelper.getInstance();
        this.logger = getLogger("GEMINI-CLIENT", chalk.blue);
    }
    /**
     * Discovers the Google Cloud project ID.
     */
    async discoverProjectId() {
        if (this.googleCloudProject) {
            return this.googleCloudProject;
        }
        if (this.projectId) {
            return this.projectId;
        }
        try {
            const initialProjectId = "default-project";
            const loadResponse = (await this.callEndpoint("loadCodeAssist", {
                cloudaicompanionProject: initialProjectId,
                metadata: { duetProject: initialProjectId },
            }));
            if (loadResponse.cloudaicompanionProject) {
                this.projectId = loadResponse.cloudaicompanionProject;
                return loadResponse.cloudaicompanionProject;
            }
            const defaultTier = loadResponse.allowedTiers?.find((tier) => tier.isDefault);
            const tierId = defaultTier?.id ?? "free-tier";
            const onboardRequest = {
                tierId,
                cloudaicompanionProject: initialProjectId,
            };
            // Poll until operation is complete with timeout protection
            const MAX_RETRIES = 30;
            let retryCount = 0;
            let lroResponse;
            while (retryCount < MAX_RETRIES) {
                lroResponse = (await this.callEndpoint("onboardUser", onboardRequest));
                if (lroResponse.done) {
                    break;
                }
                await new Promise((resolve) => setTimeout(resolve, 1000));
                retryCount++;
            }
            if (!lroResponse?.done) {
                throw new Error("common:errors.geminiCli.onboardingTimeout");
            }
            this.projectId = lroResponse.response?.cloudaicompanionProject?.id ?? initialProjectId;
            return this.projectId;
        }
        catch (error) {
            this.logger.error("Failed to discover project ID", error);
            throw new Error("Could not discover project ID.");
        }
    }
    async callEndpoint(method, body) {
        const { token } = await this.authClient.getAccessToken();
        const response = await fetch(`${CODE_ASSIST_ENDPOINT}/${CODE_ASSIST_API_VERSION}:${method}`, {
            method: "POST",
            headers: {
                "Content-Type": "application/json",
                Authorization: `Bearer ${token}`,
            },
            body: JSON.stringify(body),
        });
        if (!response.ok) {
            const errorText = await response.text();
            throw new GeminiApiError(`API call failed with status ${response.status}: ${errorText}`, response.status, errorText);
        }
        return response.json();
    }
    /**
     * Get non-streaming completion from Gemini API.
     */
    async getCompletion(geminiCompletionRequest, isRetry = false) {
        try {
            const chunks = [];
            for await (const chunk of this.streamContent(geminiCompletionRequest, isRetry)) {
                chunks.push(chunk);
            }
            let content = "";
            const tool_calls = [];
            let usage;
            for (const chunk of chunks) {
                if (chunk.choices[0].delta.content) {
                    content += chunk.choices[0].delta.content;
                }
                if (chunk.choices[0].delta.tool_calls) {
                    tool_calls.push(...chunk.choices[0].delta.tool_calls);
                }
                if (chunk.usage) {
                    usage = {
                        inputTokens: chunk.usage.prompt_tokens,
                        outputTokens: chunk.usage.completion_tokens,
                    };
                }
            }
            return {
                content,
                tool_calls: tool_calls.length > 0 ? tool_calls : undefined,
                usage,
            };
        }
        catch (error) {
            if (error instanceof GeminiApiError &&
                !this.disableAutoModelSwitch &&
                this.autoSwitcher.isRateLimitError(error.statusCode) &&
                this.autoSwitcher.shouldAttemptFallback(geminiCompletionRequest.model)) {
                // Attempt fallback using auto-switching helper
                return await this.autoSwitcher.handleNonStreamingFallback(geminiCompletionRequest.model, error.statusCode, geminiCompletionRequest, async (model, data) => {
                    const updatedRequest = { ...data, model };
                    return await this.getCompletion(updatedRequest, isRetry);
                });
            }
            throw error;
        }
    }
    /**
     * Stream content from Gemini API.
     */
    async *streamContent(geminiCompletionRequest, isRetry = false) {
        try {
            yield* this.streamContentInternal(geminiCompletionRequest, isRetry);
        }
        catch (error) {
            if (error instanceof GeminiApiError &&
                !this.disableAutoModelSwitch &&
                this.autoSwitcher.isRateLimitError(error.statusCode) &&
                this.autoSwitcher.shouldAttemptFallback(geminiCompletionRequest.model)) {
                // eslint-disable-next-line @typescript-eslint/no-this-alias
                const self = this;
                yield* this.autoSwitcher.handleStreamingFallback(geminiCompletionRequest.model, error.statusCode, geminiCompletionRequest, async function* (model, data) {
                    const updatedRequest = { ...data, model };
                    // Create new client instance to reset firstChunk state
                    const fallbackClient = new GeminiApiClient(self.authClient, self.googleCloudProject, self.disableAutoModelSwitch);
                    yield* fallbackClient.streamContent(updatedRequest, isRetry);
                }, "openai");
                return;
            }
            throw error;
        }
    }
    /**
     * Internal streaming method with no retry logic
     */
    async *streamContentInternal(geminiCompletionRequest, isRetry = false) {
        const { token } = await this.authClient.getAccessToken();
        const response = await fetch(`${CODE_ASSIST_ENDPOINT}/${CODE_ASSIST_API_VERSION}:streamGenerateContent?alt=sse`, {
            method: "POST",
            headers: {
                "Content-Type": "application/json",
                Authorization: `Bearer ${token}`,
            },
            body: JSON.stringify(geminiCompletionRequest),
        });
        if (!response.ok) {
            if (response.status === 401 && !isRetry) {
                this.logger.info("Got 401 error, forcing token refresh and retrying...");
                this.authClient.credentials.access_token = undefined;
                yield* this.streamContentInternal(geminiCompletionRequest, true);
                return;
            }
            const errorText = await response.text();
            throw new GeminiApiError(`Stream request failed: ${response.status} ${errorText}`, response.status, errorText);
        }
        if (!response.body) {
            throw new Error("Response has no body");
        }
        let toolCallId = undefined;
        let usageData;
        let thinkingInProgress = false;
        for await (const jsonData of this.parseSSEStream(response.body)) {
            const candidate = jsonData.response?.candidates?.[0];
            if (candidate?.content?.parts) {
                for (const part of candidate.content.parts) {
                    if ("text" in part) {
                        // Handle text content
                        if (part.thought === true) {
                            // Handle thinking content from Gemini
                            const thinkingText = part.text;
                            const delta = {};
                            if (!thinkingInProgress) {
                                delta.content = "<thinking>\n";
                                if (this.firstChunk) {
                                    delta.role = "assistant";
                                    this.firstChunk = false;
                                }
                                yield this.createOpenAIChunk(delta, geminiCompletionRequest.model);
                                thinkingInProgress = true;
                            }
                            const thinkingDelta = { content: thinkingText };
                            yield this.createOpenAIChunk(thinkingDelta, geminiCompletionRequest.model);
                        }
                        else {
                            // Handle regular content - only if it's not a thinking part
                            if (thinkingInProgress) {
                                // Close thinking tag before first real content if needed
                                const closingDelta = { content: "\n</thinking>\n\n" };
                                yield this.createOpenAIChunk(closingDelta, geminiCompletionRequest.model);
                                thinkingInProgress = false;
                            }
                            const delta = { content: part.text };
                            if (this.firstChunk) {
                                delta.role = "assistant";
                                this.firstChunk = false;
                            }
                            yield this.createOpenAIChunk(delta, geminiCompletionRequest.model);
                        }
                    }
                    else if ("functionCall" in part) {
                        // Handle function calls from Gemini
                        if (thinkingInProgress) {
                            // Close thinking tag before function call if needed
                            const closingDelta = { content: "\n</thinking>\n\n" };
                            yield this.createOpenAIChunk(closingDelta, geminiCompletionRequest.model);
                            thinkingInProgress = false;
                        }
                        toolCallId = `call_${crypto.randomUUID()}`;
                        const delta = {
                            tool_calls: [{
                                    index: 0,
                                    id: toolCallId,
                                    type: "function",
                                    function: {
                                        name: part.functionCall.name,
                                        arguments: JSON.stringify(part.functionCall.args)
                                    }
                                }]
                        };
                        if (this.firstChunk) {
                            delta.role = "assistant";
                            delta.content = null;
                            this.firstChunk = false;
                        }
                        yield this.createOpenAIChunk(delta, geminiCompletionRequest.model);
                    }
                }
            }
            if (jsonData.response?.usageMetadata) {
                const usage = jsonData.response.usageMetadata;
                const prompt_tokens = usage.promptTokenCount ?? 0;
                const completion_tokens = usage.candidatesTokenCount ?? 0;
                usageData = {
                    prompt_tokens,
                    completion_tokens,
                    total_tokens: prompt_tokens + completion_tokens,
                };
            }
        }
        // Send final chunk with usage data
        const finishReason = toolCallId ? "tool_calls" : "stop";
        const finalChunk = this.createOpenAIChunk({}, geminiCompletionRequest.model, finishReason);
        if (usageData) {
            finalChunk.usage = usageData;
        }
        yield finalChunk;
    }
    /**
     * Creates an OpenAI stream chunk with the given delta
     */
    createOpenAIChunk(delta, modelId, finishReason = null) {
        return {
            id: this.chatID,
            object: OPENAI_CHAT_COMPLETION_OBJECT,
            created: this.creationTime,
            model: modelId,
            choices: [{
                    index: 0,
                    delta,
                    finish_reason: finishReason,
                    logprobs: null
                }],
            usage: null
        };
    }
    /**
     * Parses a server-sent event (SSE) stream from the Gemini API.
     */
    async *parseSSEStream(stream) {
        const reader = stream.pipeThrough(new TextDecoderStream()).getReader();
        let buffer = "";
        let objectBuffer = "";
        while (true) {
            const { done, value } = await reader.read();
            if (done) {
                if (objectBuffer) {
                    try {
                        yield JSON.parse(objectBuffer);
                    }
                    catch (e) {
                        this.logger.error("Error parsing final SSE JSON object", e);
                    }
                }
                break;
            }
            buffer += value;
            const lines = buffer.split("\n");
            buffer = lines.pop() || "";
            for (const line of lines) {
                if (line.trim() === "") {
                    if (objectBuffer) {
                        try {
                            yield JSON.parse(objectBuffer);
                        }
                        catch (e) {
                            this.logger.error("Error parsing SSE JSON object", e);
                        }
                        objectBuffer = "";
                    }
                }
                else if (line.startsWith("data: ")) {
                    objectBuffer += line.substring(6);
                }
            }
        }
    }
}
//# sourceMappingURL=client.js.map