/**
 * Type for request data that can be retried with a different model
 */
export interface RetryableRequestData {
    model: string;
    [key: string]: unknown;
}
/**
 * Type for retry functions for non-streaming requests
 */
type NonStreamingRetryFunction = (model: string, data: RetryableRequestData) => Promise<unknown>;
/**
 * Type for retry functions for streaming requests
 */
type StreamingRetryFunction = (model: string, data: RetryableRequestData) => AsyncIterable<unknown>;
/**
 * Helper class for automatic model switching when rate limits are encountered
 * Provides fallback mechanism with cooldown management to avoid repeated rate limit hits
 */
export declare class AutoModelSwitchingHelper {
    private static instance;
    private cooldownState;
    private logger;
    /**
   * Singleton pattern - get the global instance
   */
    static getInstance(): AutoModelSwitchingHelper;
    /**
   * Get the fallback model for a given model
   * @param {string} model - Current model that hit rate limits
   * @returns {string | null} Fallback model or null if no fallback available
   */
    getFallbackModel(model: string): string | null;
    /**
   * Check if an error is a rate limit error based on status code
   * @param {number} statusCode - HTTP status code from the error
   * @returns {boolean} True if the status code indicates rate limiting
   */
    isRateLimitError(statusCode: number): boolean;
    /**
   * Check if a status code indicates rate limiting
   * @param {number} statusCode - HTTP status code to check
   * @returns {boolean} True if status code is in RATE_LIMIT_STATUS_CODES
   */
    isRateLimitStatus(statusCode: number): boolean;
    /**
   * Determine if fallback should be attempted for a model
   * @param {string} model - Model to check for fallback availability
   * @returns {boolean} True if fallback should be attempted
   */
    shouldAttemptFallback(model: string): boolean;
    /**
   * Create a downgrade notification message with status codes
   * @param {string} fromModel - Original model that was rate limited
   * @param {string} toModel - Fallback model being used
   * @param {number} statusCode - HTTP status code that triggered the switch
   * @returns {string} Formatted downgrade notification message
   */
    createDowngradeNotification(fromModel: string, toModel: string, statusCode: number): string;
    /**
   * Create an upgrade notification message for rate limit recovery
   * @param {string} model - Model that is now available again
   * @returns {string} Formatted upgrade notification message
   */
    createUpgradeNotification(model: string): string;
    /**
   * Add a model to cooldown state when it hits rate limits
   * @param {string} model - Model that encountered rate limits
   * @param {number} statusCode - HTTP status code that triggered rate limiting
   */
    addRateLimitedModel(model: string, statusCode: number): void;
    /**
   * Check if a model is currently in cooldown
   * @param {string} model - Model to check cooldown status
   * @returns {boolean} True if model is in cooldown period
   */
    isModelInCooldown(model: string): boolean;
    /**
   * Get the best available model considering cooldown states
   * @param {string} preferredModel - Initially preferred model
   * @returns {string} Best available model that's not in cooldown
   */
    getBestAvailableModel(preferredModel: string): string;
    /**
   * Handle fallback for non-streaming requests
   * @param {string} originalModel - The model that encountered rate limits
   * @param {number} statusCode - HTTP status code from the rate limit error
   * @param {any} requestData - Original request data to retry with fallback model
   * @param {Function} retryFunction - Function to call for retry with new model
   * @returns {Promise<any>} Result from the retry attempt
   */
    handleNonStreamingFallback(originalModel: string, statusCode: number, requestData: RetryableRequestData, retryFunction: NonStreamingRetryFunction): Promise<unknown>;
    /**
   * Handle fallback for streaming requests
   * @param {string} originalModel - The model that encountered rate limits
   * @param {number} statusCode - HTTP status code from the rate limit error
   * @param {any} requestData - Original request data to retry with fallback model
   * @param {Function} retryFunction - Function to call for retry with new model
   * @param {string} streamFormat - Format for streaming ('openai' or 'anthropic')
   * @returns {AsyncIterable<any>} Stream from the retry attempt with notification
   */
    handleStreamingFallback(originalModel: string, statusCode: number, requestData: RetryableRequestData, retryFunction: StreamingRetryFunction, _streamFormat?: "openai" | "anthropic"): AsyncIterable<unknown>;
}
export {};
