"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const defaults = require("@aws-solutions-constructs/core");
const cdk = require("aws-cdk-lib");
const lib_1 = require("../lib");
const stepfunctions = require("aws-cdk-lib/aws-stepfunctions");
const ecs = require("aws-cdk-lib/aws-ecs");
const ec2 = require("aws-cdk-lib/aws-ec2");
const assertions_1 = require("aws-cdk-lib/assertions");
const clusterName = "custom-cluster-name";
const containerName = "custom-container-name";
const serviceName = "custom-service-name";
const familyName = "family-name";
const testCidr = "172.0.0.0/16";
test('Check for construct properties', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const construct = createFargateConstructWithNewResources(stack, publicApi);
    expect(construct.vpc).toBeDefined();
    expect(construct.service).toBeDefined();
    expect(construct.container).toBeDefined();
    expect(construct.stateMachine).toBeDefined();
    expect(construct.stateMachineLogGroup).toBeDefined();
    expect(construct.cloudwatchAlarms).toBeDefined();
});
test('Check for new service', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: serviceName,
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
});
test('Check for an existing service', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const existingVpc = defaults.getTestVpc(stack);
    const createFargateServiceResponse = defaults.CreateFargateService(stack, 'test', existingVpc, { clusterName }, defaults.fakeEcrRepoArn, undefined, { family: familyName }, { containerName }, { serviceName });
    new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        existingFargateServiceObject: createFargateServiceResponse.service,
        existingContainerDefinitionObject: createFargateServiceResponse.containerDefinition,
        existingVpc,
        stateMachineProps: testStateMachineProps(stack)
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::Service", {
        ServiceName: serviceName,
        LaunchType: 'FARGATE',
        DesiredCount: 2,
        DeploymentConfiguration: {
            MaximumPercent: 150,
            MinimumHealthyPercent: 75
        },
        PlatformVersion: ecs.FargatePlatformVersion.LATEST,
    });
});
test('Check for IAM startExecution policy', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "states:StartExecution",
                    Effect: "Allow",
                    Resource: {
                        Ref: "testconstructStateMachine3333AAA9"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        PolicyName: "testconstructtaskdefTaskRoleDefaultPolicyF34A1535",
        Roles: [
            {
                Ref: "testconstructtaskdefTaskRoleC60414C4"
            }
        ]
    });
});
test('Check for public/private VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    createFargateConstructWithNewResources(stack, publicApi);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: testCidr
    });
    template.hasResourceProperties('AWS::EC2::InternetGateway', {});
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::StepFunctions::StateMachine', 1);
    template.resourceCountIs('AWS::ECS::Service', 1);
});
test('Check for isolated VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    createFargateConstructWithNewResources(stack, publicApi);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: testCidr
    });
    defaults.expectNonexistence(stack, 'AWS::EC2::InternetGateway', {});
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.resourceCountIs('AWS::StepFunctions::StateMachine', 1);
    template.resourceCountIs('AWS::ECS::Service', 1);
});
test('Check for an existing VPC', () => {
    const stack = new cdk.Stack();
    const publicApi = false;
    const existingVpc = defaults.getTestVpc(stack, publicApi);
    new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        existingVpc,
        stateMachineProps: testStateMachineProps(stack)
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: "172.168.0.0/16"
    });
    template.resourceCountIs("AWS::EC2::VPC", 1);
});
test('Check for custom ARN resource', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const customEnvName = "TEST_CUSTOM_ARN";
    new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
        stateMachineEnvironmentVariableName: customEnvName
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ECS::TaskDefinition", {
        Family: familyName,
        ContainerDefinitions: [
            {
                Environment: [
                    {
                        Name: customEnvName,
                        Value: {
                            Ref: "testconstructStateMachine3333AAA9"
                        }
                    }
                ],
                Essential: true,
                Image: {
                    "Fn::Join": [
                        "",
                        [
                            "123456789012.dkr.ecr.us-east-1.",
                            {
                                Ref: "AWS::URLSuffix"
                            },
                            "/fake-repo:latest"
                        ]
                    ]
                },
                MemoryReservation: 512,
                Name: containerName,
                PortMappings: [
                    {
                        ContainerPort: 8080,
                        Protocol: "tcp"
                    }
                ]
            }
        ]
    });
});
test('Check for no cloudwatch creation', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const construct = new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
        createCloudWatchAlarms: false
    });
    expect(construct.cloudwatchAlarms).not.toBeDefined();
    defaults.expectNonexistence(stack, "AWS::CloudWatch::Alarm", {
        ComparisonOperator: "GreaterThanOrEqualToThreshold",
        EvaluationPeriods: 1,
        AlarmDescription: "Alarm for the number of executions that aborted exceeded the threshold of 1. ",
        Dimensions: [
            {
                Name: "StateMachineArn",
                Value: {
                    Ref: "testconstructStateMachine3333AAA9"
                }
            }
        ],
        MetricName: "ExecutionsAborted",
        Namespace: "AWS/States",
        Period: 300,
        Statistic: "Maximum",
        Threshold: 1
    });
});
test('Check for custom log group props', () => {
    const stack = new cdk.Stack();
    const publicApi = true;
    const logGroupName = "custom-log-group";
    new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
        logGroupProps: {
            logGroupName
        }
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Logs::LogGroup", {
        LogGroupName: logGroupName
    });
});
function createFargateConstructWithNewResources(stack, publicApi) {
    return new lib_1.FargateToStepfunctions(stack, 'test-construct', {
        publicApi,
        ecrRepositoryArn: defaults.fakeEcrRepoArn,
        vpcProps: { ipAddresses: ec2.IpAddresses.cidr(testCidr) },
        clusterProps: { clusterName },
        containerDefinitionProps: { containerName },
        fargateTaskDefinitionProps: { family: familyName },
        fargateServiceProps: { serviceName },
        stateMachineProps: testStateMachineProps(stack),
    });
}
function testStateMachineProps(stack, userProps) {
    const defaultTestProp = { definition: new stepfunctions.Pass(stack, 'StartState') };
    return defaults.consolidateProps(defaultTestProp, userProps);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZmFyZ2F0ZS1zdGVwZnVuY3Rpb25zLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJmYXJnYXRlLXN0ZXBmdW5jdGlvbnMudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7O0FBRUgsMkRBQTJEO0FBQzNELG1DQUFtQztBQUNuQyxnQ0FBZ0Q7QUFDaEQsK0RBQStEO0FBQy9ELDJDQUEyQztBQUMzQywyQ0FBMkM7QUFDM0MsdURBQWtEO0FBRWxELE1BQU0sV0FBVyxHQUFHLHFCQUFxQixDQUFDO0FBQzFDLE1BQU0sYUFBYSxHQUFHLHVCQUF1QixDQUFDO0FBQzlDLE1BQU0sV0FBVyxHQUFHLHFCQUFxQixDQUFDO0FBQzFDLE1BQU0sVUFBVSxHQUFHLGFBQWEsQ0FBQztBQUNqQyxNQUFNLFFBQVEsR0FBRyxjQUFjLENBQUM7QUFFaEMsSUFBSSxDQUFDLGdDQUFnQyxFQUFFLEdBQUcsRUFBRTtJQUMxQyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUM7SUFFdkIsTUFBTSxTQUFTLEdBQUcsc0NBQXNDLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBRTNFLE1BQU0sQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDcEMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUN4QyxNQUFNLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQzFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsWUFBWSxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDN0MsTUFBTSxDQUFDLFNBQVMsQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLFdBQVcsRUFBRSxDQUFDO0lBQ3JELE1BQU0sQ0FBQyxTQUFTLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxXQUFXLEVBQUUsQ0FBQztBQUNuRCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx1QkFBdUIsRUFBRSxHQUFHLEVBQUU7SUFDakMsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsTUFBTSxTQUFTLEdBQUcsS0FBSyxDQUFDO0lBRXhCLHNDQUFzQyxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztJQUV6RCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxRQUFRLENBQUMscUJBQXFCLENBQUMsbUJBQW1CLEVBQUU7UUFDbEQsV0FBVyxFQUFFLFdBQVc7UUFDeEIsVUFBVSxFQUFFLFNBQVM7UUFDckIsWUFBWSxFQUFFLENBQUM7UUFDZix1QkFBdUIsRUFBRTtZQUN2QixjQUFjLEVBQUUsR0FBRztZQUNuQixxQkFBcUIsRUFBRSxFQUFFO1NBQzFCO1FBQ0QsZUFBZSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxNQUFNO0tBQ25ELENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLCtCQUErQixFQUFFLEdBQUcsRUFBRTtJQUN6QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLFNBQVMsR0FBRyxJQUFJLENBQUM7SUFFdkIsTUFBTSxXQUFXLEdBQUcsUUFBUSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUUvQyxNQUFNLDRCQUE0QixHQUFHLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQ3RFLE1BQU0sRUFDTixXQUFXLEVBQ1gsRUFBRSxXQUFXLEVBQUUsRUFDZixRQUFRLENBQUMsY0FBYyxFQUN2QixTQUFTLEVBQ1QsRUFBRSxNQUFNLEVBQUUsVUFBVSxFQUFFLEVBQ3RCLEVBQUUsYUFBYSxFQUFFLEVBQ2pCLEVBQUUsV0FBVyxFQUFFLENBQUMsQ0FBQztJQUVuQixJQUFJLDRCQUFzQixDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtRQUNsRCxTQUFTO1FBQ1QsNEJBQTRCLEVBQUUsNEJBQTRCLENBQUMsT0FBTztRQUNsRSxpQ0FBaUMsRUFBRSw0QkFBNEIsQ0FBQyxtQkFBbUI7UUFDbkYsV0FBVztRQUNYLGlCQUFpQixFQUFFLHFCQUFxQixDQUFDLEtBQUssQ0FBQztLQUNoRCxDQUFDLENBQUM7SUFFSCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxRQUFRLENBQUMscUJBQXFCLENBQUMsbUJBQW1CLEVBQUU7UUFDbEQsV0FBVyxFQUFFLFdBQVc7UUFDeEIsVUFBVSxFQUFFLFNBQVM7UUFDckIsWUFBWSxFQUFFLENBQUM7UUFDZix1QkFBdUIsRUFBRTtZQUN2QixjQUFjLEVBQUUsR0FBRztZQUNuQixxQkFBcUIsRUFBRSxFQUFFO1NBQzFCO1FBQ0QsZUFBZSxFQUFFLEdBQUcsQ0FBQyxzQkFBc0IsQ0FBQyxNQUFNO0tBQ25ELENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFDQUFxQyxFQUFFLEdBQUcsRUFBRTtJQUMvQyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUM7SUFFeEIsc0NBQXNDLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBRXpELE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxrQkFBa0IsRUFBRTtRQUNqRCxjQUFjLEVBQUU7WUFDZCxTQUFTLEVBQUU7Z0JBQ1Q7b0JBQ0UsTUFBTSxFQUFFLHVCQUF1QjtvQkFDL0IsTUFBTSxFQUFFLE9BQU87b0JBQ2YsUUFBUSxFQUFFO3dCQUNSLEdBQUcsRUFBRSxtQ0FBbUM7cUJBQ3pDO2lCQUNGO2FBQ0Y7WUFDRCxPQUFPLEVBQUUsWUFBWTtTQUN0QjtRQUNELFVBQVUsRUFBRSxtREFBbUQ7UUFDL0QsS0FBSyxFQUFFO1lBQ0w7Z0JBQ0UsR0FBRyxFQUFFLHNDQUFzQzthQUM1QztTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsOEJBQThCLEVBQUUsR0FBRyxFQUFFO0lBQ3hDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQztJQUV2QixzQ0FBc0MsQ0FBQyxLQUFLLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFFekQsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGVBQWUsRUFBRTtRQUM5QyxTQUFTLEVBQUUsUUFBUTtLQUNwQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsMkJBQTJCLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDaEUsUUFBUSxDQUFDLGVBQWUsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFDN0MsUUFBUSxDQUFDLGVBQWUsQ0FBQyxrQ0FBa0MsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUNoRSxRQUFRLENBQUMsZUFBZSxDQUFDLG1CQUFtQixFQUFFLENBQUMsQ0FBQyxDQUFDO0FBQ25ELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHdCQUF3QixFQUFFLEdBQUcsRUFBRTtJQUNsQyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLEVBQUUsQ0FBQztJQUM5QixNQUFNLFNBQVMsR0FBRyxLQUFLLENBQUM7SUFFeEIsc0NBQXNDLENBQUMsS0FBSyxFQUFFLFNBQVMsQ0FBQyxDQUFDO0lBRXpELE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxlQUFlLEVBQUU7UUFDOUMsU0FBUyxFQUFFLFFBQVE7S0FDcEIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSwyQkFBMkIsRUFBRSxFQUFFLENBQUMsQ0FBQztJQUNwRSxRQUFRLENBQUMsZUFBZSxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztJQUM3QyxRQUFRLENBQUMsZUFBZSxDQUFDLGtDQUFrQyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ2hFLFFBQVEsQ0FBQyxlQUFlLENBQUMsbUJBQW1CLEVBQUUsQ0FBQyxDQUFDLENBQUM7QUFDbkQsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsMkJBQTJCLEVBQUUsR0FBRyxFQUFFO0lBQ3JDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sU0FBUyxHQUFHLEtBQUssQ0FBQztJQUV4QixNQUFNLFdBQVcsR0FBRyxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssRUFBRSxTQUFTLENBQUMsQ0FBQztJQUUxRCxJQUFJLDRCQUFzQixDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtRQUNsRCxTQUFTO1FBQ1QsZ0JBQWdCLEVBQUUsUUFBUSxDQUFDLGNBQWM7UUFDekMsWUFBWSxFQUFFLEVBQUUsV0FBVyxFQUFFO1FBQzdCLHdCQUF3QixFQUFFLEVBQUUsYUFBYSxFQUFFO1FBQzNDLDBCQUEwQixFQUFFLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRTtRQUNsRCxtQkFBbUIsRUFBRSxFQUFFLFdBQVcsRUFBRTtRQUNwQyxXQUFXO1FBQ1gsaUJBQWlCLEVBQUUscUJBQXFCLENBQUMsS0FBSyxDQUFDO0tBQ2hELENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxlQUFlLEVBQUU7UUFDOUMsU0FBUyxFQUFFLGdCQUFnQjtLQUM1QixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsZUFBZSxDQUFDLGVBQWUsRUFBRSxDQUFDLENBQUMsQ0FBQztBQUMvQyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywrQkFBK0IsRUFBRSxHQUFHLEVBQUU7SUFDekMsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDOUIsTUFBTSxTQUFTLEdBQUcsSUFBSSxDQUFDO0lBQ3ZCLE1BQU0sYUFBYSxHQUFHLGlCQUFpQixDQUFDO0lBRXhDLElBQUksNEJBQXNCLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFO1FBQ2xELFNBQVM7UUFDVCxnQkFBZ0IsRUFBRSxRQUFRLENBQUMsY0FBYztRQUN6QyxRQUFRLEVBQUUsRUFBRSxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUU7UUFDekQsWUFBWSxFQUFFLEVBQUUsV0FBVyxFQUFFO1FBQzdCLHdCQUF3QixFQUFFLEVBQUUsYUFBYSxFQUFFO1FBQzNDLDBCQUEwQixFQUFFLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRTtRQUNsRCxtQkFBbUIsRUFBRSxFQUFFLFdBQVcsRUFBRTtRQUNwQyxpQkFBaUIsRUFBRSxxQkFBcUIsQ0FBQyxLQUFLLENBQUM7UUFDL0MsbUNBQW1DLEVBQUUsYUFBYTtLQUNuRCxDQUFDLENBQUM7SUFFSCxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxRQUFRLENBQUMscUJBQXFCLENBQUMsMEJBQTBCLEVBQUU7UUFDekQsTUFBTSxFQUFFLFVBQVU7UUFDbEIsb0JBQW9CLEVBQUU7WUFDcEI7Z0JBQ0UsV0FBVyxFQUFFO29CQUNYO3dCQUNFLElBQUksRUFBRSxhQUFhO3dCQUNuQixLQUFLLEVBQUU7NEJBQ0wsR0FBRyxFQUFFLG1DQUFtQzt5QkFDekM7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsU0FBUyxFQUFFLElBQUk7Z0JBQ2YsS0FBSyxFQUFFO29CQUNMLFVBQVUsRUFBRTt3QkFDVixFQUFFO3dCQUNGOzRCQUNFLGlDQUFpQzs0QkFDakM7Z0NBQ0UsR0FBRyxFQUFFLGdCQUFnQjs2QkFDdEI7NEJBQ0QsbUJBQW1CO3lCQUNwQjtxQkFDRjtpQkFDRjtnQkFDRCxpQkFBaUIsRUFBRSxHQUFHO2dCQUN0QixJQUFJLEVBQUUsYUFBYTtnQkFDbkIsWUFBWSxFQUFFO29CQUNaO3dCQUNFLGFBQWEsRUFBRSxJQUFJO3dCQUNuQixRQUFRLEVBQUUsS0FBSztxQkFDaEI7aUJBQ0Y7YUFDRjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsa0NBQWtDLEVBQUUsR0FBRyxFQUFFO0lBQzVDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQztJQUV2QixNQUFNLFNBQVMsR0FBRyxJQUFJLDRCQUFzQixDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtRQUNwRSxTQUFTO1FBQ1QsZ0JBQWdCLEVBQUUsUUFBUSxDQUFDLGNBQWM7UUFDekMsUUFBUSxFQUFFLEVBQUUsV0FBVyxFQUFFLEdBQUcsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1FBQ3pELFlBQVksRUFBRSxFQUFFLFdBQVcsRUFBRTtRQUM3Qix3QkFBd0IsRUFBRSxFQUFFLGFBQWEsRUFBRTtRQUMzQywwQkFBMEIsRUFBRSxFQUFFLE1BQU0sRUFBRSxVQUFVLEVBQUU7UUFDbEQsbUJBQW1CLEVBQUUsRUFBRSxXQUFXLEVBQUU7UUFDcEMsaUJBQWlCLEVBQUUscUJBQXFCLENBQUMsS0FBSyxDQUFDO1FBQy9DLHNCQUFzQixFQUFFLEtBQUs7S0FDOUIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxDQUFDLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxXQUFXLEVBQUUsQ0FBQztJQUVyRCxRQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLHdCQUF3QixFQUFFO1FBQzNELGtCQUFrQixFQUFFLCtCQUErQjtRQUNuRCxpQkFBaUIsRUFBRSxDQUFDO1FBQ3BCLGdCQUFnQixFQUFFLCtFQUErRTtRQUNqRyxVQUFVLEVBQUU7WUFDVjtnQkFDRSxJQUFJLEVBQUUsaUJBQWlCO2dCQUN2QixLQUFLLEVBQUU7b0JBQ0wsR0FBRyxFQUFFLG1DQUFtQztpQkFDekM7YUFDRjtTQUNGO1FBQ0QsVUFBVSxFQUFFLG1CQUFtQjtRQUMvQixTQUFTLEVBQUUsWUFBWTtRQUN2QixNQUFNLEVBQUUsR0FBRztRQUNYLFNBQVMsRUFBRSxTQUFTO1FBQ3BCLFNBQVMsRUFBRSxDQUFDO0tBQ2IsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsa0NBQWtDLEVBQUUsR0FBRyxFQUFFO0lBQzVDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzlCLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQztJQUN2QixNQUFNLFlBQVksR0FBRyxrQkFBa0IsQ0FBQztJQUV4QyxJQUFJLDRCQUFzQixDQUFDLEtBQUssRUFBRSxnQkFBZ0IsRUFBRTtRQUNsRCxTQUFTO1FBQ1QsZ0JBQWdCLEVBQUUsUUFBUSxDQUFDLGNBQWM7UUFDekMsUUFBUSxFQUFFLEVBQUUsV0FBVyxFQUFFLEdBQUcsQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFO1FBQ3pELFlBQVksRUFBRSxFQUFFLFdBQVcsRUFBRTtRQUM3Qix3QkFBd0IsRUFBRSxFQUFFLGFBQWEsRUFBRTtRQUMzQywwQkFBMEIsRUFBRSxFQUFFLE1BQU0sRUFBRSxVQUFVLEVBQUU7UUFDbEQsbUJBQW1CLEVBQUUsRUFBRSxXQUFXLEVBQUU7UUFDcEMsaUJBQWlCLEVBQUUscUJBQXFCLENBQUMsS0FBSyxDQUFDO1FBQy9DLGFBQWEsRUFBRTtZQUNiLFlBQVk7U0FDYjtLQUNGLENBQUMsQ0FBQztJQUVILE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxxQkFBcUIsRUFBRTtRQUNwRCxZQUFZLEVBQUUsWUFBWTtLQUMzQixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILFNBQVMsc0NBQXNDLENBQUMsS0FBZ0IsRUFBRSxTQUFrQjtJQUNsRixPQUFPLElBQUksNEJBQXNCLENBQUMsS0FBSyxFQUFFLGdCQUFnQixFQUFFO1FBQ3pELFNBQVM7UUFDVCxnQkFBZ0IsRUFBRSxRQUFRLENBQUMsY0FBYztRQUN6QyxRQUFRLEVBQUUsRUFBRSxXQUFXLEVBQUUsR0FBRyxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUU7UUFDekQsWUFBWSxFQUFFLEVBQUUsV0FBVyxFQUFFO1FBQzdCLHdCQUF3QixFQUFFLEVBQUUsYUFBYSxFQUFFO1FBQzNDLDBCQUEwQixFQUFFLEVBQUUsTUFBTSxFQUFFLFVBQVUsRUFBRTtRQUNsRCxtQkFBbUIsRUFBRSxFQUFFLFdBQVcsRUFBRTtRQUNwQyxpQkFBaUIsRUFBRSxxQkFBcUIsQ0FBQyxLQUFLLENBQUM7S0FDaEQsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQUVELFNBQVMscUJBQXFCLENBQUMsS0FBZ0IsRUFBRSxTQUEyQztJQUUxRixNQUFNLGVBQWUsR0FBRyxFQUFFLFVBQVUsRUFBRSxJQUFJLGFBQWEsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLFlBQVksQ0FBQyxFQUFFLENBQUM7SUFFcEYsT0FBTyxRQUFRLENBQUMsZ0JBQWdCLENBQUMsZUFBZSxFQUFFLFNBQVMsQ0FBQyxDQUFDO0FBQy9ELENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG5pbXBvcnQgKiBhcyBkZWZhdWx0cyBmcm9tICdAYXdzLXNvbHV0aW9ucy1jb25zdHJ1Y3RzL2NvcmUnO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuaW1wb3J0IHsgRmFyZ2F0ZVRvU3RlcGZ1bmN0aW9ucyB9IGZyb20gXCIuLi9saWJcIjtcbmltcG9ydCAqIGFzIHN0ZXBmdW5jdGlvbnMgZnJvbSAnYXdzLWNkay1saWIvYXdzLXN0ZXBmdW5jdGlvbnMnO1xuaW1wb3J0ICogYXMgZWNzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1lY3MnO1xuaW1wb3J0ICogYXMgZWMyIGZyb20gJ2F3cy1jZGstbGliL2F3cy1lYzInO1xuaW1wb3J0IHsgVGVtcGxhdGUgfSBmcm9tICdhd3MtY2RrLWxpYi9hc3NlcnRpb25zJztcblxuY29uc3QgY2x1c3Rlck5hbWUgPSBcImN1c3RvbS1jbHVzdGVyLW5hbWVcIjtcbmNvbnN0IGNvbnRhaW5lck5hbWUgPSBcImN1c3RvbS1jb250YWluZXItbmFtZVwiO1xuY29uc3Qgc2VydmljZU5hbWUgPSBcImN1c3RvbS1zZXJ2aWNlLW5hbWVcIjtcbmNvbnN0IGZhbWlseU5hbWUgPSBcImZhbWlseS1uYW1lXCI7XG5jb25zdCB0ZXN0Q2lkciA9IFwiMTcyLjAuMC4wLzE2XCI7XG5cbnRlc3QoJ0NoZWNrIGZvciBjb25zdHJ1Y3QgcHJvcGVydGllcycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHB1YmxpY0FwaSA9IHRydWU7XG5cbiAgY29uc3QgY29uc3RydWN0ID0gY3JlYXRlRmFyZ2F0ZUNvbnN0cnVjdFdpdGhOZXdSZXNvdXJjZXMoc3RhY2ssIHB1YmxpY0FwaSk7XG5cbiAgZXhwZWN0KGNvbnN0cnVjdC52cGMpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChjb25zdHJ1Y3Quc2VydmljZSkudG9CZURlZmluZWQoKTtcbiAgZXhwZWN0KGNvbnN0cnVjdC5jb250YWluZXIpLnRvQmVEZWZpbmVkKCk7XG4gIGV4cGVjdChjb25zdHJ1Y3Quc3RhdGVNYWNoaW5lKS50b0JlRGVmaW5lZCgpO1xuICBleHBlY3QoY29uc3RydWN0LnN0YXRlTWFjaGluZUxvZ0dyb3VwKS50b0JlRGVmaW5lZCgpO1xuICBleHBlY3QoY29uc3RydWN0LmNsb3Vkd2F0Y2hBbGFybXMpLnRvQmVEZWZpbmVkKCk7XG59KTtcblxudGVzdCgnQ2hlY2sgZm9yIG5ldyBzZXJ2aWNlJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY29uc3QgcHVibGljQXBpID0gZmFsc2U7XG5cbiAgY3JlYXRlRmFyZ2F0ZUNvbnN0cnVjdFdpdGhOZXdSZXNvdXJjZXMoc3RhY2ssIHB1YmxpY0FwaSk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OkVDUzo6U2VydmljZVwiLCB7XG4gICAgU2VydmljZU5hbWU6IHNlcnZpY2VOYW1lLFxuICAgIExhdW5jaFR5cGU6ICdGQVJHQVRFJyxcbiAgICBEZXNpcmVkQ291bnQ6IDIsXG4gICAgRGVwbG95bWVudENvbmZpZ3VyYXRpb246IHtcbiAgICAgIE1heGltdW1QZXJjZW50OiAxNTAsXG4gICAgICBNaW5pbXVtSGVhbHRoeVBlcmNlbnQ6IDc1XG4gICAgfSxcbiAgICBQbGF0Zm9ybVZlcnNpb246IGVjcy5GYXJnYXRlUGxhdGZvcm1WZXJzaW9uLkxBVEVTVCxcbiAgfSk7XG59KTtcblxudGVzdCgnQ2hlY2sgZm9yIGFuIGV4aXN0aW5nIHNlcnZpY2UnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCBwdWJsaWNBcGkgPSB0cnVlO1xuXG4gIGNvbnN0IGV4aXN0aW5nVnBjID0gZGVmYXVsdHMuZ2V0VGVzdFZwYyhzdGFjayk7XG5cbiAgY29uc3QgY3JlYXRlRmFyZ2F0ZVNlcnZpY2VSZXNwb25zZSA9IGRlZmF1bHRzLkNyZWF0ZUZhcmdhdGVTZXJ2aWNlKHN0YWNrLFxuICAgICd0ZXN0JyxcbiAgICBleGlzdGluZ1ZwYyxcbiAgICB7IGNsdXN0ZXJOYW1lIH0sXG4gICAgZGVmYXVsdHMuZmFrZUVjclJlcG9Bcm4sXG4gICAgdW5kZWZpbmVkLFxuICAgIHsgZmFtaWx5OiBmYW1pbHlOYW1lIH0sXG4gICAgeyBjb250YWluZXJOYW1lIH0sXG4gICAgeyBzZXJ2aWNlTmFtZSB9KTtcblxuICBuZXcgRmFyZ2F0ZVRvU3RlcGZ1bmN0aW9ucyhzdGFjaywgJ3Rlc3QtY29uc3RydWN0Jywge1xuICAgIHB1YmxpY0FwaSxcbiAgICBleGlzdGluZ0ZhcmdhdGVTZXJ2aWNlT2JqZWN0OiBjcmVhdGVGYXJnYXRlU2VydmljZVJlc3BvbnNlLnNlcnZpY2UsXG4gICAgZXhpc3RpbmdDb250YWluZXJEZWZpbml0aW9uT2JqZWN0OiBjcmVhdGVGYXJnYXRlU2VydmljZVJlc3BvbnNlLmNvbnRhaW5lckRlZmluaXRpb24sXG4gICAgZXhpc3RpbmdWcGMsXG4gICAgc3RhdGVNYWNoaW5lUHJvcHM6IHRlc3RTdGF0ZU1hY2hpbmVQcm9wcyhzdGFjaylcbiAgfSk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OkVDUzo6U2VydmljZVwiLCB7XG4gICAgU2VydmljZU5hbWU6IHNlcnZpY2VOYW1lLFxuICAgIExhdW5jaFR5cGU6ICdGQVJHQVRFJyxcbiAgICBEZXNpcmVkQ291bnQ6IDIsXG4gICAgRGVwbG95bWVudENvbmZpZ3VyYXRpb246IHtcbiAgICAgIE1heGltdW1QZXJjZW50OiAxNTAsXG4gICAgICBNaW5pbXVtSGVhbHRoeVBlcmNlbnQ6IDc1XG4gICAgfSxcbiAgICBQbGF0Zm9ybVZlcnNpb246IGVjcy5GYXJnYXRlUGxhdGZvcm1WZXJzaW9uLkxBVEVTVCxcbiAgfSk7XG59KTtcblxudGVzdCgnQ2hlY2sgZm9yIElBTSBzdGFydEV4ZWN1dGlvbiBwb2xpY3knLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCBwdWJsaWNBcGkgPSBmYWxzZTtcblxuICBjcmVhdGVGYXJnYXRlQ29uc3RydWN0V2l0aE5ld1Jlc291cmNlcyhzdGFjaywgcHVibGljQXBpKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6SUFNOjpQb2xpY3lcIiwge1xuICAgIFBvbGljeURvY3VtZW50OiB7XG4gICAgICBTdGF0ZW1lbnQ6IFtcbiAgICAgICAge1xuICAgICAgICAgIEFjdGlvbjogXCJzdGF0ZXM6U3RhcnRFeGVjdXRpb25cIixcbiAgICAgICAgICBFZmZlY3Q6IFwiQWxsb3dcIixcbiAgICAgICAgICBSZXNvdXJjZToge1xuICAgICAgICAgICAgUmVmOiBcInRlc3Rjb25zdHJ1Y3RTdGF0ZU1hY2hpbmUzMzMzQUFBOVwiXG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICBdLFxuICAgICAgVmVyc2lvbjogXCIyMDEyLTEwLTE3XCJcbiAgICB9LFxuICAgIFBvbGljeU5hbWU6IFwidGVzdGNvbnN0cnVjdHRhc2tkZWZUYXNrUm9sZURlZmF1bHRQb2xpY3lGMzRBMTUzNVwiLFxuICAgIFJvbGVzOiBbXG4gICAgICB7XG4gICAgICAgIFJlZjogXCJ0ZXN0Y29uc3RydWN0dGFza2RlZlRhc2tSb2xlQzYwNDE0QzRcIlxuICAgICAgfVxuICAgIF1cbiAgfSk7XG59KTtcblxudGVzdCgnQ2hlY2sgZm9yIHB1YmxpYy9wcml2YXRlIFZQQycsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHB1YmxpY0FwaSA9IHRydWU7XG5cbiAgY3JlYXRlRmFyZ2F0ZUNvbnN0cnVjdFdpdGhOZXdSZXNvdXJjZXMoc3RhY2ssIHB1YmxpY0FwaSk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OkVDMjo6VlBDXCIsIHtcbiAgICBDaWRyQmxvY2s6IHRlc3RDaWRyXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFQzI6OkludGVybmV0R2F0ZXdheScsIHt9KTtcbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKCdBV1M6OkVDMjo6VlBDJywgMSk7XG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcygnQVdTOjpTdGVwRnVuY3Rpb25zOjpTdGF0ZU1hY2hpbmUnLCAxKTtcbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKCdBV1M6OkVDUzo6U2VydmljZScsIDEpO1xufSk7XG5cbnRlc3QoJ0NoZWNrIGZvciBpc29sYXRlZCBWUEMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCBwdWJsaWNBcGkgPSBmYWxzZTtcblxuICBjcmVhdGVGYXJnYXRlQ29uc3RydWN0V2l0aE5ld1Jlc291cmNlcyhzdGFjaywgcHVibGljQXBpKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6RUMyOjpWUENcIiwge1xuICAgIENpZHJCbG9jazogdGVzdENpZHJcbiAgfSk7XG5cbiAgZGVmYXVsdHMuZXhwZWN0Tm9uZXhpc3RlbmNlKHN0YWNrLCAnQVdTOjpFQzI6OkludGVybmV0R2F0ZXdheScsIHt9KTtcbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKCdBV1M6OkVDMjo6VlBDJywgMSk7XG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcygnQVdTOjpTdGVwRnVuY3Rpb25zOjpTdGF0ZU1hY2hpbmUnLCAxKTtcbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKCdBV1M6OkVDUzo6U2VydmljZScsIDEpO1xufSk7XG5cbnRlc3QoJ0NoZWNrIGZvciBhbiBleGlzdGluZyBWUEMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjaygpO1xuICBjb25zdCBwdWJsaWNBcGkgPSBmYWxzZTtcblxuICBjb25zdCBleGlzdGluZ1ZwYyA9IGRlZmF1bHRzLmdldFRlc3RWcGMoc3RhY2ssIHB1YmxpY0FwaSk7XG5cbiAgbmV3IEZhcmdhdGVUb1N0ZXBmdW5jdGlvbnMoc3RhY2ssICd0ZXN0LWNvbnN0cnVjdCcsIHtcbiAgICBwdWJsaWNBcGksXG4gICAgZWNyUmVwb3NpdG9yeUFybjogZGVmYXVsdHMuZmFrZUVjclJlcG9Bcm4sXG4gICAgY2x1c3RlclByb3BzOiB7IGNsdXN0ZXJOYW1lIH0sXG4gICAgY29udGFpbmVyRGVmaW5pdGlvblByb3BzOiB7IGNvbnRhaW5lck5hbWUgfSxcbiAgICBmYXJnYXRlVGFza0RlZmluaXRpb25Qcm9wczogeyBmYW1pbHk6IGZhbWlseU5hbWUgfSxcbiAgICBmYXJnYXRlU2VydmljZVByb3BzOiB7IHNlcnZpY2VOYW1lIH0sXG4gICAgZXhpc3RpbmdWcGMsXG4gICAgc3RhdGVNYWNoaW5lUHJvcHM6IHRlc3RTdGF0ZU1hY2hpbmVQcm9wcyhzdGFjaylcbiAgfSk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OkVDMjo6VlBDXCIsIHtcbiAgICBDaWRyQmxvY2s6IFwiMTcyLjE2OC4wLjAvMTZcIlxuICB9KTtcblxuICB0ZW1wbGF0ZS5yZXNvdXJjZUNvdW50SXMoXCJBV1M6OkVDMjo6VlBDXCIsIDEpO1xufSk7XG5cbnRlc3QoJ0NoZWNrIGZvciBjdXN0b20gQVJOIHJlc291cmNlJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY29uc3QgcHVibGljQXBpID0gdHJ1ZTtcbiAgY29uc3QgY3VzdG9tRW52TmFtZSA9IFwiVEVTVF9DVVNUT01fQVJOXCI7XG5cbiAgbmV3IEZhcmdhdGVUb1N0ZXBmdW5jdGlvbnMoc3RhY2ssICd0ZXN0LWNvbnN0cnVjdCcsIHtcbiAgICBwdWJsaWNBcGksXG4gICAgZWNyUmVwb3NpdG9yeUFybjogZGVmYXVsdHMuZmFrZUVjclJlcG9Bcm4sXG4gICAgdnBjUHJvcHM6IHsgaXBBZGRyZXNzZXM6IGVjMi5JcEFkZHJlc3Nlcy5jaWRyKHRlc3RDaWRyKSB9LFxuICAgIGNsdXN0ZXJQcm9wczogeyBjbHVzdGVyTmFtZSB9LFxuICAgIGNvbnRhaW5lckRlZmluaXRpb25Qcm9wczogeyBjb250YWluZXJOYW1lIH0sXG4gICAgZmFyZ2F0ZVRhc2tEZWZpbml0aW9uUHJvcHM6IHsgZmFtaWx5OiBmYW1pbHlOYW1lIH0sXG4gICAgZmFyZ2F0ZVNlcnZpY2VQcm9wczogeyBzZXJ2aWNlTmFtZSB9LFxuICAgIHN0YXRlTWFjaGluZVByb3BzOiB0ZXN0U3RhdGVNYWNoaW5lUHJvcHMoc3RhY2spLFxuICAgIHN0YXRlTWFjaGluZUVudmlyb25tZW50VmFyaWFibGVOYW1lOiBjdXN0b21FbnZOYW1lXG4gIH0pO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpFQ1M6OlRhc2tEZWZpbml0aW9uXCIsIHtcbiAgICBGYW1pbHk6IGZhbWlseU5hbWUsXG4gICAgQ29udGFpbmVyRGVmaW5pdGlvbnM6IFtcbiAgICAgIHtcbiAgICAgICAgRW52aXJvbm1lbnQ6IFtcbiAgICAgICAgICB7XG4gICAgICAgICAgICBOYW1lOiBjdXN0b21FbnZOYW1lLFxuICAgICAgICAgICAgVmFsdWU6IHtcbiAgICAgICAgICAgICAgUmVmOiBcInRlc3Rjb25zdHJ1Y3RTdGF0ZU1hY2hpbmUzMzMzQUFBOVwiXG4gICAgICAgICAgICB9XG4gICAgICAgICAgfVxuICAgICAgICBdLFxuICAgICAgICBFc3NlbnRpYWw6IHRydWUsXG4gICAgICAgIEltYWdlOiB7XG4gICAgICAgICAgXCJGbjo6Sm9pblwiOiBbXG4gICAgICAgICAgICBcIlwiLFxuICAgICAgICAgICAgW1xuICAgICAgICAgICAgICBcIjEyMzQ1Njc4OTAxMi5ka3IuZWNyLnVzLWVhc3QtMS5cIixcbiAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgIFJlZjogXCJBV1M6OlVSTFN1ZmZpeFwiXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIFwiL2Zha2UtcmVwbzpsYXRlc3RcIlxuICAgICAgICAgICAgXVxuICAgICAgICAgIF1cbiAgICAgICAgfSxcbiAgICAgICAgTWVtb3J5UmVzZXJ2YXRpb246IDUxMixcbiAgICAgICAgTmFtZTogY29udGFpbmVyTmFtZSxcbiAgICAgICAgUG9ydE1hcHBpbmdzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgQ29udGFpbmVyUG9ydDogODA4MCxcbiAgICAgICAgICAgIFByb3RvY29sOiBcInRjcFwiXG4gICAgICAgICAgfVxuICAgICAgICBdXG4gICAgICB9XG4gICAgXVxuICB9KTtcbn0pO1xuXG50ZXN0KCdDaGVjayBmb3Igbm8gY2xvdWR3YXRjaCBjcmVhdGlvbicsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKCk7XG4gIGNvbnN0IHB1YmxpY0FwaSA9IHRydWU7XG5cbiAgY29uc3QgY29uc3RydWN0ID0gbmV3IEZhcmdhdGVUb1N0ZXBmdW5jdGlvbnMoc3RhY2ssICd0ZXN0LWNvbnN0cnVjdCcsIHtcbiAgICBwdWJsaWNBcGksXG4gICAgZWNyUmVwb3NpdG9yeUFybjogZGVmYXVsdHMuZmFrZUVjclJlcG9Bcm4sXG4gICAgdnBjUHJvcHM6IHsgaXBBZGRyZXNzZXM6IGVjMi5JcEFkZHJlc3Nlcy5jaWRyKHRlc3RDaWRyKSB9LFxuICAgIGNsdXN0ZXJQcm9wczogeyBjbHVzdGVyTmFtZSB9LFxuICAgIGNvbnRhaW5lckRlZmluaXRpb25Qcm9wczogeyBjb250YWluZXJOYW1lIH0sXG4gICAgZmFyZ2F0ZVRhc2tEZWZpbml0aW9uUHJvcHM6IHsgZmFtaWx5OiBmYW1pbHlOYW1lIH0sXG4gICAgZmFyZ2F0ZVNlcnZpY2VQcm9wczogeyBzZXJ2aWNlTmFtZSB9LFxuICAgIHN0YXRlTWFjaGluZVByb3BzOiB0ZXN0U3RhdGVNYWNoaW5lUHJvcHMoc3RhY2spLFxuICAgIGNyZWF0ZUNsb3VkV2F0Y2hBbGFybXM6IGZhbHNlXG4gIH0pO1xuXG4gIGV4cGVjdChjb25zdHJ1Y3QuY2xvdWR3YXRjaEFsYXJtcykubm90LnRvQmVEZWZpbmVkKCk7XG5cbiAgZGVmYXVsdHMuZXhwZWN0Tm9uZXhpc3RlbmNlKHN0YWNrLCBcIkFXUzo6Q2xvdWRXYXRjaDo6QWxhcm1cIiwge1xuICAgIENvbXBhcmlzb25PcGVyYXRvcjogXCJHcmVhdGVyVGhhbk9yRXF1YWxUb1RocmVzaG9sZFwiLFxuICAgIEV2YWx1YXRpb25QZXJpb2RzOiAxLFxuICAgIEFsYXJtRGVzY3JpcHRpb246IFwiQWxhcm0gZm9yIHRoZSBudW1iZXIgb2YgZXhlY3V0aW9ucyB0aGF0IGFib3J0ZWQgZXhjZWVkZWQgdGhlIHRocmVzaG9sZCBvZiAxLiBcIixcbiAgICBEaW1lbnNpb25zOiBbXG4gICAgICB7XG4gICAgICAgIE5hbWU6IFwiU3RhdGVNYWNoaW5lQXJuXCIsXG4gICAgICAgIFZhbHVlOiB7XG4gICAgICAgICAgUmVmOiBcInRlc3Rjb25zdHJ1Y3RTdGF0ZU1hY2hpbmUzMzMzQUFBOVwiXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICBdLFxuICAgIE1ldHJpY05hbWU6IFwiRXhlY3V0aW9uc0Fib3J0ZWRcIixcbiAgICBOYW1lc3BhY2U6IFwiQVdTL1N0YXRlc1wiLFxuICAgIFBlcmlvZDogMzAwLFxuICAgIFN0YXRpc3RpYzogXCJNYXhpbXVtXCIsXG4gICAgVGhyZXNob2xkOiAxXG4gIH0pO1xufSk7XG5cbnRlc3QoJ0NoZWNrIGZvciBjdXN0b20gbG9nIGdyb3VwIHByb3BzJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2soKTtcbiAgY29uc3QgcHVibGljQXBpID0gdHJ1ZTtcbiAgY29uc3QgbG9nR3JvdXBOYW1lID0gXCJjdXN0b20tbG9nLWdyb3VwXCI7XG5cbiAgbmV3IEZhcmdhdGVUb1N0ZXBmdW5jdGlvbnMoc3RhY2ssICd0ZXN0LWNvbnN0cnVjdCcsIHtcbiAgICBwdWJsaWNBcGksXG4gICAgZWNyUmVwb3NpdG9yeUFybjogZGVmYXVsdHMuZmFrZUVjclJlcG9Bcm4sXG4gICAgdnBjUHJvcHM6IHsgaXBBZGRyZXNzZXM6IGVjMi5JcEFkZHJlc3Nlcy5jaWRyKHRlc3RDaWRyKSB9LFxuICAgIGNsdXN0ZXJQcm9wczogeyBjbHVzdGVyTmFtZSB9LFxuICAgIGNvbnRhaW5lckRlZmluaXRpb25Qcm9wczogeyBjb250YWluZXJOYW1lIH0sXG4gICAgZmFyZ2F0ZVRhc2tEZWZpbml0aW9uUHJvcHM6IHsgZmFtaWx5OiBmYW1pbHlOYW1lIH0sXG4gICAgZmFyZ2F0ZVNlcnZpY2VQcm9wczogeyBzZXJ2aWNlTmFtZSB9LFxuICAgIHN0YXRlTWFjaGluZVByb3BzOiB0ZXN0U3RhdGVNYWNoaW5lUHJvcHMoc3RhY2spLFxuICAgIGxvZ0dyb3VwUHJvcHM6IHtcbiAgICAgIGxvZ0dyb3VwTmFtZVxuICAgIH1cbiAgfSk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OkxvZ3M6OkxvZ0dyb3VwXCIsIHtcbiAgICBMb2dHcm91cE5hbWU6IGxvZ0dyb3VwTmFtZVxuICB9KTtcbn0pO1xuXG5mdW5jdGlvbiBjcmVhdGVGYXJnYXRlQ29uc3RydWN0V2l0aE5ld1Jlc291cmNlcyhzdGFjazogY2RrLlN0YWNrLCBwdWJsaWNBcGk6IGJvb2xlYW4pIHtcbiAgcmV0dXJuIG5ldyBGYXJnYXRlVG9TdGVwZnVuY3Rpb25zKHN0YWNrLCAndGVzdC1jb25zdHJ1Y3QnLCB7XG4gICAgcHVibGljQXBpLFxuICAgIGVjclJlcG9zaXRvcnlBcm46IGRlZmF1bHRzLmZha2VFY3JSZXBvQXJuLFxuICAgIHZwY1Byb3BzOiB7IGlwQWRkcmVzc2VzOiBlYzIuSXBBZGRyZXNzZXMuY2lkcih0ZXN0Q2lkcikgfSxcbiAgICBjbHVzdGVyUHJvcHM6IHsgY2x1c3Rlck5hbWUgfSxcbiAgICBjb250YWluZXJEZWZpbml0aW9uUHJvcHM6IHsgY29udGFpbmVyTmFtZSB9LFxuICAgIGZhcmdhdGVUYXNrRGVmaW5pdGlvblByb3BzOiB7IGZhbWlseTogZmFtaWx5TmFtZSB9LFxuICAgIGZhcmdhdGVTZXJ2aWNlUHJvcHM6IHsgc2VydmljZU5hbWUgfSxcbiAgICBzdGF0ZU1hY2hpbmVQcm9wczogdGVzdFN0YXRlTWFjaGluZVByb3BzKHN0YWNrKSxcbiAgfSk7XG59XG5cbmZ1bmN0aW9uIHRlc3RTdGF0ZU1hY2hpbmVQcm9wcyhzdGFjazogY2RrLlN0YWNrLCB1c2VyUHJvcHM/OiBzdGVwZnVuY3Rpb25zLlN0YXRlTWFjaGluZVByb3BzKTpcbiAgc3RlcGZ1bmN0aW9ucy5TdGF0ZU1hY2hpbmVQcm9wcyB7XG4gIGNvbnN0IGRlZmF1bHRUZXN0UHJvcCA9IHsgZGVmaW5pdGlvbjogbmV3IHN0ZXBmdW5jdGlvbnMuUGFzcyhzdGFjaywgJ1N0YXJ0U3RhdGUnKSB9O1xuXG4gIHJldHVybiBkZWZhdWx0cy5jb25zb2xpZGF0ZVByb3BzKGRlZmF1bHRUZXN0UHJvcCwgdXNlclByb3BzKTtcbn1cbiJdfQ==