"""
Base integration driver for Unfolded Circle Remote integrations.

Provides common event handlers and device lifecycle management.

:copyright: (c) 2025 by Jack Powell.
:license: Mozilla Public License Version 2.0, see LICENSE for more details.
"""

import asyncio
import logging
from abc import ABC
from typing import Any, Generic, TypeVar

import ucapi
import ucapi.api as uc
from ucapi import (
    Entity,
    EntityTypes,
    button,
    climate,
    cover,
    light,
    media_player,
    remote,
    sensor,
    switch,
)
from .device import BaseDeviceInterface, DeviceEvents

# Type variables for generic device and entity types
DeviceT = TypeVar("DeviceT", bound=BaseDeviceInterface)  # Device interface type
ConfigT = TypeVar("ConfigT")  # Device configuration type (any object with attributes)

_LOG = logging.getLogger(__name__)

# Common attribute names for device configuration extraction
_DEVICE_ID_ATTRIBUTES = ("identifier", "id", "device_id")
_DEVICE_NAME_ATTRIBUTES = ("name", "friendly_name", "device_name")
_DEVICE_ADDRESS_ATTRIBUTES = (
    "address",
    "host_address",
    "ip_address",
    "device_address",
    "host",
)


def _get_first_valid_attr(obj: Any, *attrs: str) -> str | None:
    """
    Get the first valid attribute value from an object.

    Helper function to extract configuration values by trying multiple
    common attribute names in order.

    :param obj: Object to inspect
    :param attrs: Attribute names to try in order
    :return: String value of first found attribute, or None
    """
    for attr in attrs:
        if hasattr(obj, attr):
            value = getattr(obj, attr)
            if value:
                return str(value)
    return None


def create_entity_id(
    device_id: str, entity_type: EntityTypes | str, entity_id: str | None = None
) -> str:
    """
    Create a unique entity identifier for the given device and entity type.

    Entity IDs follow the format:
    - Simple: "{entity_type}.{device_id}"
    - With entity: "{entity_type}.{device_id}.{entity_id}"

    Use the optional entity_id parameter for devices that expose multiple entities
    of the same type, such as a hub with multiple lights or zones.

    Examples:
        >>> create_entity_id("device_123", EntityTypes.MEDIA_PLAYER)
        'media_player.device_123'
        >>> create_entity_id("hub_1", EntityTypes.LIGHT, "light_bedroom")
        'light.hub_1.light_bedroom'
        >>> create_entity_id("receiver_abc", "media_player", "zone_2")
        'media_player.receiver_abc.zone_2'

    :param device_id: The device identifier (hub or parent device)
    :param entity_type: The entity type (EntityTypes enum or string)
    :param entity_id: Optional sub-entity identifier (e.g., light ID, zone ID)
    :return: Entity identifier in the format "entity_type.device_id" or "entity_type.device_id.entity_id"
    """
    type_str = (
        entity_type.value if isinstance(entity_type, EntityTypes) else entity_type
    )

    if entity_id:
        return f"{type_str}.{device_id}.{entity_id}"
    return f"{type_str}.{device_id}"


class BaseIntegrationDriver(ABC, Generic[DeviceT, ConfigT]):
    """
    Base class for Remote Two integration drivers.

    Handles common patterns like:
    - Event listeners (connect, disconnect, standby, subscribe/unsubscribe)
    - Device lifecycle management
    - Entity registration and updates
    - State propagation from devices to entities

    Type Parameters:
        DeviceT: The device interface class (e.g., YamahaAVR)
        ConfigT: The device configuration class (e.g., YamahaDevice)
    """

    def __init__(
        self,
        loop: asyncio.AbstractEventLoop,
        device_class: type[DeviceT],
        entity_classes: list[EntityTypes] | EntityTypes,
    ):
        """
        Initialize the integration driver.

        :param loop: The asyncio event loop
        :param device_class: The device interface class to instantiate
        :param entity_classes: EntityTypes or list of EntityTypes (e.g., EntityTypes.MEDIA_PLAYER)
                               Single EntityTypes value will be converted to a list
        """
        self.api = uc.IntegrationAPI(loop)
        self._loop = loop
        self._device_class = device_class

        # Allow passing a single EntityTypes or a list
        if isinstance(entity_classes, EntityTypes):
            self._entity_classes = [entity_classes]
        else:
            self._entity_classes = entity_classes

        self._configured_devices: dict[str, DeviceT] = {}
        self.config = None  # Will be set by integration after initialization
        self._setup_event_handlers()

    def _setup_event_handlers(self) -> None:
        """Register all event handlers with the API."""
        self.api.listens_to(ucapi.Events.CONNECT)(self.on_r2_connect_cmd)
        self.api.listens_to(ucapi.Events.DISCONNECT)(self.on_r2_disconnect_cmd)
        self.api.listens_to(ucapi.Events.ENTER_STANDBY)(self.on_r2_enter_standby)
        self.api.listens_to(ucapi.Events.EXIT_STANDBY)(self.on_r2_exit_standby)
        self.api.listens_to(ucapi.Events.SUBSCRIBE_ENTITIES)(self.on_subscribe_entities)
        self.api.listens_to(ucapi.Events.UNSUBSCRIBE_ENTITIES)(
            self.on_unsubscribe_entities
        )

    # ========================================================================
    # Remote Two Event Handlers (can be overridden)
    # ========================================================================

    async def on_r2_connect_cmd(self) -> None:
        """
        Handle Remote Two connect command.

        Default implementation: connects all configured devices and sets integration state.
        Override to add custom logic before/after device connections.

        Example:
            async def on_r2_connect_cmd(self) -> None:
                await super().on_r2_connect_cmd()
                # Custom logic after connect
        """
        _LOG.debug("Client connect command: connecting device(s)")
        await self.api.set_device_state(ucapi.DeviceStates.CONNECTED)
        for device in self._configured_devices.values():
            # start background task
            self._loop.create_task(device.connect())

    async def on_r2_disconnect_cmd(self) -> None:
        """
        Handle Remote Two disconnect command.

        Default implementation: disconnects all configured devices.
        Override to add custom disconnect logic.
        """
        _LOG.debug("Client disconnect command: disconnecting device(s)")
        for device in self._configured_devices.values():
            # start background task
            self._loop.create_task(device.disconnect())

    async def on_r2_enter_standby(self) -> None:
        """
        Handle Remote Two entering standby mode.

        Default implementation: disconnects all devices to save resources.
        Override to customize standby behavior.
        """
        _LOG.debug("Enter standby event: disconnecting device(s)")
        for device in self._configured_devices.values():
            await device.disconnect()

    async def on_r2_exit_standby(self) -> None:
        """
        Handle Remote Two exiting standby mode.

        Default implementation: reconnects all configured devices.
        Override to customize wake behavior.
        """
        _LOG.debug("Exit standby event: connecting device(s)")
        for device in self._configured_devices.values():
            # start background task
            self._loop.create_task(device.connect())

    async def on_subscribe_entities(self, entity_ids: list[str]) -> None:
        """
        Handle entity subscription events.

        Default implementation: adds devices for subscribed entities and updates their state.
        Override to customize subscription behavior.

        :param entity_ids: List of entity identifiers being subscribed
        """
        _LOG.debug("Subscribe entities event: %s", entity_ids)

        for entity_id in entity_ids:
            device_id = self.device_from_entity_id(entity_id)
            if device_id is None:
                continue

            # Check if device is already configured
            if device_id in self._configured_devices:
                device = self._configured_devices[device_id]
                _LOG.info("Entity '%s' subscribing to existing device", entity_id)
                _LOG.debug("Device State: %s", device.state)

                # Update entity state
                if device.state is None:
                    state = media_player.States.UNAVAILABLE
                else:
                    state = self.map_device_state(device.state)

                self.api.configured_entities.update_attributes(
                    entity_id, {media_player.Attributes.STATE: state}
                )
                continue

            # Device not configured yet, add it
            device_config = self.get_device_config(device_id)
            if device_config:
                # Add without connecting - connection will be handled by CONNECT event
                self.add_configured_device(device_config, connect=False)
            else:
                _LOG.error(
                    "Failed to subscribe entity %s: no device config found", entity_id
                )

    async def on_unsubscribe_entities(self, entity_ids: list[str]) -> None:
        """
        Handle entity unsubscription events.

        Default implementation: disconnects and cleans up devices when all their
        entities are unsubscribed. Override to customize cleanup behavior.

        :param entity_ids: List of entity identifiers being unsubscribed
        """
        _LOG.debug("Unsubscribe entities event: %s", entity_ids)

        # Track which devices need to be checked
        devices_to_check = set()

        for entity_id in entity_ids:
            device_id = self.device_from_entity_id(entity_id)
            if device_id is not None and device_id in self._configured_devices:
                devices_to_check.add(device_id)

        # For each device, check if any of its entities are still configured
        for device_id in devices_to_check:
            device_entities = self.get_entity_ids_for_device(device_id)
            any_entity_configured = any(
                self.api.configured_entities.get(entity_id) is not None
                for entity_id in device_entities
            )

            if not any_entity_configured:
                # No entities are configured anymore, disconnect and cleanup
                device = self._configured_devices.get(device_id)
                if device:
                    _LOG.info(
                        "No entities configured for device '%s', disconnecting and cleaning up",
                        device_id,
                    )
                    await device.disconnect()
                    device.events.remove_all_listeners()
                    self._configured_devices.pop(device_id, None)

    # ========================================================================
    # Device Lifecycle Management
    # ========================================================================

    def add_configured_device(
        self, device_config: ConfigT, connect: bool = True
    ) -> None:
        """
        Add and configure a device.

        :param device_config: Device configuration
        :param connect: Whether to initiate connection immediately
        """
        device_id = self.get_device_id(device_config)

        if device_id in self._configured_devices:
            _LOG.debug(
                "Device %s already configured, updating existing instance", device_id
            )
            device = self._configured_devices[device_id]
        else:
            _LOG.info(
                "Adding new device: %s (%s)",
                device_id,
                self.get_device_name(device_config),
            )
            device = self._device_class(
                device_config, loop=self._loop, config_manager=self.config
            )
            self.setup_device_event_handlers(device)
            self._configured_devices[device_id] = device

        if connect:
            # start background connection task
            self._loop.create_task(device.connect())

        self.register_available_entities(device_config, device)

    def setup_device_event_handlers(self, device: DeviceT) -> None:
        """
        Attach event handlers to device.

        Override this method to add custom event handlers. Call super() first
        to register the default handlers, then add your custom ones.

        :param device: Device instance
        """
        device.events.on(DeviceEvents.CONNECTED, self.on_device_connected)
        device.events.on(DeviceEvents.DISCONNECTED, self.on_device_disconnected)
        device.events.on(DeviceEvents.ERROR, self.on_device_connection_error)
        device.events.on(DeviceEvents.UPDATE, self.on_device_update)

    def register_available_entities(
        self, device_config: ConfigT, device: DeviceT
    ) -> None:
        """
        Register available entities for a device.

        Override this method to customize entity registration logic. Call super()
        to use the default implementation that calls create_entities().

        :param device_config: Device configuration
        :param device: Device instance
        """
        device_id = self.get_device_id(device_config)
        _LOG.info("Registering available entities for %s", device_id)

        entities = self.create_entities(device_config, device)

        for entity in entities:
            if self.api.available_entities.contains(entity.id):
                self.api.available_entities.remove(entity.id)
            self.api.available_entities.add(entity)

    # ========================================================================
    # Device Event Handlers (can be overridden)
    # ========================================================================

    async def on_device_connected(self, device_id: str) -> None:
        """
        Handle device connection.

        :param device_id: Device identifier
        """
        _LOG.debug("Device connected: %s", device_id)

        if device_id not in self._configured_devices:
            _LOG.warning("Device %s is not configured", device_id)
            return

        await self.api.set_device_state(ucapi.DeviceStates.CONNECTED)

        device = self._configured_devices[device_id]
        state = (
            self.map_device_state(device.state)
            if device.state
            else media_player.States.UNKNOWN
        )

        for entity_id in self.get_entity_ids_for_device(device_id):
            configured_entity = self.api.configured_entities.get(entity_id)
            if configured_entity is None:
                _LOG.debug("Entity %s is not configured, ignoring", entity_id)
                continue

            self.api.configured_entities.update_attributes(
                entity_id, {media_player.Attributes.STATE: state}
            )  # Use media_player state as a stand-in

    async def on_device_disconnected(self, device_id: str) -> None:
        """
        Handle device disconnection.

        :param device_id: Device identifier
        """
        _LOG.debug("Device disconnected: %s", device_id)

        for entity_id in self.get_entity_ids_for_device(device_id):
            configured_entity = self.api.configured_entities.get(entity_id)
            if configured_entity is None:
                continue

            self.api.configured_entities.update_attributes(
                entity_id,
                {media_player.Attributes.STATE: media_player.States.UNAVAILABLE},
            )  # Use media_player state as a stand-in

    async def on_device_connection_error(self, device_id: str, message: str) -> None:
        """
        Handle device connection error.

        :param device_id: Device identifier
        :param message: Error message
        """
        _LOG.error("[%s] Connection error: %s", device_id, message)

        for entity_id in self.get_entity_ids_for_device(device_id):
            configured_entity = self.api.configured_entities.get(entity_id)
            if configured_entity is None:
                continue

            self.api.configured_entities.update_attributes(
                entity_id,
                {media_player.Attributes.STATE: media_player.States.UNAVAILABLE},
            )  # Use media_player state as a stand-in

    async def on_device_update(
        self,
        device_id: str,
        update: dict[str, Any] | None,
        clear_media_when_off: bool = True,
    ) -> None:
        """
        Handle device state updates.

        Default implementation extracts entity-type-specific attributes from the
        update dict and updates configured/available entities accordingly.
        Override this method to customize update handling or add state mapping.

        :param device_id: Device identifier
        :param update: Dictionary containing updated properties
        :param clear_media_when_off: If True, clears all media player attributes when state is OFF
        """
        if update is None:
            _LOG.warning("[%s] Received None update, skipping", device_id)
            return

        _LOG.debug("[%s] Device update: %s", device_id, update)

        # Process update for each entity belonging to this device
        for entity_id in self.get_entity_ids_for_device(device_id):
            configured_entity = self.api.configured_entities.get(entity_id)
            if configured_entity is None:
                # Try available entities if not in configured
                configured_entity = self.api.available_entities.get(entity_id)
                if configured_entity is None:
                    continue

            # Extract relevant attributes based on entity type
            attributes: dict[str, Any] = {}

            match configured_entity.entity_type:
                case EntityTypes.BUTTON:
                    # Button entities: STATE
                    if button.Attributes.STATE.value in update:
                        state = self.map_device_state(
                            update[button.Attributes.STATE.value]
                        )
                        attributes[button.Attributes.STATE] = state

                case EntityTypes.CLIMATE:
                    # Climate entities: STATE, CURRENT_TEMPERATURE, TARGET_TEMPERATURE,
                    # TARGET_TEMPERATURE_HIGH, TARGET_TEMPERATURE_LOW, FAN_MODE
                    for attr in [
                        climate.Attributes.STATE,
                        climate.Attributes.CURRENT_TEMPERATURE,
                        climate.Attributes.TARGET_TEMPERATURE,
                        climate.Attributes.TARGET_TEMPERATURE_HIGH,
                        climate.Attributes.TARGET_TEMPERATURE_LOW,
                        climate.Attributes.FAN_MODE,
                    ]:
                        if attr.value in update:
                            value = update[attr.value]
                            # Apply state mapping for STATE attribute
                            if attr == climate.Attributes.STATE:
                                value = self.map_device_state(value)
                            attributes[attr] = value

                case EntityTypes.COVER:
                    # Cover entities: STATE, POSITION, TILT_POSITION
                    for attr in [
                        cover.Attributes.STATE,
                        cover.Attributes.POSITION,
                        cover.Attributes.TILT_POSITION,
                    ]:
                        if attr.value in update:
                            value = update[attr.value]
                            # Apply state mapping for STATE attribute
                            if attr == cover.Attributes.STATE:
                                value = self.map_device_state(value)
                            attributes[attr] = value

                case EntityTypes.LIGHT:
                    # Light entities: STATE, HUE, SATURATION, BRIGHTNESS, COLOR_TEMPERATURE
                    for attr in [
                        light.Attributes.STATE,
                        light.Attributes.HUE,
                        light.Attributes.SATURATION,
                        light.Attributes.BRIGHTNESS,
                        light.Attributes.COLOR_TEMPERATURE,
                    ]:
                        if attr.value in update:
                            value = update[attr.value]
                            # Apply state mapping for STATE attribute
                            if attr == light.Attributes.STATE:
                                value = self.map_device_state(value)
                            attributes[attr] = value

                case EntityTypes.MEDIA_PLAYER:
                    # Media player entities: STATE, VOLUME, MUTED, MEDIA_DURATION,
                    # MEDIA_POSITION, MEDIA_TYPE, MEDIA_IMAGE_URL, MEDIA_TITLE,
                    # MEDIA_ARTIST, MEDIA_ALBUM, REPEAT, SHUFFLE, SOURCE, SOURCE_LIST,
                    # SOUND_MODE, SOUND_MODE_LIST

                    # Check if state is being updated and is OFF
                    state_value = None
                    if media_player.Attributes.STATE.value in update:
                        state_value = self.map_device_state(
                            update[media_player.Attributes.STATE.value]
                        )
                        attributes[media_player.Attributes.STATE] = state_value

                    # If clear_media_when_off is True and state is OFF, clear all media attributes
                    if clear_media_when_off and state_value == media_player.States.OFF:
                        # Clear all media-related attributes (use empty strings for string fields, 0 for numbers)
                        attributes[media_player.Attributes.MEDIA_DURATION] = 0
                        attributes[media_player.Attributes.MEDIA_POSITION] = 0
                        attributes[media_player.Attributes.MEDIA_TYPE] = ""
                        attributes[media_player.Attributes.MEDIA_IMAGE_URL] = ""
                        attributes[media_player.Attributes.MEDIA_TITLE] = ""
                        attributes[media_player.Attributes.MEDIA_ARTIST] = ""
                        attributes[media_player.Attributes.MEDIA_ALBUM] = ""
                        attributes[media_player.Attributes.SOURCE] = ""
                        attributes[media_player.Attributes.SOUND_MODE] = ""
                    else:
                        # Process remaining attributes normally
                        for attr in [
                            media_player.Attributes.VOLUME,
                            media_player.Attributes.MUTED,
                            media_player.Attributes.MEDIA_DURATION,
                            media_player.Attributes.MEDIA_POSITION,
                            media_player.Attributes.MEDIA_POSITION_UPDATED_AT,
                            media_player.Attributes.MEDIA_TYPE,
                            media_player.Attributes.MEDIA_IMAGE_URL,
                            media_player.Attributes.MEDIA_TITLE,
                            media_player.Attributes.MEDIA_ARTIST,
                            media_player.Attributes.MEDIA_ALBUM,
                            media_player.Attributes.REPEAT,
                            media_player.Attributes.SHUFFLE,
                            media_player.Attributes.SOURCE,
                            media_player.Attributes.SOURCE_LIST,
                            media_player.Attributes.SOUND_MODE,
                            media_player.Attributes.SOUND_MODE_LIST,
                        ]:
                            if attr.value in update:
                                attributes[attr] = update[attr.value]

                case EntityTypes.REMOTE:
                    # Remote entities: STATE
                    if remote.Attributes.STATE.value in update:
                        state = self.map_device_state(
                            update[remote.Attributes.STATE.value]
                        )
                        attributes[remote.Attributes.STATE] = state

                case EntityTypes.SENSOR:
                    # Sensor entities: STATE, VALUE, UNIT
                    for attr in [
                        sensor.Attributes.STATE,
                        sensor.Attributes.VALUE,
                        sensor.Attributes.UNIT,
                    ]:
                        if attr.value in update:
                            value = update[attr.value]
                            # Apply state mapping for STATE attribute
                            if attr == sensor.Attributes.STATE:
                                value = self.map_device_state(value)
                            attributes[attr] = value

                case EntityTypes.SWITCH:
                    # Switch entities: STATE
                    if switch.Attributes.STATE.value in update:
                        state = self.map_device_state(
                            update[switch.Attributes.STATE.value]
                        )
                        attributes[switch.Attributes.STATE] = state

                case _:
                    # Unknown entity type - log warning
                    _LOG.warning(
                        "[%s] Unknown entity type: %s for entity %s",
                        device_id,
                        configured_entity.entity_type,
                        entity_id,
                    )
                    continue

            # Update entity attributes if any were found
            if attributes:
                if self.api.configured_entities.contains(entity_id):
                    self.api.configured_entities.update_attributes(
                        entity_id, attributes
                    )
                elif self.api.available_entities.contains(entity_id):
                    self.api.available_entities.update_attributes(entity_id, attributes)
                _LOG.debug(
                    "[%s] Updated entity %s with attributes: %s",
                    device_id,
                    entity_id,
                    attributes,
                )

    def get_device_config(self, device_id: str) -> ConfigT | None:
        """
        Get device configuration for the given device ID.

        Default implementation: checks _configured_devices first, then falls
        back to self.config.get() if config manager is available.
        Override this if your integration uses a different config structure.

        :param device_id: Device identifier
        :return: Device configuration or None
        """
        # First check if device is already configured
        device = self._configured_devices.get(device_id)
        if device:
            return device.device_config

        # Fall back to stored configuration if available
        if self.config and hasattr(self.config, "get"):
            return self.config.get(device_id)

        return None

    def get_device_id(self, device_config: ConfigT) -> str:
        """
        Extract device ID from device configuration.

        Default implementation: tries common attribute names (identifier, id, device_id).
        Override this if your config uses a different attribute name.

        :param device_config: Device configuration
        :return: Device identifier
        :raises AttributeError: If no valid ID attribute is found
        """
        value = _get_first_valid_attr(device_config, *_DEVICE_ID_ATTRIBUTES)
        if value:
            return value

        raise AttributeError(
            f"Device config {type(device_config).__name__} has no 'identifier', 'id', or 'device_id' attribute. "
            f"Override get_device_id() to specify which attribute to use."
        )

    def get_device_name(self, device_config: ConfigT) -> str:
        """
        Extract device name from device configuration.

        Default implementation: tries common attribute names (name, friendly_name, device_name).
        Override this if your config uses a different attribute name.

        :param device_config: Device configuration
        :return: Device name
        :raises AttributeError: If no valid name attribute is found
        """
        value = _get_first_valid_attr(device_config, *_DEVICE_NAME_ATTRIBUTES)
        if value:
            return value

        raise AttributeError(
            f"Device config {type(device_config).__name__} has no 'name', 'friendly_name', or 'device_name' attribute. "
            f"Override get_device_name() to specify which attribute to use."
        )

    def get_device_address(self, device_config: ConfigT) -> str:
        """
        Extract device address from device configuration.

        Default implementation: tries common attribute names (address, host_address, ip_address, device_address, host).
        Override this if your config uses a different attribute name.

        :param device_config: Device configuration
        :return: Device address
        :raises AttributeError: If no valid address attribute is found
        """
        value = _get_first_valid_attr(device_config, *_DEVICE_ADDRESS_ATTRIBUTES)
        if value:
            return value

        raise AttributeError(
            f"Device config {type(device_config).__name__} has no 'address', 'host_address', 'ip_address', 'device_address', or 'host' attribute. "
            f"Override get_device_address() to specify which attribute to use."
        )

    def create_entities(self, device_config: ConfigT, device: DeviceT) -> list[Entity]:
        """
        Create entity instances for a device.

        DEFAULT IMPLEMENTATION: Creates entities from the entity_classes passed to __init__.
        Each entity class is instantiated with (device_config, device) as parameters.

        The default implementation returns:
            [EntityClass1(device_config, device), EntityClass2(device_config, device), ...]

        Override this method if you need:
        - Conditional entity creation based on device capabilities
        - Custom parameters beyond device_config and device
        - Dynamic entity creation logic

        Example override:
            def create_entities(self, device_config, device):
                entities = []
                if device.supports_playback:
                    entities.append(YamahaMediaPlayer(device_config, device))
                if device.supports_remote:
                    entities.append(YamahaRemote(device_config, device))
                return entities

        :param device_config: Device configuration
        :param device: Device instance
        :return: List of entity instances (MediaPlayer, Remote, etc.)
        """
        return [
            entity_class(device_config, device) for entity_class in self._entity_classes
        ]

    def map_device_state(self, device_state: Any) -> media_player.States:
        """
        Map device-specific state to ucapi media player state.

        DEFAULT IMPLEMENTATION: Converts device_state to uppercase string and maps
        common state values to media_player.States:

        - UNAVAILABLE → UNAVAILABLE
        - UNKNOWN → UNKNOWN
        - ON, MENU, IDLE, ACTIVE, READY → ON
        - OFF, POWER_OFF, POWERED_OFF → OFF
        - PLAYING, PLAY → PLAYING
        - PAUSED, PAUSE → PAUSED
        - STANDBY, SLEEP → STANDBY
        - BUFFERING, LOADING → BUFFERING
        - Everything else → UNKNOWN

        Override this method if you need:
        - Different state mappings
        - Device-specific state enum handling
        - Complex state logic

        Example override:
            def map_device_state(self, device_state):
                if isinstance(device_state, MyDeviceState):
                    match device_state:
                        case MyDeviceState.POWERED_ON:
                            return media_player.States.ON
                        case MyDeviceState.POWERED_OFF:
                            return media_player.States.OFF
                        case _:
                            return media_player.States.UNKNOWN
                return super().map_device_state(device_state)

        :param device_state: Device-specific state (string, enum, or any object with __str__)
        :return: Media player state
        """
        if device_state is None:
            return media_player.States.UNKNOWN

        # If already a media_player.States enum, return it directly
        if isinstance(device_state, media_player.States):
            return device_state

        # Convert to uppercase string for comparison
        state_str = str(device_state).upper()

        match state_str:
            case "UNAVAILABLE":
                return media_player.States.UNAVAILABLE
            case "UNKNOWN":
                return media_player.States.UNKNOWN
            case "ON" | "MENU" | "IDLE" | "ACTIVE" | "READY":
                return media_player.States.ON
            case "OFF" | "POWER_OFF" | "POWERED_OFF" | "STOPPED":
                return media_player.States.OFF
            case "PLAYING" | "PLAY" | "SEEKING":
                return media_player.States.PLAYING
            case "PAUSED" | "PAUSE":
                return media_player.States.PAUSED
            case "STANDBY" | "SLEEP":
                return media_player.States.STANDBY
            case "BUFFERING" | "LOADING":
                return media_player.States.BUFFERING
            case _:
                return media_player.States.UNKNOWN

    # ========================================================================
    # Entity ID Methods (should be overridden together if custom format used)
    # ========================================================================

    def device_from_entity_id(self, entity_id: str) -> str | None:
        """
        Extract device identifier from entity identifier.

        DEFAULT IMPLEMENTATION: Parses entity IDs created by create_entity_id().
        Handles both formats:
        - Simple: "entity_type.device_id" → returns "device_id"
        - With sub-entity: "entity_type.device_id.entity_id" → returns "device_id"

        **IMPORTANT**: If you override create_entities() to use a custom entity ID format,
        you MUST also override this method to match your custom format. The default
        implementation will detect this and raise an error to prevent bugs.

        Example custom override:
            def create_entities(self, device_config, device):
                # Custom format: entity_id IS the device_id
                return [PSNMediaPlayer(device_config.identifier, ...)]

            def device_from_entity_id(self, entity_id: str) -> str | None:
                # For PSN, entity_id IS the device_id
                return entity_id

        :param entity_id: Entity identifier (e.g., "media_player.device_123")
        :return: Device identifier or None
        :raises NotImplementedError: If create_entities was overridden but this method wasn't
        """
        # Check if create_entities was overridden (indicating custom entity ID format)
        create_entities_overridden = (
            type(self).create_entities is not BaseIntegrationDriver.create_entities
        )

        if create_entities_overridden:
            # User has custom entity creation, they must override this method too
            device_from_entity_overridden = (
                type(self).device_from_entity_id
                is not BaseIntegrationDriver.device_from_entity_id
            )

            if not device_from_entity_overridden:
                raise NotImplementedError(
                    f"{type(self).__name__}.create_entities() is overridden but "
                    f"device_from_entity_id() is not. When you override create_entities() "
                    f"with a custom entity ID format, you must also override "
                    f"device_from_entity_id() to parse your custom format. "
                )

        # Default implementation: parse standard format from create_entity_id()
        if not entity_id or "." not in entity_id:
            return None

        # Split on period: "entity_type.device_id" or "entity_type.device_id.entity_id"
        parts = entity_id.split(".")

        if len(parts) < 2:
            return None

        # Second part is always the device_id in create_entity_id() format
        return parts[1]

    def get_entity_ids_for_device(self, device_id: str) -> list[str]:
        """
        Get all entity identifiers for a device.

        DEFAULT IMPLEMENTATION: Queries all registered entities from the API and
        filters them by device_id using device_from_entity_id().

        This works automatically with the standard entity ID format from create_entity_id().
        For integrations using custom entity ID formats, this will work as long as
        device_from_entity_id() is properly overridden to parse your custom format.

        Override this method only if you need:
        - Performance optimization for integrations with many entities
        - Special filtering logic beyond device_id matching
        - Caching or pre-computed entity lists

        Example override for performance:
            def get_entity_ids_for_device(self, device_id: str) -> list[str]:
                # Cache entity IDs per device for faster lookups
                if device_id not in self._entity_cache:
                    self._entity_cache[device_id] = [
                        f"media_player.{device_id}",
                        f"remote.{device_id}",
                    ]
                return self._entity_cache[device_id]

        :param device_id: Device identifier
        :return: List of entity identifiers for this device
        """
        # Query all entities (both available and configured) and filter by device_id
        entity_ids = set()  # Use set to avoid duplicates

        # Check available entities
        for entity in self.api.available_entities.get_all():
            entity_device_id = self.device_from_entity_id(entity["entity_id"])
            if entity_device_id == device_id:
                entity_ids.add(entity["entity_id"])

        # Check configured entities
        for entity in self.api.configured_entities.get_all():
            entity_device_id = self.device_from_entity_id(entity["entity_id"])
            if entity_device_id == device_id:
                entity_ids.add(entity["entity_id"])

        return list(entity_ids)

    # ========================================================================
    # Utility Methods
    # ========================================================================

    def remove_device(self, device_id: str) -> None:
        """
        Remove a configured device.

        :param device_id: Device identifier
        """
        if device_id in self._configured_devices:
            _LOG.info("Removing device %s", device_id)
            device = self._configured_devices.pop(device_id)
            device.events.remove_all_listeners()

            # Remove all associated entities
            for entity_id in self.get_entity_ids_for_device(device_id):
                self.api.configured_entities.remove(entity_id)
                self.api.available_entities.remove(entity_id)
        else:
            _LOG.warning("Device %s not found in configured devices", device_id)

    def clear_devices(self) -> None:
        """Remove all configured devices."""
        _LOG.info("Clearing all configured devices")
        for device in self._configured_devices.values():
            device.events.remove_all_listeners()
        self._configured_devices.clear()
        self.api.configured_entities.clear()
        self.api.available_entities.clear()

    # ========================================================================
    # Configuration Change Callbacks (can be overridden)
    # ========================================================================

    def on_device_added(self, device_config: ConfigT) -> None:
        """
        Handle a newly added device in the configuration.

        Default implementation: Adds the device without connecting.
        Override if you need custom behavior.

        :param device_config: Device configuration that was added
        """
        _LOG.debug("Device added: %s", self.get_device_id(device_config))
        self.add_configured_device(device_config, connect=False)

    def on_device_removed(self, device_config: ConfigT | None) -> None:
        """
        Handle a removed device in the configuration.

        Default implementation: Removes the device or clears all if None.
        Override if you need custom behavior.

        :param device_config: Device configuration that was removed, or None to clear all
        """
        if device_config is None:
            _LOG.debug("Configuration cleared, removing all devices")
            self.clear_devices()
        else:
            device_id = self.get_device_id(device_config)
            _LOG.debug("Device removed: %s", device_id)
            self.remove_device(device_id)
