"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynchronousCrawler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const pre_bundled_layer_1 = require("../common/pre-bundled-layer");
/**
 * CrawlerStartWait Construct to start an AWS Glue Crawler execution and asynchronously wait for completion
 */
class SynchronousCrawler extends constructs_1.Construct {
    /**
    * Constructs a new instance of the DataGenerator class
    * @param {Construct} scope the Scope of the CDK Construct
    * @param {string} id the ID of the CDK Construct
    * @param {SynchronousCrawlerProps} props the CrawlerStartWait [properties]{@link SynchronousCrawlerProps}
    * @access public
    */
    constructor(scope, id, props) {
        super(scope, id);
        const stack = aws_cdk_lib_1.Stack.of(this);
        //Lambda policy to allow starting a crawler
        const lambdaCRPolicy = [new aws_iam_1.PolicyStatement({
                resources: [
                    stack.formatArn({
                        region: aws_cdk_lib_1.Aws.REGION,
                        account: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        service: 'glue',
                        resource: 'crawler',
                        resourceName: props.crawlerName,
                    }),
                ],
                actions: [
                    'glue:StartCrawler',
                    'glue:GetCrawler',
                    'glue:StopCrawler',
                ],
            })];
        // AWS Lambda function for the AWS CDK Custom Resource responsible to start crawler
        const crawlerStartFn = new pre_bundled_function_1.PreBundledFunction(this, 'StartFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'synchronous-crawler/resources/lambdas',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: lambdaCRPolicy,
            handler: 'lambda.on_event',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        // AWS Lambda function for the AWS CDK Custom Resource responsible to wait for crawler completion
        const crawlerWaitFn = new pre_bundled_function_1.PreBundledFunction(this, 'WaitFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'synchronous-crawler/resources/lambdas',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: lambdaCRPolicy,
            handler: 'lambda.is_complete',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        // Create an AWS CDK Custom Resource Provider for starting the source crawler and waiting for completion
        const crawlerStartWaitCRP = new custom_resources_1.Provider(this, 'synchronousCrawlerCRP', {
            //onEventFnName: 'SynchronousCrawlerStartFn',
            //isCompleteFnName: 'SynchronousCrawlerWaitFn',
            onEventHandler: crawlerStartFn,
            isCompleteHandler: crawlerWaitFn,
            queryInterval: aws_cdk_lib_1.Duration.seconds(60),
            totalTimeout: aws_cdk_lib_1.Duration.seconds(props.timeout || 600),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        // Create an AWS CDK Custom Resource for starting the source crawler and waiting for completion
        new aws_cdk_lib_1.CustomResource(this, 'synchronousCrawlerCR', {
            serviceToken: crawlerStartWaitCRP.serviceToken,
            properties: {
                CrawlerName: props.crawlerName,
            },
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
SynchronousCrawler[_a] = { fqn: "aws-analytics-reference-architecture.SynchronousCrawler", version: "2.12.6" };
exports.SynchronousCrawler = SynchronousCrawler;
//# sourceMappingURL=data:application/json;base64,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