"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SynchronousAthenaQuery = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const pre_bundled_layer_1 = require("../common/pre-bundled-layer");
/**
 * Execute an Amazon Athena query synchronously during CDK deployment
 */
class SynchronousAthenaQuery extends constructs_1.Construct {
    /**
    * Constructs a new instance of the SynchronousAthenaQuery class
    * @param {Construct} scope the Scope of the CDK Construct
    * @param {string} id the ID of the CDK Construct
    * @param {SynchronousAthenaQueryProps} props the CrawlerStartWait [properties]{@link SynchronousAthenaQueryProps}
    */
    constructor(scope, id, props) {
        super(scope, id);
        const stack = aws_cdk_lib_1.Stack.of(this);
        let athenaQueryStartFnPolicy = [];
        // Add permissions from the Amazon IAM Policy Statements
        props.executionRoleStatements?.forEach((element) => {
            athenaQueryStartFnPolicy.push(element);
        });
        // Add permissions to the Function for starting the query
        athenaQueryStartFnPolicy.push(new aws_iam_1.PolicyStatement({
            resources: [
                stack.formatArn({
                    region: aws_cdk_lib_1.Aws.REGION,
                    account: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    service: 'athena',
                    resource: 'workgroup',
                    resourceName: 'primary',
                }),
            ],
            actions: [
                'athena:StartQueryExecution',
            ],
        }));
        // add permissions to the Function to store result in the result path
        athenaQueryStartFnPolicy.push(new aws_iam_1.PolicyStatement({
            resources: [
                stack.formatArn({
                    region: '',
                    account: '',
                    service: 's3',
                    resource: props.resultPath.bucketName,
                    resourceName: props.resultPath.objectKey,
                }),
                stack.formatArn({
                    region: '',
                    account: '',
                    service: 's3',
                    resource: props.resultPath.bucketName,
                    resourceName: props.resultPath.objectKey + '/*',
                }),
                stack.formatArn({
                    region: '',
                    account: '',
                    service: 's3',
                    resource: props.resultPath.bucketName,
                }),
            ],
            actions: [
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:ListMultipartUploadParts',
                's3:AbortMultipartUpload',
                's3:PutObject',
                's3:CreateBucket',
            ],
        }));
        // AWS Lambda function for the AWS CDK Custom Resource responsible to start query
        const athenaQueryStartFn = new pre_bundled_function_1.PreBundledFunction(this, 'StartFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'synchronous-athena-query/resources/lambdas',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: athenaQueryStartFnPolicy,
            handler: 'lambda.on_event',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        let athenaQueryWaitFnPolicy = [];
        // Add permissions to the Function
        athenaQueryWaitFnPolicy.push(new aws_iam_1.PolicyStatement({
            resources: [
                stack.formatArn({
                    region: aws_cdk_lib_1.Aws.REGION,
                    account: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    service: 'athena',
                    resource: 'workgroup',
                    resourceName: 'primary',
                }),
            ],
            actions: [
                'athena:GetQueryExecution',
                'athena:GetQueryResults',
            ],
        }));
        // AWS Lambda function for the AWS CDK Custom Resource responsible to wait for query completion
        const athenaQueryWaitFn = new pre_bundled_function_1.PreBundledFunction(this, 'WaitFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'synchronous-athena-query/resources/lambdas',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: athenaQueryWaitFnPolicy,
            handler: 'lambda.is_complete',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        /*const providerManagedPolicy = new ManagedPolicy(this, 'providerManagedPolicy', {
          statements: [new PolicyStatement({
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
            resources: [`arn:aws:logs:${Aws.REGION}:${Aws.ACCOUNT_ID}:*`],
            effect: Effect.ALLOW,
          })],
        });
    
        const providerRole = new Role(this, 'providerRole', {
          assumedBy: new ServicePrincipal('lambda.amazonaws.com'),
          managedPolicies: [providerManagedPolicy],
        });*/
        // Create an AWS CDK Custom Resource Provider for starting the source crawler and waiting for completion
        const synchronousAthenaQueryCRP = new custom_resources_1.Provider(this, 'customresourceprovider', {
            onEventHandler: athenaQueryStartFn,
            isCompleteHandler: athenaQueryWaitFn,
            queryInterval: aws_cdk_lib_1.Duration.seconds(10),
            totalTimeout: aws_cdk_lib_1.Duration.minutes(props.timeout || 1),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        synchronousAthenaQueryCRP.node.addDependency(athenaQueryStartFn);
        synchronousAthenaQueryCRP.node.addDependency(athenaQueryWaitFn);
        const resultPathBucket = aws_s3_1.Bucket.fromBucketName(this, 'ResultPathBucket', props.resultPath.bucketName);
        // Create an AWS CDK Custom Resource for starting the source crawler and waiting for completion
        const myCR = new aws_cdk_lib_1.CustomResource(this, 'SynchronousAthenaQueryCR', {
            serviceToken: synchronousAthenaQueryCRP.serviceToken,
            properties: {
                Statement: props.statement,
                ResultPath: resultPathBucket.s3UrlForObject(props.resultPath.objectKey),
            },
        });
        myCR.node.addDependency(synchronousAthenaQueryCRP);
    }
}
_a = JSII_RTTI_SYMBOL_1;
SynchronousAthenaQuery[_a] = { fqn: "aws-analytics-reference-architecture.SynchronousAthenaQuery", version: "2.12.6" };
exports.SynchronousAthenaQuery = SynchronousAthenaQuery;
//# sourceMappingURL=data:application/json;base64,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