"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LakeFormationAdmin = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const constructs_1 = require("constructs");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const utils_1 = require("../utils");
/**
 * An AWS Lake Formation administrator with privileges to do all the administration tasks in AWS Lake Formation.
 * The principal is an Amazon IAM user or role and is added/removed to the list of AWS Lake Formation administrator
 * via the Data Lake Settings API.
 * Creation/deleting first retrieves the current list of administrators and then add/remove the principal to this list.
 * These steps are done outside of any transaction. Concurrent modifications between retrieving and updating can lead to inconsistent results.
 */
class LakeFormationAdmin extends constructs_1.Construct {
    /**
    * Adds the CDK execution role to LF admins
    * It requires default cdk bootstrap
    */
    static addCdkExecRole(scope, name) {
        // get the CDK execution role
        const cdkRole = utils_1.Utils.getCdkExecRole(scope, name);
        // make the CDK execution role a Lake Formation admin
        return new LakeFormationAdmin(scope, name, {
            principal: cdkRole,
        });
    }
    /**
     * Construct a new instance of LakeFormationAdmin.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {LakeFormationAdminProps} props the LakeFormationAdminProps properties
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.catalogId = props.catalogId || aws_cdk_lib_1.Aws.ACCOUNT_ID;
        this.principal = props.principal;
        // Check if the principal is an Amazon IAM Role or User and extract the arn and name
        const principalArn = this.principal.roleArn ? this.principal.roleArn : this.principal.userArn;
        const principalName = this.principal.roleName ? this.principal.roleName : this.principal.userName;
        const stack = aws_cdk_lib_1.Stack.of(this);
        // AWS Lambda function for the AWS CDK Custom Resource responsible for creating the AWS Lake Formation tag
        const createLfAdminFn = new pre_bundled_function_1.PreBundledFunction(this, 'lfAdminCreateFn', {
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            codePath: 'lake-formation/resources/lambdas/admin',
            handler: 'lambda.on_event',
            logRetention: aws_logs_1.RetentionDays.ONE_DAY,
            timeout: aws_cdk_lib_1.Duration.seconds(20),
        });
        createLfAdminFn.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [
                stack.formatArn({
                    service: 'lakeformation',
                    resource: 'catalog',
                    resourceName: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                    arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                }),
            ],
            actions: [
                'lakeformation:PutDataLakeSettings',
                'lakeformation:GetDataLakeSettings',
            ],
        }));
        // AWS CDK Custom Resource Provider for creating the AWS Lake Formation tag
        const createLfAdminCrp = new custom_resources_1.Provider(this, 'lfAdminCreateCrp', {
            onEventHandler: createLfAdminFn,
            logRetention: aws_logs_1.RetentionDays.ONE_DAY,
        });
        // Create an AWS CDK Custom Resource for starting the source crawler and waiting for completion
        new aws_cdk_lib_1.CustomResource(this, 'lfAdminCreateCr', {
            serviceToken: createLfAdminCrp.serviceToken,
            properties: {
                CatalogId: this.catalogId,
                PrincipalArn: principalArn,
                PrincipalName: principalName,
            },
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
LakeFormationAdmin[_a] = { fqn: "aws-analytics-reference-architecture.LakeFormationAdmin", version: "2.12.6" };
exports.LakeFormationAdmin = LakeFormationAdmin;
//# sourceMappingURL=data:application/json;base64,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