"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrManagedEndpointProvider = void 0;
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const constructs_1 = require("constructs");
const pre_bundled_layer_1 = require("../common/pre-bundled-layer");
/**
 * A custom resource provider for CRUD operations on Amazon EMR on EKS Managed Endpoints.
 * @private
 */
class EmrManagedEndpointProvider extends constructs_1.Construct {
    /**
     * Constructs a new instance of the ManageEndpointProvider. The provider can then be used to create Amazon EMR on EKS Managed Endpoint custom resources
     * @param { Construct} scope the Scope of the CDK Construct
     * @param id the ID of the CDK Construct
     * @private
     */
    constructor(scope, id, props) {
        super(scope, id);
        //The policy allowing the managed endpoint custom resource to create call the APIs for managed endpoint
        const lambdaPolicy = [
            new aws_iam_1.PolicyStatement({
                resources: [props.assetBucket.bucketArn],
                actions: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
            }),
            new aws_iam_1.PolicyStatement({
                resources: ['*'],
                actions: ['emr-containers:DescribeManagedEndpoint'],
            }),
            new aws_iam_1.PolicyStatement({
                resources: ['*'],
                actions: [
                    'emr-containers:DeleteManagedEndpoint'
                ],
                // conditions: { StringEquals: { 'aws:ResourceTag/for-use-with': 'cdk-analytics-reference-architecture' } },
            }),
            new aws_iam_1.PolicyStatement({
                resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:emr-containers:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:/virtualclusters/*`],
                actions: ['emr-containers:CreateManagedEndpoint'],
                conditions: { StringEquals: { 'aws:ResourceTag/for-use-with': 'cdk-analytics-reference-architecture' } },
            }),
            new aws_iam_1.PolicyStatement({
                resources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:emr-containers:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:/virtualclusters/*`],
                actions: ['emr-containers:TagResource'],
                conditions: { StringEquals: { 'aws:ResourceTag/for-use-with': 'cdk-analytics-reference-architecture' } },
            }),
            new aws_iam_1.PolicyStatement({
                resources: ['*'],
                actions: [
                    'ec2:CreateSecurityGroup',
                    'ec2:DeleteSecurityGroup',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:AuthorizeSecurityGroupEgress',
                    'ec2:AuthorizeSecurityGroupIngress',
                    'ec2:RevokeSecurityGroupEgress',
                    'ec2:RevokeSecurityGroupIngress',
                ],
            }),
        ];
        // AWS Lambda function supporting the create, update, delete operations on Amazon EMR on EKS managed endpoints
        const onEvent = new pre_bundled_function_1.PreBundledFunction(this, 'OnEvent', {
            codePath: 'emr-eks-platform/resources/lambdas/managed-endpoint',
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            handler: 'lambda.on_event',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: lambdaPolicy,
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(120),
        });
        // AWS Lambda supporting the status check on asynchronous create, update and delete operations
        const isComplete = new pre_bundled_function_1.PreBundledFunction(this, 'IsComplete', {
            codePath: 'emr-eks-platform/resources/lambdas/managed-endpoint',
            handler: 'lambda.is_complete',
            layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
            lambdaPolicyStatements: lambdaPolicy,
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.seconds(120),
        });
        this.provider = new custom_resources_1.Provider(this, 'CustomResourceProvider', {
            onEventHandler: onEvent,
            isCompleteHandler: isComplete,
            totalTimeout: aws_cdk_lib_1.Duration.minutes(30),
            queryInterval: aws_cdk_lib_1.Duration.seconds(20),
            providerFunctionName: 'managedEndpointProviderFn',
        });
    }
}
exports.EmrManagedEndpointProvider = EmrManagedEndpointProvider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZW1yLW1hbmFnZWQtZW5kcG9pbnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvZW1yLWVrcy1wbGF0Zm9ybS9lbXItbWFuYWdlZC1lbmRwb2ludC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUEscUVBQXFFO0FBQ3JFLGlDQUFpQzs7O0FBRWpDLGlEQUE2RDtBQUM3RCx1REFBaUQ7QUFDakQsbURBQXFEO0FBRXJELDZDQUE0QztBQUM1QyxtRUFBd0Q7QUFDeEQseUVBQW9FO0FBQ3BFLDJDQUF1QztBQUN2QyxtRUFBOEQ7QUE0QzlEOzs7R0FHRztBQUNILE1BQWEsMEJBQTJCLFNBQVEsc0JBQVM7SUFNdkQ7Ozs7O09BS0c7SUFFSCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNDO1FBQzlFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsdUdBQXVHO1FBQ3ZHLE1BQU0sWUFBWSxHQUFHO1lBQ25CLElBQUkseUJBQWUsQ0FBQztnQkFDbEIsU0FBUyxFQUFFLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUM7Z0JBQ3hDLE9BQU8sRUFBRSxDQUFDLGVBQWUsRUFBRSxlQUFlLEVBQUUsVUFBVSxDQUFDO2FBQ3hELENBQUM7WUFDRixJQUFJLHlCQUFlLENBQUM7Z0JBQ2xCLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztnQkFDaEIsT0FBTyxFQUFFLENBQUMsd0NBQXdDLENBQUM7YUFDcEQsQ0FBQztZQUNGLElBQUkseUJBQWUsQ0FBQztnQkFDbEIsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO2dCQUNoQixPQUFPLEVBQUU7b0JBQ1Asc0NBQXNDO2lCQUFDO2dCQUN6Qyw0R0FBNEc7YUFDN0csQ0FBQztZQUNGLElBQUkseUJBQWUsQ0FBQztnQkFDbEIsU0FBUyxFQUFFLENBQUMsT0FBTyxpQkFBRyxDQUFDLFNBQVMsbUJBQW1CLGlCQUFHLENBQUMsTUFBTSxJQUFJLGlCQUFHLENBQUMsVUFBVSxxQkFBcUIsQ0FBQztnQkFDckcsT0FBTyxFQUFFLENBQUMsc0NBQXNDLENBQUM7Z0JBQ2pELFVBQVUsRUFBRSxFQUFFLFlBQVksRUFBRSxFQUFFLDhCQUE4QixFQUFFLHNDQUFzQyxFQUFFLEVBQUU7YUFDekcsQ0FBQztZQUNGLElBQUkseUJBQWUsQ0FBQztnQkFDbEIsU0FBUyxFQUFFLENBQUMsT0FBTyxpQkFBRyxDQUFDLFNBQVMsbUJBQW1CLGlCQUFHLENBQUMsTUFBTSxJQUFJLGlCQUFHLENBQUMsVUFBVSxxQkFBcUIsQ0FBQztnQkFDckcsT0FBTyxFQUFFLENBQUMsNEJBQTRCLENBQUM7Z0JBQ3ZDLFVBQVUsRUFBRSxFQUFFLFlBQVksRUFBRSxFQUFFLDhCQUE4QixFQUFFLHNDQUFzQyxFQUFFLEVBQUU7YUFDekcsQ0FBQztZQUNGLElBQUkseUJBQWUsQ0FBQztnQkFDbEIsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO2dCQUNoQixPQUFPLEVBQUU7b0JBQ1AseUJBQXlCO29CQUN6Qix5QkFBeUI7b0JBQ3pCLCtCQUErQjtvQkFDL0Isa0NBQWtDO29CQUNsQyxtQ0FBbUM7b0JBQ25DLCtCQUErQjtvQkFDL0IsZ0NBQWdDO2lCQUNqQzthQUNGLENBQUM7U0FDSCxDQUFDO1FBRUYsOEdBQThHO1FBQzlHLE1BQU0sT0FBTyxHQUFHLElBQUkseUNBQWtCLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUN0RCxRQUFRLEVBQUUscURBQXFEO1lBQy9ELE9BQU8sRUFBRSxvQkFBTyxDQUFDLFVBQVU7WUFDM0IsT0FBTyxFQUFFLGlCQUFpQjtZQUMxQixNQUFNLEVBQUUsQ0FBQyxtQ0FBZSxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsNENBQTRDLENBQUMsQ0FBQztZQUMxRixzQkFBc0IsRUFBRSxZQUFZO1lBQ3BDLFlBQVksRUFBRSx3QkFBYSxDQUFDLFFBQVE7WUFDcEMsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQztTQUMvQixDQUFDLENBQUM7UUFFSCw4RkFBOEY7UUFDOUYsTUFBTSxVQUFVLEdBQUcsSUFBSSx5Q0FBa0IsQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQzVELFFBQVEsRUFBRSxxREFBcUQ7WUFDL0QsT0FBTyxFQUFFLG9CQUFvQjtZQUM3QixNQUFNLEVBQUUsQ0FBQyxtQ0FBZSxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsNENBQTRDLENBQUMsQ0FBQztZQUMxRixzQkFBc0IsRUFBRSxZQUFZO1lBQ3BDLE9BQU8sRUFBRSxvQkFBTyxDQUFDLFVBQVU7WUFDM0IsWUFBWSxFQUFFLHdCQUFhLENBQUMsUUFBUTtZQUNwQyxPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsR0FBRyxDQUFDO1NBQy9CLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSwyQkFBUSxDQUFDLElBQUksRUFBRSx3QkFBd0IsRUFBRTtZQUMzRCxjQUFjLEVBQUUsT0FBTztZQUN2QixpQkFBaUIsRUFBRSxVQUFVO1lBQzdCLFlBQVksRUFBRSxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDbEMsYUFBYSxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUNuQyxvQkFBb0IsRUFBRSwyQkFBMkI7U0FDbEQsQ0FBQyxDQUFDO0lBRUwsQ0FBQztDQUNGO0FBdkZELGdFQXVGQyIsInNvdXJjZXNDb250ZW50IjpbIi8vIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuLy8gU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IE1JVC0wXG5cbmltcG9ydCB7IElSb2xlLCBQb2xpY3lTdGF0ZW1lbnQgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCB7IFJ1bnRpbWUgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCB7IFJldGVudGlvbkRheXMgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbG9ncyc7XG5pbXBvcnQgeyBJQnVja2V0IH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLXMzJztcbmltcG9ydCB7IER1cmF0aW9uLCBBd3MgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgeyBQcm92aWRlciB9IGZyb20gJ2F3cy1jZGstbGliL2N1c3RvbS1yZXNvdXJjZXMnO1xuaW1wb3J0IHsgUHJlQnVuZGxlZEZ1bmN0aW9uIH0gZnJvbSAnLi4vY29tbW9uL3ByZS1idW5kbGVkLWZ1bmN0aW9uJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgUHJlQnVuZGxlZExheWVyIH0gZnJvbSAnLi4vY29tbW9uL3ByZS1idW5kbGVkLWxheWVyJztcbmltcG9ydCB7IEVtclZlcnNpb24gfSBmcm9tICcuL2Vtci1la3MtY2x1c3Rlcic7XG5cbi8qKlxuKiBUaGUgcHJvcGVydGllcyBmb3IgdGhlIEVNUiBNYW5hZ2VkIEVuZHBvaW50IHRvIGNyZWF0ZS5cbiovXG5leHBvcnQgaW50ZXJmYWNlIEVtck1hbmFnZWRFbmRwb2ludE9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIEVNUiBtYW5hZ2VkIGVuZHBvaW50XG4gICAqL1xuICByZWFkb25seSBtYW5hZ2VkRW5kcG9pbnROYW1lOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgSWQgb2YgdGhlIEFtYXpvbiBFTVIgdmlydHVhbCBjbHVzdGVyIGNvbnRhaW5pbmcgdGhlIG1hbmFnZWQgZW5kcG9pbnRcbiAgICovXG4gIHJlYWRvbmx5IHZpcnR1YWxDbHVzdGVySWQ6IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gSUFNIHJvbGUgdXNlZCBhcyB0aGUgZXhlY3V0aW9uIHJvbGUsIHRoaXMgcm9sZSBtdXN0IHByb3ZpZGUgYWNjZXNzIHRvIGFsbCB0aGUgQVdTIHJlc291cmNlIGEgdXNlciB3aWxsIGludGVyYWN0IHdpdGhcbiAgICogVGhlc2UgY2FuIGJlIFMzLCBEeW5hbW9EQiwgR2x1ZSBDYXRhbG9nXG4gICAqL1xuICByZWFkb25seSBleGVjdXRpb25Sb2xlOiBJUm9sZTtcbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gRU1SIHZlcnNpb24gdG8gdXNlXG4gICAqIEBkZWZhdWx0IC0gVGhlIFtkZWZhdWx0IEFtYXpvbiBFTVIgdmVyc2lvbl17QGxpbmsgRW1yRWtzQ2x1c3Rlci5ERUZBVUxUX0VNUl9WRVJTSU9OfVxuICAgKi9cbiAgcmVhZG9ubHkgZW1yT25Fa3NWZXJzaW9uPzogRW1yVmVyc2lvbjtcbiAgLyoqXG4gICAqIFRoZSBKU09OIGNvbmZpZ3VyYXRpb24gb3ZlcnJpZGVzIGZvciBBbWF6b24gRU1SIG9uIEVLUyBjb25maWd1cmF0aW9uIGF0dGFjaGVkIHRvIHRoZSBtYW5hZ2VkIGVuZHBvaW50XG4gICAqIEBkZWZhdWx0IC0gQ29uZmlndXJhdGlvbiByZWxhdGVkIHRvIHRoZSBbZGVmYXVsdCBub2RlZ3JvdXAgZm9yIG5vdGVib29rXXtAbGluayBFbXJFa3NOb2RlZ3JvdXAuTk9URUJPT0tfRVhFQ1VUT1J9XG4gICAqL1xuICByZWFkb25seSBjb25maWd1cmF0aW9uT3ZlcnJpZGVzPzogc3RyaW5nO1xufVxuXG5cbi8qKlxuICogVGhlIHByb3BlcnRpZXMgZm9yIHRoZSBFTVIgb24gRUtTIE1hbmFnZWQgRW5kcG9pbnQgdG8gY3JlYXRlLlxuICogQHByaXZhdGVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFbXJNYW5hZ2VkRW5kcG9pbnRQcm92aWRlclByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBidWNrZXQgY29udGFpbmluZyB0aGUgazhzIGFzc2V0cywgaW4gdGhpcyBjYXNlIHRoZSBwb2QgdGVtcGxhdGVzXG4gICAqL1xuICByZWFkb25seSBhc3NldEJ1Y2tldDogSUJ1Y2tldDtcbn1cblxuLyoqXG4gKiBBIGN1c3RvbSByZXNvdXJjZSBwcm92aWRlciBmb3IgQ1JVRCBvcGVyYXRpb25zIG9uIEFtYXpvbiBFTVIgb24gRUtTIE1hbmFnZWQgRW5kcG9pbnRzLlxuICogQHByaXZhdGVcbiAqL1xuZXhwb3J0IGNsYXNzIEVtck1hbmFnZWRFbmRwb2ludFByb3ZpZGVyIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgLyoqXG4gICAqIFRoZSBjdXN0b20gcmVzb3VyY2UgUHJvdmlkZXIgZm9yIGNyZWF0aW5nIEFtYXpvbiBFTVIgTWFuYWdlZCBFbmRwb2ludHMgY3VzdG9tIHJlc291cmNlc1xuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHByb3ZpZGVyOiBQcm92aWRlcjtcblxuICAvKipcbiAgICogQ29uc3RydWN0cyBhIG5ldyBpbnN0YW5jZSBvZiB0aGUgTWFuYWdlRW5kcG9pbnRQcm92aWRlci4gVGhlIHByb3ZpZGVyIGNhbiB0aGVuIGJlIHVzZWQgdG8gY3JlYXRlIEFtYXpvbiBFTVIgb24gRUtTIE1hbmFnZWQgRW5kcG9pbnQgY3VzdG9tIHJlc291cmNlc1xuICAgKiBAcGFyYW0geyBDb25zdHJ1Y3R9IHNjb3BlIHRoZSBTY29wZSBvZiB0aGUgQ0RLIENvbnN0cnVjdFxuICAgKiBAcGFyYW0gaWQgdGhlIElEIG9mIHRoZSBDREsgQ29uc3RydWN0XG4gICAqIEBwcml2YXRlXG4gICAqL1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBFbXJNYW5hZ2VkRW5kcG9pbnRQcm92aWRlclByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIC8vVGhlIHBvbGljeSBhbGxvd2luZyB0aGUgbWFuYWdlZCBlbmRwb2ludCBjdXN0b20gcmVzb3VyY2UgdG8gY3JlYXRlIGNhbGwgdGhlIEFQSXMgZm9yIG1hbmFnZWQgZW5kcG9pbnRcbiAgICBjb25zdCBsYW1iZGFQb2xpY3kgPSBbXG4gICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgcmVzb3VyY2VzOiBbcHJvcHMuYXNzZXRCdWNrZXQuYnVja2V0QXJuXSxcbiAgICAgICAgYWN0aW9uczogWydzMzpHZXRPYmplY3QqJywgJ3MzOkdldEJ1Y2tldConLCAnczM6TGlzdConXSxcbiAgICAgIH0pLFxuICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgIGFjdGlvbnM6IFsnZW1yLWNvbnRhaW5lcnM6RGVzY3JpYmVNYW5hZ2VkRW5kcG9pbnQnXSxcbiAgICAgIH0pLFxuICAgICAgbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAnZW1yLWNvbnRhaW5lcnM6RGVsZXRlTWFuYWdlZEVuZHBvaW50J10sXG4gICAgICAgIC8vIGNvbmRpdGlvbnM6IHsgU3RyaW5nRXF1YWxzOiB7ICdhd3M6UmVzb3VyY2VUYWcvZm9yLXVzZS13aXRoJzogJ2Nkay1hbmFseXRpY3MtcmVmZXJlbmNlLWFyY2hpdGVjdHVyZScgfSB9LFxuICAgICAgfSksXG4gICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgcmVzb3VyY2VzOiBbYGFybjoke0F3cy5QQVJUSVRJT059OmVtci1jb250YWluZXJzOiR7QXdzLlJFR0lPTn06JHtBd3MuQUNDT1VOVF9JRH06L3ZpcnR1YWxjbHVzdGVycy8qYF0sXG4gICAgICAgIGFjdGlvbnM6IFsnZW1yLWNvbnRhaW5lcnM6Q3JlYXRlTWFuYWdlZEVuZHBvaW50J10sXG4gICAgICAgIGNvbmRpdGlvbnM6IHsgU3RyaW5nRXF1YWxzOiB7ICdhd3M6UmVzb3VyY2VUYWcvZm9yLXVzZS13aXRoJzogJ2Nkay1hbmFseXRpY3MtcmVmZXJlbmNlLWFyY2hpdGVjdHVyZScgfSB9LFxuICAgICAgfSksXG4gICAgICBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgcmVzb3VyY2VzOiBbYGFybjoke0F3cy5QQVJUSVRJT059OmVtci1jb250YWluZXJzOiR7QXdzLlJFR0lPTn06JHtBd3MuQUNDT1VOVF9JRH06L3ZpcnR1YWxjbHVzdGVycy8qYF0sXG4gICAgICAgIGFjdGlvbnM6IFsnZW1yLWNvbnRhaW5lcnM6VGFnUmVzb3VyY2UnXSxcbiAgICAgICAgY29uZGl0aW9uczogeyBTdHJpbmdFcXVhbHM6IHsgJ2F3czpSZXNvdXJjZVRhZy9mb3ItdXNlLXdpdGgnOiAnY2RrLWFuYWx5dGljcy1yZWZlcmVuY2UtYXJjaGl0ZWN0dXJlJyB9IH0sXG4gICAgICB9KSxcbiAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgJ2VjMjpDcmVhdGVTZWN1cml0eUdyb3VwJyxcbiAgICAgICAgICAnZWMyOkRlbGV0ZVNlY3VyaXR5R3JvdXAnLFxuICAgICAgICAgICdlYzI6RGVzY3JpYmVOZXR3b3JrSW50ZXJmYWNlcycsXG4gICAgICAgICAgJ2VjMjpBdXRob3JpemVTZWN1cml0eUdyb3VwRWdyZXNzJyxcbiAgICAgICAgICAnZWMyOkF1dGhvcml6ZVNlY3VyaXR5R3JvdXBJbmdyZXNzJyxcbiAgICAgICAgICAnZWMyOlJldm9rZVNlY3VyaXR5R3JvdXBFZ3Jlc3MnLFxuICAgICAgICAgICdlYzI6UmV2b2tlU2VjdXJpdHlHcm91cEluZ3Jlc3MnLFxuICAgICAgICBdLFxuICAgICAgfSksXG4gICAgXTtcblxuICAgIC8vIEFXUyBMYW1iZGEgZnVuY3Rpb24gc3VwcG9ydGluZyB0aGUgY3JlYXRlLCB1cGRhdGUsIGRlbGV0ZSBvcGVyYXRpb25zIG9uIEFtYXpvbiBFTVIgb24gRUtTIG1hbmFnZWQgZW5kcG9pbnRzXG4gICAgY29uc3Qgb25FdmVudCA9IG5ldyBQcmVCdW5kbGVkRnVuY3Rpb24odGhpcywgJ09uRXZlbnQnLCB7XG4gICAgICBjb2RlUGF0aDogJ2Vtci1la3MtcGxhdGZvcm0vcmVzb3VyY2VzL2xhbWJkYXMvbWFuYWdlZC1lbmRwb2ludCcsXG4gICAgICBydW50aW1lOiBSdW50aW1lLlBZVEhPTl8zXzksXG4gICAgICBoYW5kbGVyOiAnbGFtYmRhLm9uX2V2ZW50JyxcbiAgICAgIGxheWVyczogW1ByZUJ1bmRsZWRMYXllci5nZXRPckNyZWF0ZShzY29wZSwgJ2NvbW1vbi9yZXNvdXJjZXMvbGFtYmRhcy9wcmUtYnVuZGxlZC1sYXllcicpXSxcbiAgICAgIGxhbWJkYVBvbGljeVN0YXRlbWVudHM6IGxhbWJkYVBvbGljeSxcbiAgICAgIGxvZ1JldGVudGlvbjogUmV0ZW50aW9uRGF5cy5PTkVfV0VFSyxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLnNlY29uZHMoMTIwKSxcbiAgICB9KTtcblxuICAgIC8vIEFXUyBMYW1iZGEgc3VwcG9ydGluZyB0aGUgc3RhdHVzIGNoZWNrIG9uIGFzeW5jaHJvbm91cyBjcmVhdGUsIHVwZGF0ZSBhbmQgZGVsZXRlIG9wZXJhdGlvbnNcbiAgICBjb25zdCBpc0NvbXBsZXRlID0gbmV3IFByZUJ1bmRsZWRGdW5jdGlvbih0aGlzLCAnSXNDb21wbGV0ZScsIHtcbiAgICAgIGNvZGVQYXRoOiAnZW1yLWVrcy1wbGF0Zm9ybS9yZXNvdXJjZXMvbGFtYmRhcy9tYW5hZ2VkLWVuZHBvaW50JyxcbiAgICAgIGhhbmRsZXI6ICdsYW1iZGEuaXNfY29tcGxldGUnLFxuICAgICAgbGF5ZXJzOiBbUHJlQnVuZGxlZExheWVyLmdldE9yQ3JlYXRlKHNjb3BlLCAnY29tbW9uL3Jlc291cmNlcy9sYW1iZGFzL3ByZS1idW5kbGVkLWxheWVyJyldLFxuICAgICAgbGFtYmRhUG9saWN5U3RhdGVtZW50czogbGFtYmRhUG9saWN5LFxuICAgICAgcnVudGltZTogUnVudGltZS5QWVRIT05fM185LFxuICAgICAgbG9nUmV0ZW50aW9uOiBSZXRlbnRpb25EYXlzLk9ORV9XRUVLLFxuICAgICAgdGltZW91dDogRHVyYXRpb24uc2Vjb25kcygxMjApLFxuICAgIH0pO1xuXG4gICAgdGhpcy5wcm92aWRlciA9IG5ldyBQcm92aWRlcih0aGlzLCAnQ3VzdG9tUmVzb3VyY2VQcm92aWRlcicsIHtcbiAgICAgIG9uRXZlbnRIYW5kbGVyOiBvbkV2ZW50LFxuICAgICAgaXNDb21wbGV0ZUhhbmRsZXI6IGlzQ29tcGxldGUsXG4gICAgICB0b3RhbFRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMzApLFxuICAgICAgcXVlcnlJbnRlcnZhbDogRHVyYXRpb24uc2Vjb25kcygyMCksXG4gICAgICBwcm92aWRlckZ1bmN0aW9uTmFtZTogJ21hbmFnZWRFbmRwb2ludFByb3ZpZGVyRm4nLFxuICAgIH0pO1xuXG4gIH1cbn1cbiJdfQ==