"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrEksCluster = exports.EmrVersion = exports.Autoscaler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const aws_emrcontainers_1 = require("aws-cdk-lib/aws-emrcontainers");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const SimpleBase = require("simple-base");
const ara_bucket_1 = require("../ara-bucket");
const context_options_1 = require("../common/context-options");
const tracked_construct_1 = require("../common/tracked-construct");
const singleton_kms_key_1 = require("../singleton-kms-key");
const emr_managed_endpoint_1 = require("./emr-managed-endpoint");
const CriticalDefaultConfig = require("./resources/k8s/emr-eks-config/critical.json");
const NotebookDefaultConfig = require("./resources/k8s/emr-eks-config/notebook-pod-template-ready.json");
const SharedDefaultConfig = require("./resources/k8s/emr-eks-config/shared.json");
const K8sRoleBinding = require("./resources/k8s/rbac/emr-containers-role-binding.json");
const K8sRole = require("./resources/k8s/rbac/emr-containers-role.json");
const emr_eks_cluster_helpers_1 = require("./emr-eks-cluster-helpers");
const singleton_launch_template_1 = require("../singleton-launch-template");
const emr_eks_nodegroup_asg_tag_1 = require("./emr-eks-nodegroup-asg-tag");
const emr_eks_job_template_1 = require("./emr-eks-job-template");
const vpc_helper_1 = require("./vpc-helper");
/**
 * The different autoscaler available with EmrEksCluster
 */
var Autoscaler;
(function (Autoscaler) {
    Autoscaler["KARPENTER"] = "KARPENTER";
    Autoscaler["CLUSTER_AUTOSCALER"] = "CLUSTER_AUTOSCALER";
})(Autoscaler = exports.Autoscaler || (exports.Autoscaler = {}));
/**
 * The different EMR versions available on EKS
 */
var EmrVersion;
(function (EmrVersion) {
    EmrVersion["V6_10"] = "emr-6.10.0-latest";
    EmrVersion["V6_9"] = "emr-6.9.0-latest";
    EmrVersion["V6_8"] = "emr-6.8.0-latest";
    EmrVersion["V6_7"] = "emr-6.7.0-latest";
    EmrVersion["V6_6"] = "emr-6.6.0-latest";
    EmrVersion["V6_5"] = "emr-6.5.0-latest";
    EmrVersion["V6_4"] = "emr-6.4.0-latest";
    EmrVersion["V6_3"] = "emr-6.3.0-latest";
    EmrVersion["V6_2"] = "emr-6.2.0-latest";
    EmrVersion["V5_33"] = "emr-5.33.0-latest";
    EmrVersion["V5_32"] = "emr-5.32.0-latest";
})(EmrVersion = exports.EmrVersion || (exports.EmrVersion = {}));
/**
 * EmrEksCluster Construct packaging all the resources and configuration required to run Amazon EMR on EKS.
 * It deploys:
 * * An EKS cluster (VPC configuration can be customized)
 * * A tooling nodegroup to run tools including the Kubedashboard and the Cluster Autoscaler
 * * Optionally multiple nodegroups (one per AZ) for critical/shared/notebook EMR workloads
 * * Additional nodegroups can be configured
 *
 * The construct will upload on S3 the Pod templates required to run EMR jobs on the default nodegroups.
 * It will also parse and store the configuration of EMR on EKS jobs for each default nodegroup in object parameters
 *
 * Methods are available to add EMR Virtual Clusters to the EKS cluster and to create execution roles for the virtual clusters.
 *
 * Usage example:
 *
 * ```typescript
 * const emrEks: EmrEksCluster = EmrEksCluster.getOrCreate(stack, {
 *   eksAdminRoleArn: <ROLE_ARN>,
 *   eksClusterName: <CLUSTER_NAME>,
 * });
 *
 * const virtualCluster = emrEks.addEmrVirtualCluster(stack, {
 *   name: <Virtual_Cluster_Name>,
 *   createNamespace: <TRUE OR FALSE>,
 *   eksNamespace: <K8S_namespace>,
 * });
 *
 * const role = emrEks.createExecutionRole(stack, 'ExecRole',{
 *   policy: <POLICY>,
 * })
 *
 * // EMR on EKS virtual cluster ID
 * cdk.CfnOutput(self, 'VirtualClusterId',value = virtualCluster.attr_id)
 * // Job config for each nodegroup
 * cdk.CfnOutput(self, "CriticalConfig", value = emrEks.criticalDefaultConfig)
 * cdk.CfnOutput(self, "SharedConfig", value = emrEks.sharedDefaultConfig)
 * // Execution role arn
 * cdk.CfnOutput(self,'ExecRoleArn', value = role.roleArn)
 * ```
 *
 */
class EmrEksCluster extends tracked_construct_1.TrackedConstruct {
    /**
     * Get an existing EmrEksCluster based on the cluster name property or create a new one
     * only one EKS cluster can exist per stack
     * @param {Construct} scope the CDK scope used to search or create the cluster
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps} if created
     */
    static getOrCreate(scope, props) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = props.eksClusterName || EmrEksCluster.DEFAULT_CLUSTER_NAME;
        let emrEksCluster;
        if (stack.node.tryFindChild(id) == undefined) {
            emrEksCluster = new EmrEksCluster(stack, id, props);
        }
        return stack.node.tryFindChild(id) || emrEksCluster;
    }
    /**
     * Constructs a new instance of the EmrEksCluster construct.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps}
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingCode: context_options_1.ContextOptions.EMR_EKS_TRACKING_ID,
        };
        super(scope, id, trackedConstructProps);
        this.clusterName = props.eksClusterName ?? EmrEksCluster.DEFAULT_CLUSTER_NAME;
        //Define EKS cluster logging
        const eksClusterLogging = [
            aws_eks_1.ClusterLoggingTypes.API,
            aws_eks_1.ClusterLoggingTypes.AUTHENTICATOR,
            aws_eks_1.ClusterLoggingTypes.SCHEDULER,
            aws_eks_1.ClusterLoggingTypes.CONTROLLER_MANAGER,
            aws_eks_1.ClusterLoggingTypes.AUDIT,
        ];
        //Set the autoscaler mechanism flag
        this.isKarpenter = props.autoscaling == Autoscaler.KARPENTER ? true : false;
        this.defaultNodes = props.defaultNodes == undefined ? true : props.defaultNodes;
        // Create a role to be used as instance profile for nodegroups
        this.ec2InstanceNodeGroupRole = new aws_iam_1.Role(this, 'ec2InstanceNodeGroupRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
        });
        //attach policies to the role to be used by the nodegroups
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        // Create the custom resource provider for tagging the EC2 Auto Scaling groups
        this.nodegroupAsgTagsProviderServiceToken = new emr_eks_nodegroup_asg_tag_1.EmrEksNodegroupAsgTagProvider(this, 'AsgTagProvider', {
            eksClusterName: this.clusterName,
        }).provider.serviceToken;
        // Create the custom resource provider for tagging the EC2 Auto Scaling groups
        this.jobTemplateProviderToken = new emr_eks_job_template_1.EmrEksJobTemplateProvider(this, 'jobTemplateProvider').provider.serviceToken;
        // create an Amazon EKS CLuster with default parameters if not provided in the properties
        if (props.eksCluster == undefined) {
            let eksVpc = props.vpcCidr ? (0, vpc_helper_1.vpcBootstrap)(scope, props.vpcCidr, this.clusterName) : props.eksVpc;
            this.eksCluster = new aws_eks_1.Cluster(scope, `${this.clusterName}Cluster`, {
                defaultCapacity: 0,
                clusterName: this.clusterName,
                version: props.kubernetesVersion || EmrEksCluster.DEFAULT_EKS_VERSION,
                clusterLogging: eksClusterLogging,
                kubectlLayer: props.kubectlLambdaLayer ?? undefined,
                vpc: eksVpc
            });
            //Create VPC flow log for the EKS VPC
            let eksVpcFlowLogLogGroup = new aws_logs_1.LogGroup(this, 'eksVpcFlowLogLogGroup', {
                logGroupName: `/aws/emr-eks-vpc-flow/${this.clusterName}`,
                encryptionKey: singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey'),
                retention: aws_logs_1.RetentionDays.ONE_WEEK,
                removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            });
            //Allow vpc flowlog to access KMS key to encrypt logs
            singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey').addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                principals: [new aws_iam_1.ServicePrincipal(`logs.${aws_cdk_lib_1.Stack.of(this).region}.amazonaws.com`)],
                actions: [
                    'kms:Encrypt*',
                    'kms:Decrypt*',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                    'kms:Describe*',
                ],
                conditions: {
                    ArnLike: {
                        'kms:EncryptionContext:aws:logs:arn': `arn:aws:logs:${aws_cdk_lib_1.Stack.of(this).region}:${aws_cdk_lib_1.Stack.of(this).account}:*`,
                    },
                },
                resources: ['*'],
            }));
            //Setup the VPC flow logs
            const iamRoleforFlowLog = new aws_iam_1.Role(this, 'iamRoleforFlowLog', {
                assumedBy: new aws_iam_1.ServicePrincipal('vpc-flow-logs.amazonaws.com'),
            });
            this.eksCluster.vpc.addFlowLog('eksVpcFlowLog', {
                destination: aws_ec2_1.FlowLogDestination.toCloudWatchLogs(eksVpcFlowLogLogGroup, iamRoleforFlowLog),
            });
            //Setting up the cluster with the required controller
            (0, emr_eks_cluster_helpers_1.eksClusterSetup)(this, this, props.eksAdminRoleArn);
            //Deploy the right autoscaler using the flag set earlier 
            if (this.isKarpenter) {
                this.karpenterChart = (0, emr_eks_cluster_helpers_1.karpenterSetup)(this.eksCluster, this.clusterName, this, props.karpenterVersion || EmrEksCluster.DEFAULT_KARPENTER_VERSION);
            }
            else {
                const kubernetesVersion = props.kubernetesVersion ?? EmrEksCluster.DEFAULT_EKS_VERSION;
                (0, emr_eks_cluster_helpers_1.clusterAutoscalerSetup)(this.eksCluster, this.clusterName, this, kubernetesVersion);
            }
        }
        else {
            //Initialize with the provided EKS Cluster
            this.eksCluster = props.eksCluster;
        }
        //Check if the user want to use the default nodegroup and
        //Add the default nodegroup configured and optimized to run Spark workloads
        if (this.defaultNodes && props.autoscaling == Autoscaler.CLUSTER_AUTOSCALER) {
            (0, emr_eks_cluster_helpers_1.setDefaultManagedNodeGroups)(this);
        }
        //Check if there user want to use the default Karpenter provisioners and
        //Add the defaults pre-configured and optimized to run Spark workloads
        if (this.defaultNodes && props.autoscaling == Autoscaler.KARPENTER) {
            (0, emr_eks_cluster_helpers_1.setDefaultKarpenterProvisioners)(this);
        }
        ara_bucket_1.AraBucket.getOrCreate(this, { bucketName: 's3-access-logs', objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_PREFERRED });
        // Tags the Amazon VPC and Subnets of the Amazon EKS Cluster
        aws_cdk_lib_1.Tags.of(this.eksCluster.vpc).add('for-use-with-amazon-emr-managed-policies', 'true');
        this.eksCluster.vpc.privateSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('for-use-with-amazon-emr-managed-policies', 'true'));
        this.eksCluster.vpc.publicSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('for-use-with-amazon-emr-managed-policies', 'true'));
        // Create Amazon IAM ServiceLinkedRole for Amazon EMR and add to kubernetes configmap
        // required to add a dependency on the Amazon EMR virtual cluster
        this.emrServiceRole = new aws_iam_1.CfnServiceLinkedRole(this, 'EmrServiceRole', {
            awsServiceName: 'emr-containers.amazonaws.com',
        });
        this.eksCluster.awsAuth.addRoleMapping(aws_iam_1.Role.fromRoleArn(this, 'ServiceRoleForAmazonEMRContainers', `arn:aws:iam::${aws_cdk_lib_1.Stack.of(this).account}:role/AWSServiceRoleForAmazonEMRContainers`), {
            username: 'emr-containers',
            groups: ['']
        });
        // Create an Amazon S3 Bucket for default podTemplate assets
        this.assetBucket = ara_bucket_1.AraBucket.getOrCreate(this, { bucketName: `${this.clusterName.toLowerCase()}-emr-eks-assets`, encryption: aws_s3_1.BucketEncryption.KMS_MANAGED });
        // Configure the podTemplate location
        this.podTemplateLocation = {
            bucketName: this.assetBucket.bucketName,
            objectKey: `${this.clusterName}/pod-template`,
        };
        aws_cdk_lib_1.Tags.of(this.assetBucket).add('for-use-with', 'cdk-analytics-reference-architecture');
        let s3DeploymentLambdaPolicyStatement = [];
        s3DeploymentLambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        //Policy to allow lambda access to cloudwatch logs
        const lambdaExecutionRolePolicy = new aws_iam_1.ManagedPolicy(this, 's3BucketDeploymentPolicy', {
            statements: s3DeploymentLambdaPolicyStatement,
            description: 'Policy used by S3 deployment cdk construct',
        });
        //Create an execution role for the lambda and attach to it a policy formed from user input
        this.assetUploadBucketRole = new aws_iam_1.Role(this, 's3BucketDeploymentRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            description: 'Role used by S3 deployment cdk construct',
            managedPolicies: [lambdaExecutionRolePolicy],
        });
        // Upload the default podTemplate to the Amazon S3 asset bucket
        this.uploadPodTemplate('defaultPodTemplates', (0, path_1.join)(__dirname, 'resources/k8s/pod-template'));
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the notebook default config
        NotebookDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-driver.yaml`);
        NotebookDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-executor.yaml`);
        this.notebookDefaultConfig = JSON.parse(JSON.stringify(NotebookDefaultConfig));
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the critical default config
        CriticalDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-driver.yaml`);
        CriticalDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-executor.yaml`);
        this.criticalDefaultConfig = JSON.stringify(CriticalDefaultConfig);
        // Replace the pod template location for driver and executor with the correct Amazon S3 path in the shared default config
        SharedDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-driver.yaml`);
        SharedDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-executor.yaml`);
        this.sharedDefaultConfig = JSON.stringify(SharedDefaultConfig);
        // Set the custom resource provider service token here to avoid circular dependencies
        this.managedEndpointProviderServiceToken = new emr_managed_endpoint_1.EmrManagedEndpointProvider(this, 'ManagedEndpointProvider', {
            assetBucket: this.assetBucket,
        }).provider.serviceToken;
        // Provide the podTemplate location on Amazon S3
        new aws_cdk_lib_1.CfnOutput(this, 'podTemplateLocation', {
            description: 'Use podTemplates in Amazon EMR jobs from this Amazon S3 Location',
            value: this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}`),
        });
    }
    /**
     * Add a new Amazon EMR Virtual Cluster linked to Amazon EKS Cluster.
     * @param {Construct} scope of the stack where virtual cluster is deployed
     * @param {EmrVirtualClusterOptions} options the EmrVirtualClusterProps [properties]{@link EmrVirtualClusterProps}
     */
    addEmrVirtualCluster(scope, options) {
        const eksNamespace = options.eksNamespace ?? 'default';
        const regex = /^[a-z0-9]+$/g;
        if (!eksNamespace.match(regex)) {
            throw new Error(`Namespace provided violates the constraints of Namespace naming ${eksNamespace}`);
        }
        const ns = options.createNamespace
            ? this.eksCluster.addManifest(`${options.name}Namespace`, {
                apiVersion: 'v1',
                kind: 'Namespace',
                metadata: { name: eksNamespace },
            })
            : null;
        // deep clone the Role template object and replace the namespace
        const k8sRole = JSON.parse(JSON.stringify(K8sRole));
        k8sRole.metadata.namespace = eksNamespace;
        const role = this.eksCluster.addManifest(`${options.name}Role`, k8sRole);
        role.node.addDependency(this.emrServiceRole);
        if (ns)
            role.node.addDependency(ns);
        // deep clone the Role Binding template object and replace the namespace
        const k8sRoleBinding = JSON.parse(JSON.stringify(K8sRoleBinding));
        k8sRoleBinding.metadata.namespace = eksNamespace;
        const roleBinding = this.eksCluster.addManifest(`${options.name}RoleBinding`, k8sRoleBinding);
        roleBinding.node.addDependency(role);
        const virtCluster = new aws_emrcontainers_1.CfnVirtualCluster(scope, `${options.name}VirtualCluster`, {
            name: options.name,
            containerProvider: {
                id: this.clusterName,
                type: 'EKS',
                info: { eksInfo: { namespace: options.eksNamespace || 'default' } },
            },
            tags: [{
                    key: 'for-use-with',
                    value: 'cdk-analytics-reference-architecture',
                }],
        });
        virtCluster.node.addDependency(roleBinding);
        virtCluster.node.addDependency(this.emrServiceRole);
        if (ns)
            virtCluster.node.addDependency(ns);
        aws_cdk_lib_1.Tags.of(virtCluster).add('for-use-with', 'cdk-analytics-reference-architecture');
        return virtCluster;
    }
    /**
     * Creates a new Amazon EMR managed endpoint to be used with Amazon EMR Virtual Cluster .
     * CfnOutput can be customized.
     * @param {Construct} scope the scope of the stack where managed endpoint is deployed
     * @param {string} id the CDK id for endpoint
     * @param {EmrManagedEndpointOptions} options the EmrManagedEndpointOptions to configure the Amazon EMR managed endpoint
     */
    addManagedEndpoint(scope, id, options) {
        if (options.managedEndpointName.length > 64) {
            throw new Error(`error managed endpoint name length is greater than 64 ${id}`);
        }
        if (this.notebookDefaultConfig == undefined) {
            throw new Error('error empty configuration override is not supported on non-default nodegroups');
        }
        let jsonConfigurationOverrides;
        // TODO this need to be broadended to all possible emr configuration
        // try {
        //   //Check if the configOverride provided by user is valid
        //   let isConfigOverrideValid: boolean = validateSchema(JSON.stringify(configOverrideSchema), options.configurationOverrides);
        //   jsonConfigurationOverrides = isConfigOverrideValid ? options.configurationOverrides : this.notebookDefaultConfig;
        // } catch (error) {
        //   throw new Error(`The configuration override is not valid JSON : ${options.configurationOverrides}`);
        // }
        jsonConfigurationOverrides = options.configurationOverrides ? options.configurationOverrides : this.notebookDefaultConfig;
        // Create custom resource with async waiter until the Amazon EMR Managed Endpoint is created
        const cr = new aws_cdk_lib_1.CustomResource(scope, id, {
            serviceToken: this.managedEndpointProviderServiceToken,
            properties: {
                clusterId: options.virtualClusterId,
                executionRoleArn: options.executionRole.roleArn,
                endpointName: options.managedEndpointName,
                releaseLabel: options.emrOnEksVersion || EmrEksCluster.DEFAULT_EMR_VERSION,
                configurationOverrides: jsonConfigurationOverrides,
            },
        });
        cr.node.addDependency(this.eksCluster);
        return cr;
    }
    /**
   * Add new nodegroups to the cluster for Amazon EMR on EKS. This method overrides Amazon EKS nodegroup options then create the nodegroup.
   * If no subnet is provided, it creates one nodegroup per private subnet in the Amazon EKS Cluster.
   * If NVME local storage is used, the user_data is modified.
   * @param {string} id the CDK ID of the resource
   * @param {EmrEksNodegroupOptions} props the EmrEksNodegroupOptions [properties]{@link EmrEksNodegroupOptions}
   */
    addEmrEksNodegroup(id, props) {
        if (this.isKarpenter) {
            throw new Error(`You cannot use this method when the autoscaler is set to ${Autoscaler.KARPENTER}`);
        }
        // Get the subnet from Properties or one private subnet for each AZ
        const subnetList = props.subnet ? [props.subnet] : this.eksCluster.vpc.selectSubnets({
            onePerAz: true,
            subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
        }).subnets;
        // Add Amazon SSM agent to the user data
        var userData = [
            'yum install -y https://s3.amazonaws.com/ec2-downloads-windows/SSMAgent/latest/linux_amd64/amazon-ssm-agent.rpm',
            'systemctl enable amazon-ssm-agent',
            'systemctl start amazon-ssm-agent',
        ];
        var launchTemplateName = `EmrEksLaunch-${this.clusterName}`;
        // If the Nodegroup uses NVMe, add user data to configure them
        if (props.mountNvme) {
            userData = userData.concat([
                'INSTANCE_TYPE=$(ec2-metadata -t)',
                'if [[ $INSTANCE_TYPE == *"2xlarge"* ]]; then',
                'DEVICE="/dev/nvme1n1"',
                'mkfs.ext4 $DEVICE',
                'else',
                'yum install -y mdadm',
                'SSD_NVME_DEVICE_LIST=("/dev/nvme1n1" "/dev/nvme2n1")',
                'SSD_NVME_DEVICE_COUNT=${#SSD_NVME_DEVICE_LIST[@]}',
                'RAID_DEVICE=${RAID_DEVICE:-/dev/md0}',
                'RAID_CHUNK_SIZE=${RAID_CHUNK_SIZE:-512}  # Kilo Bytes',
                'FILESYSTEM_BLOCK_SIZE=${FILESYSTEM_BLOCK_SIZE:-4096}  # Bytes',
                'STRIDE=$((RAID_CHUNK_SIZE * 1024 / FILESYSTEM_BLOCK_SIZE))',
                'STRIPE_WIDTH=$((SSD_NVME_DEVICE_COUNT * STRIDE))',
                'mdadm --create --verbose "$RAID_DEVICE" --level=0 -c "${RAID_CHUNK_SIZE}" --raid-devices=${#SSD_NVME_DEVICE_LIST[@]} "${SSD_NVME_DEVICE_LIST[@]}"',
                'while [ -n "$(mdadm --detail "$RAID_DEVICE" | grep -ioE \'State :.*resyncing\')" ]; do',
                'echo "Raid is resyncing.."',
                'sleep 1',
                'done',
                'echo "Raid0 device $RAID_DEVICE has been created with disks ${SSD_NVME_DEVICE_LIST[*]}"',
                'mkfs.ext4 -m 0 -b "$FILESYSTEM_BLOCK_SIZE" -E "stride=$STRIDE,stripe-width=$STRIPE_WIDTH" "$RAID_DEVICE"',
                'DEVICE=$RAID_DEVICE',
                'fi',
                'systemctl stop docker',
                'mkdir -p /var/lib/kubelet/pods',
                'mount $DEVICE /var/lib/kubelet/pods',
                'chmod 750 /var/lib/docker',
                'systemctl start docker',
            ]);
            launchTemplateName = `EmrEksNvmeLaunch-${this.clusterName}`;
        }
        // Add headers and footers to user data
        const userDataMime = aws_cdk_lib_1.Fn.base64(`MIME-Version: 1.0
Content-Type: multipart/mixed; boundary="==MYBOUNDARY=="

--==MYBOUNDARY==
Content-Type: text/x-shellscript; charset="us-ascii"

#!/bin/bash
${userData.join('\r\n')}

--==MYBOUNDARY==--\\
`);
        // Create a new LaunchTemplate or reuse existing one
        const lt = singleton_launch_template_1.SingletonCfnLaunchTemplate.getOrCreate(this, launchTemplateName, userDataMime);
        // Create one Amazon EKS Nodegroup per subnet
        subnetList.forEach((subnet, index) => {
            // Make the ID unique across AZ using the index of subnet in the subnet list
            const resourceId = `${id}-${index}`;
            const nodegroupName = props.nodegroupName ? `${props.nodegroupName}-${index}` : resourceId;
            // Add the user data to the NodegroupOptions
            const nodeGroupParameters = {
                ...props,
                ...{
                    launchTemplateSpec: {
                        id: lt.ref,
                        version: lt.attrLatestVersionNumber,
                    },
                    subnets: {
                        subnets: [subnet],
                    },
                    nodegroupName: nodegroupName,
                },
            };
            // Create the Amazon EKS Nodegroup
            this.addNodegroupCapacity(resourceId, nodeGroupParameters);
        });
    }
    /**
     * Add a new Amazon EKS Nodegroup to the cluster.
     * This method is used to add a nodegroup to the Amazon EKS cluster and automatically set tags based on labels and taints
     *  so it can be used for the cluster autoscaler.
     * @param {string} nodegroupId the ID of the nodegroup
     * @param {EmrEksNodegroupOptions} options the EmrEksNodegroup [properties]{@link EmrEksNodegroupOptions}
     */
    addNodegroupCapacity(nodegroupId, options) {
        const nodegroup = this.eksCluster.addNodegroupCapacity(nodegroupId, options);
        // Adding the Amazon SSM policy
        nodegroup.role.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        // Add tags for the Cluster Autoscaler IAM scoping
        aws_cdk_lib_1.Tags.of(nodegroup).add('eks:cluster-name', `${this.clusterName}`);
        // Add tags for the Cluster Autoscaler management
        aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/enabled', 'true', {
            applyToLaunchedInstances: true,
        });
        // Add tag for the AZ
        if (options.subnets && options.subnets.subnets) {
            aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/node-template/label/topology.kubernetes.io/zone', options.subnets.subnets[0].availabilityZone, {
                applyToLaunchedInstances: true,
            });
        }
        // Add tag for the lifecycle type (spot or on-demand)
        aws_cdk_lib_1.Tags.of(nodegroup).add('k8s.io/cluster-autoscaler/node-template/label/eks.amazonaws.com/capacityType', (options.capacityType == aws_eks_1.CapacityType.SPOT) ? 'SPOT' : 'ON_DEMAND', {
            applyToLaunchedInstances: true,
        });
        // Iterate over labels and add appropriate tags
        if (options.labels) {
            for (const [key, value] of Object.entries(options.labels)) {
                aws_cdk_lib_1.Tags.of(nodegroup).add(`k8s.io/cluster-autoscaler/node-template/label/${key}`, value, {
                    applyToLaunchedInstances: true,
                });
                new aws_cdk_lib_1.CustomResource(this, `${nodegroupId}Label${key}`, {
                    serviceToken: this.nodegroupAsgTagsProviderServiceToken,
                    properties: {
                        nodegroupName: options.nodegroupName,
                        tagKey: `k8s.io/cluster-autoscaler/node-template/label/${key}`,
                        tagValue: value,
                    },
                }).node.addDependency(nodegroup);
            }
        }
        // Iterate over taints and add appropriate tags
        if (options.taints) {
            options.taints.forEach((taint) => {
                aws_cdk_lib_1.Tags.of(nodegroup).add(`k8s.io/cluster-autoscaler/node-template/taint/${taint.key}`, `${taint.value}:${taint.effect}`, {
                    applyToLaunchedInstances: true,
                });
                new aws_cdk_lib_1.CustomResource(this, `${nodegroupId}Taint${taint.key}`, {
                    serviceToken: this.nodegroupAsgTagsProviderServiceToken,
                    properties: {
                        nodegroupName: options.nodegroupName,
                        tagKey: `k8s.io/cluster-autoscaler/node-template/taint/${taint.key}`,
                        tagValue: `${taint.value}:${taint.effect}`,
                    },
                }).node.addDependency(nodegroup);
            });
        }
        return nodegroup;
    }
    /**
     * Create and configure a new Amazon IAM Role usable as an execution role.
     * This method makes the created role assumed by the Amazon EKS cluster Open ID Connect provider.
     * @param {Construct} scope of the IAM role
     * @param {string} id of the CDK resource to be created, it should be unique across the stack
     * @param {IManagedPolicy} policy the execution policy to attach to the role
     * @param {string} namespace The namespace from which the role is going to be used. MUST be the same as the namespace of the Virtual Cluster from which the job is submitted
     * @param {string} name Name to use for the role, required and is used to scope the iam role
     */
    createExecutionRole(scope, id, policy, namespace, name) {
        const stack = aws_cdk_lib_1.Stack.of(this);
        let irsaConditionkey = new aws_cdk_lib_1.CfnJson(this, `${id}irsaConditionkey'`, {
            value: {
                [`${this.eksCluster.openIdConnectProvider.openIdConnectProviderIssuer}:sub`]: 'system:serviceaccount:' + namespace + ':emr-containers-sa-*-*-' + aws_cdk_lib_1.Aws.ACCOUNT_ID.toString() + '-' + SimpleBase.base36.encode(name),
            },
        });
        // Create an execution role assumable by EKS OIDC provider
        return new aws_iam_1.Role(scope, `${id}ExecutionRole`, {
            assumedBy: new aws_iam_1.FederatedPrincipal(this.eksCluster.openIdConnectProvider.openIdConnectProviderArn, {
                StringLike: irsaConditionkey,
            }, 'sts:AssumeRoleWithWebIdentity'),
            roleName: name,
            managedPolicies: [policy],
            inlinePolicies: {
                PodTemplateAccess: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                's3:getObject',
                            ],
                            resources: [
                                stack.formatArn({
                                    region: '',
                                    account: '',
                                    service: 's3',
                                    resource: this.podTemplateLocation.bucketName,
                                    resourceName: `${this.podTemplateLocation.objectKey}/*`,
                                }),
                            ],
                        }),
                    ],
                }),
            },
        });
    }
    /**
     * Upload podTemplates to the Amazon S3 location used by the cluster.
     * @param {string} id the unique ID of the CDK resource
     * @param {string} filePath The local path of the yaml podTemplate files to upload
     */
    uploadPodTemplate(id, filePath) {
        new aws_s3_deployment_1.BucketDeployment(this, `${id}AssetDeployment`, {
            destinationBucket: this.assetBucket,
            destinationKeyPrefix: this.podTemplateLocation.objectKey,
            sources: [aws_s3_deployment_1.Source.asset(filePath)],
            role: this.assetUploadBucketRole,
        });
    }
    /**
     * Creates a new Amazon EMR on EKS job template based on the props passed
     * @param {Construct} scope the scope of the stack where job template is created
     * @param {string} id the CDK id for job template resource
     * @param {EmrEksJobTemplateDefinition} options the EmrManagedEndpointOptions to configure the Amazon EMR managed endpoint
     */
    addJobTemplate(scope, id, options) {
        // Create custom resource to execute the create job template boto3 call
        const cr = new aws_cdk_lib_1.CustomResource(scope, id, {
            serviceToken: this.jobTemplateProviderToken,
            properties: {
                name: options.name,
                jobTemplateData: options.jobTemplateData,
            },
        });
        cr.node.addDependency(this.eksCluster);
        return cr;
    }
    /**
     * Apply the provided manifest and add the CDK dependency on EKS cluster
     * @param {string} id the unique ID of the CDK resource
     * @param {any} manifest The manifest to apply.
     * You can use the Utils class that offers method to read yaml file and load it as a manifest
     */
    addKarpenterProvisioner(id, manifest) {
        if (!this.isKarpenter) {
            throw new Error(`You can\'t use this method when the autoscaler is set to ${Autoscaler.KARPENTER}`);
        }
        let manifestApply = this.eksCluster.addManifest(id, ...manifest);
        if (this.karpenterChart) {
            manifestApply.node.addDependency(this.karpenterChart);
        }
        return manifestApply;
    }
}
_a = JSII_RTTI_SYMBOL_1;
EmrEksCluster[_a] = { fqn: "aws-analytics-reference-architecture.EmrEksCluster", version: "2.12.6" };
EmrEksCluster.DEFAULT_EMR_VERSION = EmrVersion.V6_10;
EmrEksCluster.DEFAULT_EKS_VERSION = aws_eks_1.KubernetesVersion.V1_25;
EmrEksCluster.DEFAULT_CLUSTER_NAME = 'data-platform';
EmrEksCluster.DEFAULT_KARPENTER_VERSION = 'v0.20.0';
exports.EmrEksCluster = EmrEksCluster;
//# sourceMappingURL=data:application/json;base64,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