import { CustomResource } from 'aws-cdk-lib';
import { IVpc } from 'aws-cdk-lib/aws-ec2';
import { Cluster, KubernetesVersion, Nodegroup } from 'aws-cdk-lib/aws-eks';
import { CfnVirtualCluster } from 'aws-cdk-lib/aws-emrcontainers';
import { IManagedPolicy, Role } from 'aws-cdk-lib/aws-iam';
import { Bucket, Location } from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
import { TrackedConstruct } from '../common/tracked-construct';
import { EmrEksNodegroup, EmrEksNodegroupOptions } from './emr-eks-nodegroup';
import { EmrManagedEndpointOptions } from './emr-managed-endpoint';
import { EmrVirtualClusterOptions } from './emr-virtual-cluster';
import { ILayerVersion } from 'aws-cdk-lib/aws-lambda';
import { EmrEksJobTemplateDefinition } from './emr-eks-job-template';
/**
 * The different autoscaler available with EmrEksCluster
 */
export declare enum Autoscaler {
    KARPENTER = "KARPENTER",
    CLUSTER_AUTOSCALER = "CLUSTER_AUTOSCALER"
}
/**
 * The different EMR versions available on EKS
 */
export declare enum EmrVersion {
    V6_10 = "emr-6.10.0-latest",
    V6_9 = "emr-6.9.0-latest",
    V6_8 = "emr-6.8.0-latest",
    V6_7 = "emr-6.7.0-latest",
    V6_6 = "emr-6.6.0-latest",
    V6_5 = "emr-6.5.0-latest",
    V6_4 = "emr-6.4.0-latest",
    V6_3 = "emr-6.3.0-latest",
    V6_2 = "emr-6.2.0-latest",
    V5_33 = "emr-5.33.0-latest",
    V5_32 = "emr-5.32.0-latest"
}
/**
 * The properties for the EmrEksCluster Construct class.
 */
export interface EmrEksClusterProps {
    /**
     * Name of the Amazon EKS cluster to be created
     * @default -  The [default cluster name]{@link DEFAULT_CLUSTER_NAME}
     */
    readonly eksClusterName?: string;
    /**
     * The autoscaling mechanism to use
     */
    readonly autoscaling: Autoscaler;
    /**
     * Amazon IAM Role to be added to Amazon EKS master roles that will give access to kubernetes cluster from AWS console UI.
     * An admin role must be passed if `eksCluster` property is not set.
     * @default - No admin role is used and EKS cluster creation fails
     */
    readonly eksAdminRoleArn?: string;
    /**
     * The EKS cluster to setup EMR on. The cluster needs to be created in the same CDK Stack.
     * If the EKS cluster is provided, the cluster AddOns and all the controllers (Ingress controller, Cluster Autoscaler or Karpenter...) need to be configured.
     * When providing an EKS cluster, the methods for adding nodegroups can still be used. They implement the best practices for running Spark on EKS.
     * @default - An EKS Cluster is created
     */
    readonly eksCluster?: Cluster;
    /**
     * List of EmrEksNodegroup to create in the cluster in addition to the default [nodegroups]{@link EmrEksNodegroup}
     * @default -  Don't create additional nodegroups
     */
    readonly emrEksNodegroups?: EmrEksNodegroup[];
    /**
     * Kubernetes version for Amazon EKS cluster that will be created
     * @default -  Kubernetes v1.21 version is used
     */
    readonly kubernetesVersion?: KubernetesVersion;
    /**
     * If set to true, the Construct will create default EKS nodegroups or node provisioners (based on the autoscaler mechanism used).
     * There are three types of nodes:
     *  * Nodes for critical jobs which use on-demand instances, high speed disks and workload isolation
     *  * Nodes for shared worklaods which uses spot instances and no isolation to optimize costs
     *  * Nodes for notebooks which leverage a cost optimized configuration for running EMR managed endpoints and spark drivers/executors.
     * @default -  true
     */
    readonly defaultNodes?: boolean;
    /**
     * The version of karpenter to pass to Helm
     * @default - The [default Karpenter version]{@link DEFAULT_KARPENTER_VERSION}
     */
    readonly karpenterVersion?: string;
    /**
     * Starting k8s 1.22, CDK no longer bundle the kubectl layer with the code due to breaking npm package size.
     * A layer needs to be passed to the Construct.
     *
     * The cdk [documentation] (https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_eks.KubernetesVersion.html#static-v1_22)
     * contains the libraries that you should add for the right Kubernetes version
     * @default - No layer is used
     */
    readonly kubectlLambdaLayer?: ILayerVersion;
    /**
    * The CIDR of the VPC to use with EKS, if provided a VPC with three public subnets and three private subnet is create
    * The size of the private subnets is four time the one of the public subnet
    * @default - A vpc with the following CIDR 10.0.0.0/16 will be used
    */
    readonly vpcCidr?: string;
    /**
    * The VPC object where to deploy the EKS cluster
    * VPC should have at least two private and public subnets in different Availability Zones
    * All private subnets should have the following tags:
    * 'for-use-with-amazon-emr-managed-policies'='true'
    * 'kubernetes.io/role/internal-elb'='1'
    * All public subnets should have the following tag:
    * 'kubernetes.io/role/elb'='1'
    * Cannot be combined with vpcCidr, if combined vpcCidr takes precendency
    */
    readonly eksVpc?: IVpc;
}
/**
 * EmrEksCluster Construct packaging all the resources and configuration required to run Amazon EMR on EKS.
 * It deploys:
 * * An EKS cluster (VPC configuration can be customized)
 * * A tooling nodegroup to run tools including the Kubedashboard and the Cluster Autoscaler
 * * Optionally multiple nodegroups (one per AZ) for critical/shared/notebook EMR workloads
 * * Additional nodegroups can be configured
 *
 * The construct will upload on S3 the Pod templates required to run EMR jobs on the default nodegroups.
 * It will also parse and store the configuration of EMR on EKS jobs for each default nodegroup in object parameters
 *
 * Methods are available to add EMR Virtual Clusters to the EKS cluster and to create execution roles for the virtual clusters.
 *
 * Usage example:
 *
 * ```typescript
 * const emrEks: EmrEksCluster = EmrEksCluster.getOrCreate(stack, {
 *   eksAdminRoleArn: <ROLE_ARN>,
 *   eksClusterName: <CLUSTER_NAME>,
 * });
 *
 * const virtualCluster = emrEks.addEmrVirtualCluster(stack, {
 *   name: <Virtual_Cluster_Name>,
 *   createNamespace: <TRUE OR FALSE>,
 *   eksNamespace: <K8S_namespace>,
 * });
 *
 * const role = emrEks.createExecutionRole(stack, 'ExecRole',{
 *   policy: <POLICY>,
 * })
 *
 * // EMR on EKS virtual cluster ID
 * cdk.CfnOutput(self, 'VirtualClusterId',value = virtualCluster.attr_id)
 * // Job config for each nodegroup
 * cdk.CfnOutput(self, "CriticalConfig", value = emrEks.criticalDefaultConfig)
 * cdk.CfnOutput(self, "SharedConfig", value = emrEks.sharedDefaultConfig)
 * // Execution role arn
 * cdk.CfnOutput(self,'ExecRoleArn', value = role.roleArn)
 * ```
 *
 */
export declare class EmrEksCluster extends TrackedConstruct {
    /**
     * Get an existing EmrEksCluster based on the cluster name property or create a new one
     * only one EKS cluster can exist per stack
     * @param {Construct} scope the CDK scope used to search or create the cluster
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps} if created
     */
    static getOrCreate(scope: Construct, props: EmrEksClusterProps): EmrEksCluster;
    static readonly DEFAULT_EMR_VERSION = EmrVersion.V6_10;
    static readonly DEFAULT_EKS_VERSION: KubernetesVersion;
    static readonly DEFAULT_CLUSTER_NAME = "data-platform";
    static readonly DEFAULT_KARPENTER_VERSION = "v0.20.0";
    readonly eksCluster: Cluster;
    readonly notebookDefaultConfig: string;
    readonly criticalDefaultConfig: string;
    readonly sharedDefaultConfig: string;
    readonly podTemplateLocation: Location;
    readonly assetBucket: Bucket;
    readonly clusterName: string;
    readonly ec2InstanceNodeGroupRole: Role;
    private readonly managedEndpointProviderServiceToken;
    private readonly jobTemplateProviderToken;
    private readonly emrServiceRole;
    private readonly assetUploadBucketRole;
    private readonly karpenterChart?;
    private readonly isKarpenter;
    private readonly nodegroupAsgTagsProviderServiceToken;
    private readonly defaultNodes;
    /**
     * Constructs a new instance of the EmrEksCluster construct.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps}
     */
    private constructor();
    /**
     * Add a new Amazon EMR Virtual Cluster linked to Amazon EKS Cluster.
     * @param {Construct} scope of the stack where virtual cluster is deployed
     * @param {EmrVirtualClusterOptions} options the EmrVirtualClusterProps [properties]{@link EmrVirtualClusterProps}
     */
    addEmrVirtualCluster(scope: Construct, options: EmrVirtualClusterOptions): CfnVirtualCluster;
    /**
     * Creates a new Amazon EMR managed endpoint to be used with Amazon EMR Virtual Cluster .
     * CfnOutput can be customized.
     * @param {Construct} scope the scope of the stack where managed endpoint is deployed
     * @param {string} id the CDK id for endpoint
     * @param {EmrManagedEndpointOptions} options the EmrManagedEndpointOptions to configure the Amazon EMR managed endpoint
     */
    addManagedEndpoint(scope: Construct, id: string, options: EmrManagedEndpointOptions): CustomResource;
    /**
   * Add new nodegroups to the cluster for Amazon EMR on EKS. This method overrides Amazon EKS nodegroup options then create the nodegroup.
   * If no subnet is provided, it creates one nodegroup per private subnet in the Amazon EKS Cluster.
   * If NVME local storage is used, the user_data is modified.
   * @param {string} id the CDK ID of the resource
   * @param {EmrEksNodegroupOptions} props the EmrEksNodegroupOptions [properties]{@link EmrEksNodegroupOptions}
   */
    addEmrEksNodegroup(id: string, props: EmrEksNodegroupOptions): void;
    /**
     * Add a new Amazon EKS Nodegroup to the cluster.
     * This method is used to add a nodegroup to the Amazon EKS cluster and automatically set tags based on labels and taints
     *  so it can be used for the cluster autoscaler.
     * @param {string} nodegroupId the ID of the nodegroup
     * @param {EmrEksNodegroupOptions} options the EmrEksNodegroup [properties]{@link EmrEksNodegroupOptions}
     */
    addNodegroupCapacity(nodegroupId: string, options: EmrEksNodegroupOptions): Nodegroup;
    /**
     * Create and configure a new Amazon IAM Role usable as an execution role.
     * This method makes the created role assumed by the Amazon EKS cluster Open ID Connect provider.
     * @param {Construct} scope of the IAM role
     * @param {string} id of the CDK resource to be created, it should be unique across the stack
     * @param {IManagedPolicy} policy the execution policy to attach to the role
     * @param {string} namespace The namespace from which the role is going to be used. MUST be the same as the namespace of the Virtual Cluster from which the job is submitted
     * @param {string} name Name to use for the role, required and is used to scope the iam role
     */
    createExecutionRole(scope: Construct, id: string, policy: IManagedPolicy, namespace: string, name: string): Role;
    /**
     * Upload podTemplates to the Amazon S3 location used by the cluster.
     * @param {string} id the unique ID of the CDK resource
     * @param {string} filePath The local path of the yaml podTemplate files to upload
     */
    uploadPodTemplate(id: string, filePath: string): void;
    /**
     * Creates a new Amazon EMR on EKS job template based on the props passed
     * @param {Construct} scope the scope of the stack where job template is created
     * @param {string} id the CDK id for job template resource
     * @param {EmrEksJobTemplateDefinition} options the EmrManagedEndpointOptions to configure the Amazon EMR managed endpoint
     */
    addJobTemplate(scope: Construct, id: string, options: EmrEksJobTemplateDefinition): CustomResource;
    /**
     * Apply the provided manifest and add the CDK dependency on EKS cluster
     * @param {string} id the unique ID of the CDK resource
     * @param {any} manifest The manifest to apply.
     * You can use the Utils class that offers method to read yaml file and load it as a manifest
     */
    addKarpenterProvisioner(id: string, manifest: any): any;
}
