"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.clusterAutoscalerSetup = exports.karpenterSetup = exports.karpenterManifestSetup = exports.setDefaultKarpenterProvisioners = exports.setDefaultManagedNodeGroups = exports.eksClusterSetup = void 0;
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_sqs_1 = require("aws-cdk-lib/aws-sqs");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const utils_1 = require("../utils");
const emr_eks_nodegroup_1 = require("./emr-eks-nodegroup");
const emr_eks_cluster_1 = require("./emr-eks-cluster");
const IamPolicyAlb = require("./resources/k8s/iam-policy-alb.json");
const IamPolicyEbsCsiDriver = require("./resources/k8s/iam-policy-ebs-csi-driver.json");
/**
 * @internal
 * Upload podTemplates to the Amazon S3 location used by the cluster.
 * @param {Cluster} cluster the unique ID of the CDK resource
 * @param {Construct} scope The local path of the yaml podTemplate files to upload
 * @param {string} eksAdminRoleArn The admin role of the EKS cluster
 */
function eksClusterSetup(cluster, scope, eksAdminRoleArn) {
    // Add the provided Amazon IAM Role as Amazon EKS Admin
    if (eksAdminRoleArn != undefined) {
        cluster.eksCluster.awsAuth.addMastersRole(aws_iam_1.Role.fromRoleArn(scope, 'AdminRole', eksAdminRoleArn), 'AdminRole');
    }
    const ebsCsiDriverIrsa = cluster.eksCluster.addServiceAccount('ebsCSIDriverRoleSA', {
        name: 'ebs-csi-controller-sa',
        namespace: 'kube-system',
    });
    const ebsCsiDriverPolicyDocument = aws_iam_1.PolicyDocument.fromJson(IamPolicyEbsCsiDriver);
    const ebsCsiDriverPolicy = new aws_iam_1.Policy(scope, 'IamPolicyEbsCsiDriverIAMPolicy', { document: ebsCsiDriverPolicyDocument });
    ebsCsiDriverPolicy.attachToRole(ebsCsiDriverIrsa.role);
    const ebsCSIDriver = new aws_eks_1.CfnAddon(scope, 'ebsCsiDriver', {
        addonName: 'aws-ebs-csi-driver',
        clusterName: cluster.eksCluster.clusterName,
        serviceAccountRoleArn: ebsCsiDriverIrsa.role.roleArn,
        addonVersion: 'v1.18.0-eksbuild.1',
        resolveConflicts: "OVERWRITE"
    });
    ebsCSIDriver.node.addDependency(ebsCsiDriverIrsa);
    // Deploy the Helm Chart for the Certificate Manager. Required for EMR Studio ALB.
    const certManager = cluster.eksCluster.addHelmChart('CertManager', {
        createNamespace: true,
        namespace: 'cert-manager',
        chart: 'cert-manager',
        repository: 'https://charts.jetstack.io',
        version: '1.11.2',
        timeout: aws_cdk_lib_1.Duration.minutes(14),
        values: {
            startupapicheck: {
                timeout: '5m'
            },
            installCRDs: true
        }
    });
    //Create service account for ALB and install ALB
    const albPolicyDocument = aws_iam_1.PolicyDocument.fromJson(IamPolicyAlb);
    const albIAMPolicy = new aws_iam_1.Policy(scope, 'AWSLoadBalancerControllerIAMPolicy', { document: albPolicyDocument });
    const albServiceAccount = cluster.eksCluster.addServiceAccount('ALB', {
        name: 'aws-load-balancer-controller',
        namespace: 'kube-system',
    });
    albIAMPolicy.attachToRole(albServiceAccount.role);
    const albService = cluster.eksCluster.addHelmChart('ALB', {
        chart: 'aws-load-balancer-controller',
        repository: 'https://aws.github.io/eks-charts',
        namespace: 'kube-system',
        version: '1.5.2',
        timeout: aws_cdk_lib_1.Duration.minutes(14),
        values: {
            clusterName: cluster.clusterName,
            serviceAccount: {
                name: 'aws-load-balancer-controller',
                create: false,
            },
        },
    });
    albService.node.addDependency(albServiceAccount);
    albService.node.addDependency(certManager);
    // Add the kubernetes dashboard from helm chart
    cluster.eksCluster.addHelmChart('KubernetesDashboard', {
        createNamespace: true,
        namespace: 'kubernetes-dashboard',
        chart: 'kubernetes-dashboard',
        repository: 'https://kubernetes.github.io/dashboard/',
        version: 'v6.0.0',
        timeout: aws_cdk_lib_1.Duration.minutes(2),
        values: {
            fullnameOverride: 'kubernetes-dashboard',
            resources: {
                limits: {
                    memory: '600Mi',
                },
            },
        },
    });
    // Add the kubernetes dashboard service account
    cluster.eksCluster.addManifest('kubedashboard', {
        apiVersion: 'v1',
        kind: 'ServiceAccount',
        metadata: {
            name: 'eks-admin',
            namespace: 'kube-system',
        },
    });
    // Add the kubernetes dashboard cluster role binding
    cluster.eksCluster.addManifest('kubedashboardrolebinding', {
        apiVersion: 'rbac.authorization.k8s.io/v1',
        kind: 'ClusterRoleBinding',
        metadata: {
            name: 'eks-admin',
        },
        roleRef: {
            apiGroup: 'rbac.authorization.k8s.io',
            kind: 'ClusterRole',
            name: 'cluster-admin',
        },
        subjects: [
            {
                kind: 'ServiceAccount',
                name: 'eks-admin',
                namespace: 'kube-system',
            },
        ],
    });
    // Nodegroup capacity needed for all the tooling components including Karpenter
    let EmrEksNodeGroupTooling = { ...emr_eks_nodegroup_1.EmrEksNodegroup.TOOLING_ALL };
    EmrEksNodeGroupTooling.nodeRole = cluster.ec2InstanceNodeGroupRole;
    // Create the Amazon EKS Nodegroup for tooling
    cluster.addNodegroupCapacity('Tooling', EmrEksNodeGroupTooling);
    //IAM role created for the aws-node pod following AWS best practice not to use the EC2 instance role
    const awsNodeRole = new aws_iam_1.Role(scope, 'awsNodeRole', {
        assumedBy: new aws_iam_1.FederatedPrincipal(cluster.eksCluster.openIdConnectProvider.openIdConnectProviderArn, { ...[] }, 'sts:AssumeRoleWithWebIdentity'),
        roleName: `awsNodeRole-${cluster.clusterName}`,
        managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy')],
    });
    // update the aws-node service account with IAM role created for it
    new aws_eks_1.KubernetesManifest(scope, 'awsNodeServiceAccountUpdateManifest', {
        cluster: cluster.eksCluster,
        manifest: [
            {
                apiVersion: 'v1',
                kind: 'ServiceAccount',
                metadata: {
                    name: 'aws-node',
                    namespace: 'kube-system',
                    annotations: {
                        'eks.amazonaws.com/role-arn': awsNodeRole.roleArn,
                    },
                },
            }
        ],
        overwrite: true,
    });
    // Provide the Kubernetes Dashboard URL in AWS CloudFormation output
    new aws_cdk_lib_1.CfnOutput(scope, 'kubernetesDashboardURL', {
        description: 'Access Kubernetes Dashboard via kubectl proxy and this URL',
        value: 'http://localhost:8001/api/v1/namespaces/kubernetes-dashboard/services/https:kubernetes-dashboard:https/proxy/#/login',
    });
}
exports.eksClusterSetup = eksClusterSetup;
/**
 * @internal
 * Method to add the default EKS Managed Nodegroups configured for Spark workloads
 */
function setDefaultManagedNodeGroups(cluster) {
    let EmrEksNodeGroupCritical = { ...emr_eks_nodegroup_1.EmrEksNodegroup.CRITICAL_ALL };
    EmrEksNodeGroupCritical.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('criticalAll', EmrEksNodeGroupCritical);
    let EmrEksNodeGroupsharedDriver = { ...emr_eks_nodegroup_1.EmrEksNodegroup.SHARED_DRIVER };
    EmrEksNodeGroupsharedDriver.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('sharedDriver', EmrEksNodeGroupsharedDriver);
    let EmrEksNodeGroupsharedExecutor = { ...emr_eks_nodegroup_1.EmrEksNodegroup.SHARED_EXECUTOR };
    EmrEksNodeGroupsharedExecutor.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('sharedExecutor', EmrEksNodeGroupsharedExecutor);
    let EmrEksNodeGroupnotebookDriver = { ...emr_eks_nodegroup_1.EmrEksNodegroup.NOTEBOOK_DRIVER };
    EmrEksNodeGroupnotebookDriver.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('notebookDriver', EmrEksNodeGroupnotebookDriver);
    let EmrEksNodeGroupnotebookExecutor = { ...emr_eks_nodegroup_1.EmrEksNodegroup.NOTEBOOK_EXECUTOR };
    EmrEksNodeGroupnotebookExecutor.nodeRole = cluster.ec2InstanceNodeGroupRole;
    cluster.addEmrEksNodegroup('notebookExecutor', EmrEksNodeGroupnotebookExecutor);
}
exports.setDefaultManagedNodeGroups = setDefaultManagedNodeGroups;
/**
 * @internal
 * Method to add the default Karpenter provisioners for Spark workloads
 */
function setDefaultKarpenterProvisioners(cluster) {
    const subnets = cluster.eksCluster.vpc.selectSubnets({
        onePerAz: true,
        subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
    }).subnets;
    subnets.forEach((subnet, index) => {
        let criticalManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/critical-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterCriticalManifest-${index}`, criticalManfifestYAML);
        let sharedDriverManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/shared-driver-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterSharedDriverManifest-${index}`, sharedDriverManfifestYAML);
        let sharedExecutorManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/shared-executor-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterSharedExecutorManifest-${index}`, sharedExecutorManfifestYAML);
        let notebookDriverManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/notebook-driver-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterNotebookDriverManifest-${index}`, notebookDriverManfifestYAML);
        let notebookExecutorManfifestYAML = karpenterManifestSetup(cluster.clusterName, `${__dirname}/resources/k8s/karpenter-provisioner-config/notebook-executor-provisioner.yml`, subnet);
        cluster.addKarpenterProvisioner(`karpenterNotebookExecutorManifest-${index}`, notebookExecutorManfifestYAML);
    });
}
exports.setDefaultKarpenterProvisioners = setDefaultKarpenterProvisioners;
/**
 * @internal
 * Method to generate the Karpenter manifests from templates and targeted to the specific EKS cluster
 */
function karpenterManifestSetup(clusterName, path, subnet) {
    let manifest = utils_1.Utils.readYamlDocument(path);
    manifest = manifest.replace('{{subnet-id}}', subnet.subnetId);
    manifest = manifest.replace(/(\{{az}})/g, subnet.availabilityZone);
    manifest = manifest.replace('{{cluster-name}}', clusterName);
    let manfifestYAML = manifest.split("---").map((e) => utils_1.Utils.loadYaml(e));
    return manfifestYAML;
}
exports.karpenterManifestSetup = karpenterManifestSetup;
/**
 * @internal
 * Install all the required configurations of Karpenter SQS and Event rules to handle spot and unhealthy instance termination
 * Create a security group to be used by nodes created with karpenter
 * Tags the subnets and VPC to be used by karpenter
 * create a tooling provisioner that will deploy in each of the AZs, one per AZ
 */
function karpenterSetup(cluster, eksClusterName, scope, karpenterVersion) {
    const karpenterInterruptionQueue = new aws_sqs_1.Queue(scope, 'karpenterInterruptionQueue', {
        queueName: eksClusterName,
        retentionPeriod: aws_cdk_lib_1.Duration.seconds(300)
    });
    karpenterInterruptionQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: ['sqs:SendMessage'],
        principals: [new aws_iam_1.ServicePrincipal('sqs.amazonaws.com'), new aws_iam_1.ServicePrincipal('events.amazonaws.com')]
    }));
    new aws_events_1.Rule(scope, 'scheduledChangeRule', {
        eventPattern: {
            source: ['aws.heatlh'],
            detail: ['AWS Health Event']
        },
        targets: [new aws_events_targets_1.SqsQueue(karpenterInterruptionQueue)]
    });
    new aws_events_1.Rule(scope, 'instanceStateChangeRule', {
        eventPattern: {
            source: ['aws.ec2'],
            detail: ['EC2 Instance State-change Notification']
        },
        targets: [new aws_events_targets_1.SqsQueue(karpenterInterruptionQueue)]
    });
    const karpenterNodeRole = new aws_iam_1.Role(cluster, 'karpenter-node-role', {
        assumedBy: new aws_iam_1.ServicePrincipal(`ec2.${cluster.stack.urlSuffix}`),
        managedPolicies: [
            aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'),
            aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'),
            aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'),
            aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
        ],
        roleName: `KarpenterNodeRole-${eksClusterName}`,
    });
    const karpenterControllerPolicyStatementSSM = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: ['ssm:GetParameter', 'pricing:GetProducts'],
        resources: ['*'],
    });
    const karpenterControllerPolicyStatementEC2 = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: [
            'ec2:CreateLaunchTemplate',
            'ec2:DeleteLaunchTemplate',
            'ec2:CreateFleet',
            'ec2:RunInstances',
            'ec2:CreateTags',
            'ec2:TerminateInstances',
            'ec2:DescribeLaunchTemplates',
            'ec2:DescribeInstances',
            'ec2:DescribeSecurityGroups',
            'ec2:DescribeSubnets',
            'ec2:DescribeInstanceTypes',
            'ec2:DescribeInstanceTypeOfferings',
            'ec2:DescribeAvailabilityZones',
        ],
        resources: ['*'],
    });
    const karpenterControllerPolicyStatementIAM = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: ['iam:PassRole'],
        resources: [`arn:aws:iam::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:role/KarpenterNodeRole-${eksClusterName}`],
    });
    const karpenterInstanceProfile = new aws_iam_1.CfnInstanceProfile(cluster, 'karpenter-instance-profile', {
        roles: [karpenterNodeRole.roleName],
        instanceProfileName: `karpenterNodeInstanceProfile-${eksClusterName}`,
        path: '/'
    });
    cluster.awsAuth.addRoleMapping(karpenterNodeRole, {
        username: 'system:node:{{EC2PrivateDNSName}}',
        groups: ['system:bootstrappers', 'system:nodes'],
    });
    const karpenterNS = cluster.addManifest('karpenterNS', {
        apiVersion: 'v1',
        kind: 'Namespace',
        metadata: { name: 'karpenter' },
    });
    const karpenterAccount = cluster.addServiceAccount('Karpenter', {
        name: 'karpenter',
        namespace: 'karpenter',
    });
    karpenterAccount.node.addDependency(karpenterNS);
    karpenterAccount.addToPrincipalPolicy(karpenterControllerPolicyStatementSSM);
    karpenterAccount.addToPrincipalPolicy(karpenterControllerPolicyStatementEC2);
    karpenterAccount.addToPrincipalPolicy(karpenterControllerPolicyStatementIAM);
    //Deploy Karpenter Chart
    const karpenterChart = cluster.addHelmChart('Karpenter', {
        chart: 'karpenter',
        release: 'karpenter',
        repository: 'oci://public.ecr.aws/karpenter/karpenter',
        namespace: 'karpenter',
        version: karpenterVersion || emr_eks_cluster_1.EmrEksCluster.DEFAULT_KARPENTER_VERSION,
        timeout: aws_cdk_lib_1.Duration.minutes(14),
        wait: true,
        values: {
            serviceAccount: {
                name: 'karpenter',
                create: false,
                annotations: {
                    'eks.amazonaws.com/role-arn': karpenterAccount.role.roleArn,
                },
            },
            settings: {
                aws: {
                    defaultInstanceProfile: karpenterInstanceProfile.instanceProfileName,
                    clusterName: eksClusterName,
                    clusterEndpoint: cluster.clusterEndpoint,
                    interruptionQueueName: karpenterInterruptionQueue.queueName
                },
            }
        },
    });
    karpenterChart.node.addDependency(karpenterAccount);
    const karpenterInstancesSg = new aws_ec2_1.SecurityGroup(scope, 'karpenterSg', {
        vpc: cluster.vpc,
        allowAllOutbound: true,
        description: 'security group for a karpenter instances',
        securityGroupName: 'karpenterSg',
        disableInlineRules: true,
    });
    aws_cdk_lib_1.Tags.of(karpenterInstancesSg).add('karpenter.sh/discovery', `${eksClusterName}`);
    cluster.clusterSecurityGroup.addIngressRule(karpenterInstancesSg, aws_ec2_1.Port.allTraffic());
    karpenterInstancesSg.addIngressRule(karpenterInstancesSg, aws_ec2_1.Port.allTraffic());
    karpenterInstancesSg.addIngressRule(cluster.clusterSecurityGroup, aws_ec2_1.Port.allTraffic());
    aws_cdk_lib_1.Tags.of(cluster.vpc).add('karpenter.sh/discovery', eksClusterName);
    cluster.vpc.privateSubnets.forEach((subnet) => {
        aws_cdk_lib_1.Tags.of(subnet).add('karpenter.sh/discovery', eksClusterName);
    });
    cluster.vpc.publicSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('karpenter.sh/discovery', eksClusterName));
    const privateSubnets = cluster.vpc.selectSubnets({
        onePerAz: true,
        subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
    }).subnets;
    let listPrivateSubnets = privateSubnets.map(subnet => subnet.subnetId);
    let manifest = utils_1.Utils.readYamlDocument(`${__dirname}/resources/k8s/karpenter-provisioner-config/tooling-provisioner.yml`);
    manifest = manifest.replace(/(\{{cluster-name}})/g, eksClusterName);
    manifest = manifest.replace(/(\{{subnet-list}})/g, listPrivateSubnets.join(','));
    let manfifestYAML = manifest.split("---").map((e) => utils_1.Utils.loadYaml(e));
    const manifestApply = cluster.addManifest('provisioner-tooling', ...manfifestYAML);
    manifestApply.node.addDependency(karpenterChart);
    return karpenterChart;
}
exports.karpenterSetup = karpenterSetup;
/**
 * @internal
 * Deploy the cluster autoscaler controller in the k8s cluster
 */
function clusterAutoscalerSetup(cluster, eksClusterName, scope, k8sVersion) {
    //Version of the autoscaler, controls the image tag
    const versionMap = new Map([
        [aws_eks_1.KubernetesVersion.V1_25, "9.25.0"],
        [aws_eks_1.KubernetesVersion.V1_24, "9.25.0"],
        [aws_eks_1.KubernetesVersion.V1_23, "9.21.0"],
        [aws_eks_1.KubernetesVersion.V1_22, "9.13.1"]
    ]);
    // Create a Kubernetes Service Account for the Cluster Autoscaler with Amazon IAM Role
    const AutoscalerServiceAccount = cluster.addServiceAccount('Autoscaler', {
        name: 'cluster-autoscaler',
        namespace: 'kube-system',
    });
    //Iam policy attached to the Role used by k8s autoscaller
    let autoscalingPolicyDescribe = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: [
            'autoscaling:DescribeAutoScalingGroups',
            'autoscaling:DescribeAutoScalingInstances',
            'autoscaling:DescribeLaunchConfigurations',
            'autoscaling:DescribeTags',
            'ec2:DescribeLaunchTemplateVersions',
            'eks:DescribeNodegroup',
        ],
        resources: ['*'],
    });
    let autoscalingPolicyMutateGroup = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: [
            'autoscaling:SetDesiredCapacity',
            'autoscaling:TerminateInstanceInAutoScalingGroup',
        ],
        resources: ['*'],
        conditions: {
            StringEquals: {
                'aws:ResourceTag/eks:cluster-name': eksClusterName,
            },
        },
    });
    // Add the right Amazon IAM Policy to the Amazon IAM Role for the Cluster Autoscaler
    AutoscalerServiceAccount.addToPrincipalPolicy(autoscalingPolicyDescribe);
    AutoscalerServiceAccount.addToPrincipalPolicy(autoscalingPolicyMutateGroup);
    cluster.addHelmChart('AutoScaler', {
        chart: 'cluster-autoscaler',
        repository: 'https://kubernetes.github.io/autoscaler',
        version: versionMap.get(k8sVersion),
        namespace: 'kube-system',
        timeout: aws_cdk_lib_1.Duration.minutes(14),
        values: {
            cloudProvider: 'aws',
            awsRegion: aws_cdk_lib_1.Stack.of(scope).region,
            autoDiscovery: { clusterName: eksClusterName },
            rbac: {
                serviceAccount: {
                    name: 'cluster-autoscaler',
                    create: false,
                },
            },
            extraArgs: {
                'skip-nodes-with-local-storage': false,
                'scan-interval': '5s',
                'expander': 'least-waste',
                'balance-similar-node-groups': true,
                'skip-nodes-with-system-pods': false,
            },
        },
    });
}
exports.clusterAutoscalerSetup = clusterAutoscalerSetup;
//# sourceMappingURL=data:application/json;base64,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