"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.emrOnEksImageMap = exports.EmrEksImageBuilderCRProviderSetup = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const pre_bundled_layer_1 = require("../common/pre-bundled-layer");
/**
 * @internal
 * Create a Customer Resource to start a codebuild project
 * The policy allow access only to a single bucket to store notebooks
 * @returns Return the token to the Provider for CR
 */
function EmrEksImageBuilderCRProviderSetup(scope, codeBuildProjectArn) {
    //The policy allowing the creatio of the job template
    const lambdaPolicy = [
        new aws_iam_1.PolicyStatement({
            resources: [codeBuildProjectArn],
            actions: ['codebuild:BatchGetBuilds', 'codebuild:StartBuild'],
        })
    ];
    // AWS Lambda function supporting the start a codebuild project
    const onEvent = new pre_bundled_function_1.PreBundledFunction(scope, 'OnEvent', {
        codePath: 'docker-builder/resources/lambdas',
        runtime: aws_lambda_1.Runtime.PYTHON_3_10,
        handler: 'lambda.on_event',
        layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
        lambdaPolicyStatements: lambdaPolicy,
        logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        timeout: aws_cdk_lib_1.Duration.seconds(120),
    });
    // AWS Lambda function that check the status of codebuild porject
    const isComplete = new pre_bundled_function_1.PreBundledFunction(scope, 'IsComplete', {
        codePath: 'docker-builder/resources/lambdas',
        handler: 'lambda.is_complete',
        layers: [pre_bundled_layer_1.PreBundledLayer.getOrCreate(scope, 'common/resources/lambdas/pre-bundled-layer')],
        lambdaPolicyStatements: lambdaPolicy,
        runtime: aws_lambda_1.Runtime.PYTHON_3_10,
        logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        timeout: aws_cdk_lib_1.Duration.seconds(120),
    });
    const provider = new custom_resources_1.Provider(scope, 'CustomResourceProvider', {
        onEventHandler: onEvent,
        isCompleteHandler: isComplete,
        totalTimeout: aws_cdk_lib_1.Duration.minutes(15),
        queryInterval: aws_cdk_lib_1.Duration.seconds(20),
        providerFunctionName: 'dockerBuildPublishFn',
    });
    return provider.serviceToken;
}
exports.EmrEksImageBuilderCRProviderSetup = EmrEksImageBuilderCRProviderSetup;
/**
 * @internal
 * a Map containing the account for each region where the docker image are stored
 * The list is maintained in this AWS documentation (link)[https://docs.aws.amazon.com/emr/latest/EMR-on-EKS-DevelopmentGuide/docker-custom-images-tag.html]
 */
exports.emrOnEksImageMap = new Map([
    ['ap-northeast-1', "059004520145"],
    ['ap-northeast-2', "996579266876"],
    ['ap-south-1', "235914868574"],
    ['ap-southeast-1', "671219180197"],
    ['ap-southeast-2', "038297999601"],
    ['ca-central-1', "351826393999"],
    ['eu-central-1', "107292555468"],
    ['eu-north-1', "830386416364"],
    ['eu-west-1', "483788554619"],
    ['eu-west-2', "118780647275"],
    ['eu-west-3', "307523725174"],
    ['sa-east-1', "052806832358"],
    ['us-east-1', "755674844232"],
    ['us-east-2', "711395599931"],
    ['us-west-1', "608033475327"],
    ['us-west-2', "895885662937"],
]);
//# sourceMappingURL=data:application/json;base64,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