"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmrEksImageBuilder = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_ecr_1 = require("aws-cdk-lib/aws-ecr");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const constructs_1 = require("constructs");
const ara_bucket_1 = require("../ara-bucket");
const utils_1 = require("../utils");
const docker_builder_util_1 = require("./docker-builder-util");
/**
 * A CDK construct to create build and publish EMR on EKS custom image
 * The construct will create an ECR repository to publish the images
 * It provide a method {@link publishImage} to build a docker file and publish it to the ECR repository
 *
 * Resources deployed:
 *
 * * Multiple Session Policies that are used to map an EMR Studio user or group to a set of resources they are allowed to access. These resources are:
 *   * ECR Repository
 *   * Codebuild project
 *   * A custom resource to build and publish a custom EMR on EKS image
 *
 *
 * Usage example:
 *
 * ```typescript
 *
 * const app = new App();
 *
 * const account = process.env.CDK_DEFAULT_ACCOUNT;
 * const region = process.env.CDK_DEFAULT_REGION;
 *
 * const stack = new Stack(app, 'EmrEksImageBuilderStack', {
 * env: { account: account, region: region },
 * });
 *
 * const publish = new EmrEksImageBuilder(stack, 'EmrEksImageBuilder', {
 *  repositoryName: 'my-repo',
 *  ecrRemovalPolicy: RemovalPolicy.RETAIN
 * });
 *
 * publish.publishImage('PATH-TO-DOCKER-FILE-FOLDER', 'v4');
 *
 * ```
 */
class EmrEksImageBuilder extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.assetBucket = ara_bucket_1.AraBucket.getOrCreate(this, { bucketName: `${utils_1.Utils.stringSanitizer(aws_cdk_lib_1.Stack.of(this).stackName)}-ara-docker-assets`, encryption: aws_s3_1.BucketEncryption.KMS_MANAGED });
        let codeBuildRole = new aws_iam_1.Role(this, 'codebuildarn', {
            assumedBy: new aws_iam_1.ServicePrincipal('codebuild.amazonaws.com'),
        });
        const ecrRepo = new aws_ecr_1.Repository(this, `ecr-${props.repositoryName}`, {
            repositoryName: props.repositoryName,
            removalPolicy: props.ecrRemovalPolicy ? props.ecrRemovalPolicy : aws_cdk_lib_1.RemovalPolicy.RETAIN,
            imageScanOnPush: true
        });
        this.ecrURI = ecrRepo.repositoryUri;
        this.ecrName = props.repositoryName;
        let dockerImageAccount = docker_builder_util_1.emrOnEksImageMap.get(aws_cdk_lib_1.Stack.of(this).region);
        if (dockerImageAccount === undefined)
            throw new Error('Docker Image is not available in the selected region');
        let commands = [
            'echo logging into docker',
            `aws ecr get-login-password --region ${aws_cdk_lib_1.Aws.REGION} | docker login --username AWS --password-stdin ${dockerImageAccount}.dkr.ecr.${aws_cdk_lib_1.Aws.REGION}.amazonaws.com`,
            'echo Build start',
            'echo $DOCKER_FILE_S3_PATH',
            'aws s3 cp $DOCKER_FILE_S3_PATH Dockerfile',
            'docker build -t local .',
            'docker logout',
            'echo $ecrURI',
            `aws ecr get-login-password --region ${aws_cdk_lib_1.Aws.REGION} | docker login --username AWS --password-stdin ${this.ecrURI}`,
            'docker tag local $ecrURI:$tag',
            'docker push $ecrURI:$tag',
            'docker logout'
        ];
        const codebuildProject = new aws_codebuild_1.Project(this, `DockerImageDeployProject-${props.repositoryName}`, {
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    build: {
                        commands: commands,
                    },
                },
            }),
            environment: {
                privileged: true,
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
                computeType: aws_codebuild_1.ComputeType.SMALL
            },
            role: codeBuildRole,
        });
        ecrRepo.grantPullPush(codeBuildRole);
        this.assetBucket.grantRead(codeBuildRole);
        codeBuildRole.addToPolicy(new aws_iam_1.PolicyStatement({
            resources: [`arn:aws:ecr:${aws_cdk_lib_1.Aws.REGION}:${dockerImageAccount}:repository/*`],
            actions: [
                'ecr:BatchGetImage',
                'ecr:GetAuthorizationToken',
                'ecr:BatchCheckLayerAvailability',
                'ecr:GetDownloadUrlForLayer',
            ]
        }));
        this.codebuildProjectName = codebuildProject.projectName;
        this.dockerBuildPublishCrToken = (0, docker_builder_util_1.EmrEksImageBuilderCRProviderSetup)(this, codebuildProject.projectArn);
    }
    /**
     * A method to build and publish the custom image from a Dockerfile
     * The method invoke the custom resource deployed by the construct
     * and publish the **URI** of the published custom image as Cloudformation output
     * @param {string} dockerfilePath Path to the folder for Dockerfile
     * @param {string} tag The tag used to publish to the ECR repository
     */
    publishImage(dockerfilePath, tag) {
        new aws_s3_deployment_1.BucketDeployment(this, `DockerfileAssetDeployment-${tag}`, {
            destinationBucket: this.assetBucket,
            destinationKeyPrefix: `${this.ecrName}/${tag}`,
            sources: [aws_s3_deployment_1.Source.asset(dockerfilePath)],
        });
        const containerImage = new aws_cdk_lib_1.CustomResource(this, `EmrEksImageBuild-${tag}`, {
            serviceToken: this.dockerBuildPublishCrToken,
            properties: {
                s3Path: `s3://${this.assetBucket.bucketName}/${this.ecrName}/${tag}/Dockerfile`,
                tag: tag,
                ecrURI: this.ecrURI,
                codebuildProjectName: this.codebuildProjectName,
                ecrName: this.ecrName,
            },
        });
        new aws_cdk_lib_1.CfnOutput(this, `URI-${tag}`, {
            value: containerImage.getAttString('ContainerUri'),
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
EmrEksImageBuilder[_a] = { fqn: "aws-analytics-reference-architecture.EmrEksImageBuilder", version: "2.12.6" };
exports.EmrEksImageBuilder = EmrEksImageBuilder;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZG9ja2VyLWJ1aWxkLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vc3JjL2RvY2tlci1idWlsZGVyL2RvY2tlci1idWlsZC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDZDQUFtRjtBQUNuRiw2REFBNkY7QUFDN0YsaURBQWlEO0FBQ2pELGlEQUE4RTtBQUM5RSwrQ0FBc0Q7QUFDdEQscUVBQXlFO0FBQ3pFLDJDQUF1QztBQUN2Qyw4Q0FBMEM7QUFDMUMsb0NBQWlDO0FBQ2pDLCtEQUE0RjtBQW9CNUY7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0FrQ0c7QUFFSCxNQUFhLGtCQUFtQixTQUFRLHNCQUFTO0lBUS9DLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBOEI7UUFFdEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsV0FBVyxHQUFHLHNCQUFTLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxFQUFFLFVBQVUsRUFBRSxHQUFHLGFBQUssQ0FBQyxlQUFlLENBQUMsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDLG9CQUFvQixFQUFFLFVBQVUsRUFBRSx5QkFBZ0IsQ0FBQyxXQUFXLEVBQUUsQ0FBQyxDQUFDO1FBRWpMLElBQUksYUFBYSxHQUFHLElBQUksY0FBSSxDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7WUFDakQsU0FBUyxFQUFFLElBQUksMEJBQWdCLENBQUMseUJBQXlCLENBQUM7U0FDM0QsQ0FBQyxDQUFDO1FBRUgsTUFBTSxPQUFPLEdBQWUsSUFBSSxvQkFBVSxDQUFDLElBQUksRUFBRSxPQUFPLEtBQUssQ0FBQyxjQUFjLEVBQUUsRUFBRTtZQUM5RSxjQUFjLEVBQUUsS0FBSyxDQUFDLGNBQWM7WUFDcEMsYUFBYSxFQUFFLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQywyQkFBYSxDQUFDLE1BQU07WUFDckYsZUFBZSxFQUFFLElBQUk7U0FDdEIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLE1BQU0sR0FBRyxPQUFPLENBQUMsYUFBYSxDQUFDO1FBQ3BDLElBQUksQ0FBQyxPQUFPLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQztRQUVwQyxJQUFJLGtCQUFrQixHQUF1QixzQ0FBZ0IsQ0FBQyxHQUFHLENBQUMsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUM7UUFFekYsSUFBSSxrQkFBa0IsS0FBSyxTQUFTO1lBQ2xDLE1BQU0sSUFBSSxLQUFLLENBQUMsc0RBQXNELENBQUMsQ0FBQztRQUUxRSxJQUFJLFFBQVEsR0FBRztZQUNiLDBCQUEwQjtZQUMxQix1Q0FBdUMsaUJBQUcsQ0FBQyxNQUFNLG1EQUFtRCxrQkFBbUIsWUFBWSxpQkFBRyxDQUFDLE1BQU0sZ0JBQWdCO1lBQzdKLGtCQUFrQjtZQUNsQiwyQkFBMkI7WUFDM0IsMkNBQTJDO1lBQzNDLHlCQUF5QjtZQUN6QixlQUFlO1lBQ2YsY0FBYztZQUNkLHVDQUF1QyxpQkFBRyxDQUFDLE1BQU0sbURBQW1ELElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDakgsK0JBQStCO1lBQy9CLDBCQUEwQjtZQUMxQixlQUFlO1NBQ2hCLENBQUM7UUFFRixNQUFNLGdCQUFnQixHQUFHLElBQUksdUJBQU8sQ0FBQyxJQUFJLEVBQUUsNEJBQTRCLEtBQUssQ0FBQyxjQUFjLEVBQUUsRUFBRTtZQUM3RixTQUFTLEVBQUUseUJBQVMsQ0FBQyxVQUFVLENBQUM7Z0JBQzlCLE9BQU8sRUFBRSxLQUFLO2dCQUNkLE1BQU0sRUFBRTtvQkFDTixLQUFLLEVBQUU7d0JBQ0wsUUFBUSxFQUFFLFFBQVE7cUJBQ25CO2lCQUNGO2FBQ0YsQ0FBQztZQUNGLFdBQVcsRUFBRTtnQkFDWCxVQUFVLEVBQUUsSUFBSTtnQkFDaEIsVUFBVSxFQUFFLCtCQUFlLENBQUMsWUFBWTtnQkFDeEMsV0FBVyxFQUFFLDJCQUFXLENBQUMsS0FBSzthQUMvQjtZQUNELElBQUksRUFBRSxhQUFhO1NBQ3BCLENBQUMsQ0FBQztRQUVILE9BQU8sQ0FBQyxhQUFhLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDckMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFMUMsYUFBYSxDQUFDLFdBQVcsQ0FBQyxJQUFJLHlCQUFlLENBQUU7WUFDN0MsU0FBUyxFQUFFLENBQUMsZUFBZSxpQkFBRyxDQUFDLE1BQU0sSUFBSSxrQkFBa0IsZUFBZSxDQUFDO1lBQzNFLE9BQU8sRUFBRTtnQkFDUCxtQkFBbUI7Z0JBQ25CLDJCQUEyQjtnQkFDM0IsaUNBQWlDO2dCQUNqQyw0QkFBNEI7YUFDN0I7U0FDRixDQUFDLENBQUMsQ0FBQztRQUVKLElBQUksQ0FBQyxvQkFBb0IsR0FBRyxnQkFBZ0IsQ0FBQyxXQUFXLENBQUM7UUFFekQsSUFBSSxDQUFDLHlCQUF5QixHQUFHLElBQUEsdURBQWlDLEVBQUMsSUFBSSxFQUFFLGdCQUFnQixDQUFDLFVBQVUsQ0FBQyxDQUFDO0lBQ3hHLENBQUM7SUFFRDs7Ozs7O09BTUc7SUFFSSxZQUFZLENBQUMsY0FBc0IsRUFBRSxHQUFXO1FBRXJELElBQUksb0NBQWdCLENBQUMsSUFBSSxFQUFFLDZCQUE2QixHQUFHLEVBQUUsRUFBRTtZQUM3RCxpQkFBaUIsRUFBRSxJQUFJLENBQUMsV0FBVztZQUNuQyxvQkFBb0IsRUFBRSxHQUFHLElBQUksQ0FBQyxPQUFPLElBQUksR0FBRyxFQUFFO1lBQzlDLE9BQU8sRUFBRSxDQUFDLDBCQUFNLENBQUMsS0FBSyxDQUFDLGNBQWMsQ0FBQyxDQUFDO1NBQ3hDLENBQUMsQ0FBQztRQUVILE1BQU0sY0FBYyxHQUFJLElBQUksNEJBQWMsQ0FBQyxJQUFJLEVBQUUsb0JBQW9CLEdBQUcsRUFBRSxFQUFFO1lBQzFFLFlBQVksRUFBRSxJQUFJLENBQUMseUJBQXlCO1lBQzVDLFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsUUFBUSxJQUFJLENBQUMsV0FBVyxDQUFDLFVBQVUsSUFBSSxJQUFJLENBQUMsT0FBTyxJQUFJLEdBQUcsYUFBYTtnQkFDL0UsR0FBRyxFQUFFLEdBQUc7Z0JBQ1IsTUFBTSxFQUFFLElBQUksQ0FBQyxNQUFNO2dCQUNuQixvQkFBb0IsRUFBRSxJQUFJLENBQUMsb0JBQW9CO2dCQUMvQyxPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87YUFDdEI7U0FDRixDQUFDLENBQUM7UUFFSCxJQUFJLHVCQUFTLENBQUMsSUFBSSxFQUFFLE9BQU8sR0FBRyxFQUFFLEVBQUU7WUFDaEMsS0FBSyxFQUFFLGNBQWMsQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDO1NBQ25ELENBQUMsQ0FBQTtJQUVKLENBQUM7Ozs7QUFqSFUsZ0RBQWtCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQXdzLCBDZm5PdXRwdXQsIEN1c3RvbVJlc291cmNlLCBSZW1vdmFsUG9saWN5LCBTdGFjayB9IGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuaW1wb3J0IHsgQnVpbGRTcGVjLCBDb21wdXRlVHlwZSwgTGludXhCdWlsZEltYWdlLCBQcm9qZWN0IH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1jb2RlYnVpbGRcIjtcbmltcG9ydCB7IFJlcG9zaXRvcnkgfSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLWVjclwiO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBSb2xlLCBTZXJ2aWNlUHJpbmNpcGFsIH0gZnJvbSBcImF3cy1jZGstbGliL2F3cy1pYW1cIjtcbmltcG9ydCB7IEJ1Y2tldEVuY3J5cHRpb24gfSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLXMzXCI7XG5pbXBvcnQgeyBCdWNrZXREZXBsb3ltZW50LCBTb3VyY2UgfSBmcm9tIFwiYXdzLWNkay1saWIvYXdzLXMzLWRlcGxveW1lbnRcIjtcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gXCJjb25zdHJ1Y3RzXCI7XG5pbXBvcnQgeyBBcmFCdWNrZXQgfSBmcm9tIFwiLi4vYXJhLWJ1Y2tldFwiO1xuaW1wb3J0IHsgVXRpbHMgfSBmcm9tIFwiLi4vdXRpbHNcIjtcbmltcG9ydCB7IEVtckVrc0ltYWdlQnVpbGRlckNSUHJvdmlkZXJTZXR1cCwgZW1yT25Fa3NJbWFnZU1hcCB9IGZyb20gXCIuL2RvY2tlci1idWlsZGVyLXV0aWxcIjtcblxuXG4vKipcbiAqIFRoZSBwcm9wZXJ0aWVzIGZvciBpbml0aWFsaXppbmcgdGhlIGNvbnN0cnVjdCB0byBidWlsZCBjdXN0b20gRU1SIG9uIEVLUyBpbWFnZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIEVtckVrc0ltYWdlQnVpbGRlclByb3BzIHtcbiAgLyoqXG4gICAqIFJlcXVpcmVkXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBFQ1IgcmVwb3NpdG9yeSB0byBjcmVhdGVcbiAgICogKi9cbiAgcmVhZG9ubHkgcmVwb3NpdG9yeU5hbWU6IHN0cmluZztcbiAgLyoqXG4gICAqIEBkZWZhdWx0IFJlbW92YWxQb2xpY3kuUkVUQUlOXG4gICAqIFRoaXMgb3B0aW9uIGFsbG93IHRvIGRlbGV0ZSBvciBub3QgdGhlIEVDUiByZXBvc2l0b3J5XG4gICAqIElmIGl0IGlzIHNldCB0byBSZW1vdmFsUG9saWN5LkRFU1RST1ksIHlvdSBuZWVkIHRvIGRlbGV0ZSB0aGUgaW1hZ2VzIGJlZm9yZSB3ZSBkZWxldGUgdGhlIFJlcG9zaXRvcnkgXG4gICAqICovXG4gIHJlYWRvbmx5IGVjclJlbW92YWxQb2xpY3k/OiBSZW1vdmFsUG9saWN5O1xufVxuXG4vKipcbiAqIEEgQ0RLIGNvbnN0cnVjdCB0byBjcmVhdGUgYnVpbGQgYW5kIHB1Ymxpc2ggRU1SIG9uIEVLUyBjdXN0b20gaW1hZ2UgXG4gKiBUaGUgY29uc3RydWN0IHdpbGwgY3JlYXRlIGFuIEVDUiByZXBvc2l0b3J5IHRvIHB1Ymxpc2ggdGhlIGltYWdlcyBcbiAqIEl0IHByb3ZpZGUgYSBtZXRob2Qge0BsaW5rIHB1Ymxpc2hJbWFnZX0gdG8gYnVpbGQgYSBkb2NrZXIgZmlsZSBhbmQgcHVibGlzaCBpdCB0byB0aGUgRUNSIHJlcG9zaXRvcnkgXG4gKlxuICogUmVzb3VyY2VzIGRlcGxveWVkOlxuICpcbiAqICogTXVsdGlwbGUgU2Vzc2lvbiBQb2xpY2llcyB0aGF0IGFyZSB1c2VkIHRvIG1hcCBhbiBFTVIgU3R1ZGlvIHVzZXIgb3IgZ3JvdXAgdG8gYSBzZXQgb2YgcmVzb3VyY2VzIHRoZXkgYXJlIGFsbG93ZWQgdG8gYWNjZXNzLiBUaGVzZSByZXNvdXJjZXMgYXJlOlxuICogICAqIEVDUiBSZXBvc2l0b3J5XG4gKiAgICogQ29kZWJ1aWxkIHByb2plY3RcbiAqICAgKiBBIGN1c3RvbSByZXNvdXJjZSB0byBidWlsZCBhbmQgcHVibGlzaCBhIGN1c3RvbSBFTVIgb24gRUtTIGltYWdlIFxuICpcbiAqXG4gKiBVc2FnZSBleGFtcGxlOlxuICpcbiAqIGBgYHR5cGVzY3JpcHRcbiAqIFxuICogY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICogICBcbiAqIGNvbnN0IGFjY291bnQgPSBwcm9jZXNzLmVudi5DREtfREVGQVVMVF9BQ0NPVU5UO1xuICogY29uc3QgcmVnaW9uID0gcHJvY2Vzcy5lbnYuQ0RLX0RFRkFVTFRfUkVHSU9OO1xuICogXG4gKiBjb25zdCBzdGFjayA9IG5ldyBTdGFjayhhcHAsICdFbXJFa3NJbWFnZUJ1aWxkZXJTdGFjaycsIHtcbiAqIGVudjogeyBhY2NvdW50OiBhY2NvdW50LCByZWdpb246IHJlZ2lvbiB9LFxuICogfSk7XG4gKiBcbiAqIGNvbnN0IHB1Ymxpc2ggPSBuZXcgRW1yRWtzSW1hZ2VCdWlsZGVyKHN0YWNrLCAnRW1yRWtzSW1hZ2VCdWlsZGVyJywge1xuICogIHJlcG9zaXRvcnlOYW1lOiAnbXktcmVwbycsXG4gKiAgZWNyUmVtb3ZhbFBvbGljeTogUmVtb3ZhbFBvbGljeS5SRVRBSU5cbiAqIH0pO1xuICogXG4gKiBwdWJsaXNoLnB1Ymxpc2hJbWFnZSgnUEFUSC1UTy1ET0NLRVItRklMRS1GT0xERVInLCAndjQnKTtcbiAqXG4gKiBgYGBcbiAqL1xuXG5leHBvcnQgY2xhc3MgRW1yRWtzSW1hZ2VCdWlsZGVyIGV4dGVuZHMgQ29uc3RydWN0IHtcblxuICBwcml2YXRlIHJlYWRvbmx5IGVjclVSSTogc3RyaW5nO1xuICBwcml2YXRlIHJlYWRvbmx5IGRvY2tlckJ1aWxkUHVibGlzaENyVG9rZW46IHN0cmluZztcbiAgcHJpdmF0ZSByZWFkb25seSBhc3NldEJ1Y2tldDogQXJhQnVja2V0O1xuICBwcml2YXRlIHJlYWRvbmx5IGNvZGVidWlsZFByb2plY3ROYW1lOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgZWNyTmFtZTogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBFbXJFa3NJbWFnZUJ1aWxkZXJQcm9wcykge1xuXG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMuYXNzZXRCdWNrZXQgPSBBcmFCdWNrZXQuZ2V0T3JDcmVhdGUodGhpcywgeyBidWNrZXROYW1lOiBgJHtVdGlscy5zdHJpbmdTYW5pdGl6ZXIoU3RhY2sub2YodGhpcykuc3RhY2tOYW1lKX0tYXJhLWRvY2tlci1hc3NldHNgLCBlbmNyeXB0aW9uOiBCdWNrZXRFbmNyeXB0aW9uLktNU19NQU5BR0VEIH0pO1xuXG4gICAgbGV0IGNvZGVCdWlsZFJvbGUgPSBuZXcgUm9sZSh0aGlzLCAnY29kZWJ1aWxkYXJuJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgnY29kZWJ1aWxkLmFtYXpvbmF3cy5jb20nKSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGVjclJlcG86IFJlcG9zaXRvcnkgPSBuZXcgUmVwb3NpdG9yeSh0aGlzLCBgZWNyLSR7cHJvcHMucmVwb3NpdG9yeU5hbWV9YCwge1xuICAgICAgcmVwb3NpdG9yeU5hbWU6IHByb3BzLnJlcG9zaXRvcnlOYW1lLFxuICAgICAgcmVtb3ZhbFBvbGljeTogcHJvcHMuZWNyUmVtb3ZhbFBvbGljeSA/IHByb3BzLmVjclJlbW92YWxQb2xpY3kgOiBSZW1vdmFsUG9saWN5LlJFVEFJTixcbiAgICAgIGltYWdlU2Nhbk9uUHVzaDogdHJ1ZVxuICAgIH0pO1xuXG4gICAgdGhpcy5lY3JVUkkgPSBlY3JSZXBvLnJlcG9zaXRvcnlVcmk7XG4gICAgdGhpcy5lY3JOYW1lID0gcHJvcHMucmVwb3NpdG9yeU5hbWU7XG5cbiAgICBsZXQgZG9ja2VySW1hZ2VBY2NvdW50OiBzdHJpbmcgfCB1bmRlZmluZWQgPSBlbXJPbkVrc0ltYWdlTWFwLmdldChTdGFjay5vZih0aGlzKS5yZWdpb24pO1xuXG4gICAgaWYgKGRvY2tlckltYWdlQWNjb3VudCA9PT0gdW5kZWZpbmVkKVxuICAgICAgdGhyb3cgbmV3IEVycm9yKCdEb2NrZXIgSW1hZ2UgaXMgbm90IGF2YWlsYWJsZSBpbiB0aGUgc2VsZWN0ZWQgcmVnaW9uJyk7XG4gICAgICBcbiAgICBsZXQgY29tbWFuZHMgPSBbXG4gICAgICAnZWNobyBsb2dnaW5nIGludG8gZG9ja2VyJyxcbiAgICAgIGBhd3MgZWNyIGdldC1sb2dpbi1wYXNzd29yZCAtLXJlZ2lvbiAke0F3cy5SRUdJT059IHwgZG9ja2VyIGxvZ2luIC0tdXNlcm5hbWUgQVdTIC0tcGFzc3dvcmQtc3RkaW4gJHtkb2NrZXJJbWFnZUFjY291bnQhfS5ka3IuZWNyLiR7QXdzLlJFR0lPTn0uYW1hem9uYXdzLmNvbWAsXG4gICAgICAnZWNobyBCdWlsZCBzdGFydCcsXG4gICAgICAnZWNobyAkRE9DS0VSX0ZJTEVfUzNfUEFUSCcsXG4gICAgICAnYXdzIHMzIGNwICRET0NLRVJfRklMRV9TM19QQVRIIERvY2tlcmZpbGUnLFxuICAgICAgJ2RvY2tlciBidWlsZCAtdCBsb2NhbCAuJyxcbiAgICAgICdkb2NrZXIgbG9nb3V0JyxcbiAgICAgICdlY2hvICRlY3JVUkknLFxuICAgICAgYGF3cyBlY3IgZ2V0LWxvZ2luLXBhc3N3b3JkIC0tcmVnaW9uICR7QXdzLlJFR0lPTn0gfCBkb2NrZXIgbG9naW4gLS11c2VybmFtZSBBV1MgLS1wYXNzd29yZC1zdGRpbiAke3RoaXMuZWNyVVJJfWAsXG4gICAgICAnZG9ja2VyIHRhZyBsb2NhbCAkZWNyVVJJOiR0YWcnLFxuICAgICAgJ2RvY2tlciBwdXNoICRlY3JVUkk6JHRhZycsXG4gICAgICAnZG9ja2VyIGxvZ291dCdcbiAgICBdO1xuXG4gICAgY29uc3QgY29kZWJ1aWxkUHJvamVjdCA9IG5ldyBQcm9qZWN0KHRoaXMsIGBEb2NrZXJJbWFnZURlcGxveVByb2plY3QtJHtwcm9wcy5yZXBvc2l0b3J5TmFtZX1gLCB7XG4gICAgICBidWlsZFNwZWM6IEJ1aWxkU3BlYy5mcm9tT2JqZWN0KHtcbiAgICAgICAgdmVyc2lvbjogJzAuMicsXG4gICAgICAgIHBoYXNlczoge1xuICAgICAgICAgIGJ1aWxkOiB7XG4gICAgICAgICAgICBjb21tYW5kczogY29tbWFuZHMsXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0pLFxuICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgcHJpdmlsZWdlZDogdHJ1ZSxcbiAgICAgICAgYnVpbGRJbWFnZTogTGludXhCdWlsZEltYWdlLlNUQU5EQVJEXzVfMCxcbiAgICAgICAgY29tcHV0ZVR5cGU6IENvbXB1dGVUeXBlLlNNQUxMXG4gICAgICB9LFxuICAgICAgcm9sZTogY29kZUJ1aWxkUm9sZSxcbiAgICB9KTtcblxuICAgIGVjclJlcG8uZ3JhbnRQdWxsUHVzaChjb2RlQnVpbGRSb2xlKTtcbiAgICB0aGlzLmFzc2V0QnVja2V0LmdyYW50UmVhZChjb2RlQnVpbGRSb2xlKTtcblxuICAgIGNvZGVCdWlsZFJvbGUuYWRkVG9Qb2xpY3kobmV3IFBvbGljeVN0YXRlbWVudCAoe1xuICAgICAgcmVzb3VyY2VzOiBbYGFybjphd3M6ZWNyOiR7QXdzLlJFR0lPTn06JHtkb2NrZXJJbWFnZUFjY291bnR9OnJlcG9zaXRvcnkvKmBdLFxuICAgICAgYWN0aW9uczogW1xuICAgICAgICAnZWNyOkJhdGNoR2V0SW1hZ2UnLFxuICAgICAgICAnZWNyOkdldEF1dGhvcml6YXRpb25Ub2tlbicsXG4gICAgICAgICdlY3I6QmF0Y2hDaGVja0xheWVyQXZhaWxhYmlsaXR5JyxcbiAgICAgICAgJ2VjcjpHZXREb3dubG9hZFVybEZvckxheWVyJyxcbiAgICAgIF1cbiAgICB9KSk7XG5cbiAgICB0aGlzLmNvZGVidWlsZFByb2plY3ROYW1lID0gY29kZWJ1aWxkUHJvamVjdC5wcm9qZWN0TmFtZTtcblxuICAgIHRoaXMuZG9ja2VyQnVpbGRQdWJsaXNoQ3JUb2tlbiA9IEVtckVrc0ltYWdlQnVpbGRlckNSUHJvdmlkZXJTZXR1cCh0aGlzLCBjb2RlYnVpbGRQcm9qZWN0LnByb2plY3RBcm4pO1xuICB9XG5cbiAgLyoqXG4gICAqIEEgbWV0aG9kIHRvIGJ1aWxkIGFuZCBwdWJsaXNoIHRoZSBjdXN0b20gaW1hZ2UgZnJvbSBhIERvY2tlcmZpbGUgXG4gICAqIFRoZSBtZXRob2QgaW52b2tlIHRoZSBjdXN0b20gcmVzb3VyY2UgZGVwbG95ZWQgYnkgdGhlIGNvbnN0cnVjdCBcbiAgICogYW5kIHB1Ymxpc2ggdGhlICoqVVJJKiogb2YgdGhlIHB1Ymxpc2hlZCBjdXN0b20gaW1hZ2UgYXMgQ2xvdWRmb3JtYXRpb24gb3V0cHV0ICBcbiAgICogQHBhcmFtIHtzdHJpbmd9IGRvY2tlcmZpbGVQYXRoIFBhdGggdG8gdGhlIGZvbGRlciBmb3IgRG9ja2VyZmlsZSBcbiAgICogQHBhcmFtIHtzdHJpbmd9IHRhZyBUaGUgdGFnIHVzZWQgdG8gcHVibGlzaCB0byB0aGUgRUNSIHJlcG9zaXRvcnlcbiAgICovXG5cbiAgcHVibGljIHB1Ymxpc2hJbWFnZShkb2NrZXJmaWxlUGF0aDogc3RyaW5nLCB0YWc6IHN0cmluZykge1xuXG4gICAgbmV3IEJ1Y2tldERlcGxveW1lbnQodGhpcywgYERvY2tlcmZpbGVBc3NldERlcGxveW1lbnQtJHt0YWd9YCwge1xuICAgICAgZGVzdGluYXRpb25CdWNrZXQ6IHRoaXMuYXNzZXRCdWNrZXQsXG4gICAgICBkZXN0aW5hdGlvbktleVByZWZpeDogYCR7dGhpcy5lY3JOYW1lfS8ke3RhZ31gLFxuICAgICAgc291cmNlczogW1NvdXJjZS5hc3NldChkb2NrZXJmaWxlUGF0aCldLFxuICAgIH0pO1xuXG4gICAgY29uc3QgY29udGFpbmVySW1hZ2UgPSAgbmV3IEN1c3RvbVJlc291cmNlKHRoaXMsIGBFbXJFa3NJbWFnZUJ1aWxkLSR7dGFnfWAsIHtcbiAgICAgIHNlcnZpY2VUb2tlbjogdGhpcy5kb2NrZXJCdWlsZFB1Ymxpc2hDclRva2VuLFxuICAgICAgcHJvcGVydGllczoge1xuICAgICAgICBzM1BhdGg6IGBzMzovLyR7dGhpcy5hc3NldEJ1Y2tldC5idWNrZXROYW1lfS8ke3RoaXMuZWNyTmFtZX0vJHt0YWd9L0RvY2tlcmZpbGVgLFxuICAgICAgICB0YWc6IHRhZyxcbiAgICAgICAgZWNyVVJJOiB0aGlzLmVjclVSSSxcbiAgICAgICAgY29kZWJ1aWxkUHJvamVjdE5hbWU6IHRoaXMuY29kZWJ1aWxkUHJvamVjdE5hbWUsXG4gICAgICAgIGVjck5hbWU6IHRoaXMuZWNyTmFtZSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBuZXcgQ2ZuT3V0cHV0KHRoaXMsIGBVUkktJHt0YWd9YCwge1xuICAgICAgdmFsdWU6IGNvbnRhaW5lckltYWdlLmdldEF0dFN0cmluZygnQ29udGFpbmVyVXJpJyksXG4gICAgfSlcblxuICB9XG5cbn0iXX0=