"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.FlywayRunner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const path = require("path");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const s3deploy = require("aws-cdk-lib/aws-s3-deployment");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const constructs_1 = require("constructs");
/**
 * A CDK construct that runs flyway migration scripts against a redshift cluster.
 *
 * This construct is based on two main resource, an AWS Lambda hosting a flyway runner
 * and one custom resource invoking it when content of migrationScriptsFolderAbsolutePath changes.
 *
 * Usage example:
 *
 * *This example assume that migration SQL files are located in `resources/sql` of the cdk project.*
 * ```typescript
 * import * as path from 'path';
 * import * as ec2 from 'aws-cdk-lib/aws-ec2';
 * import * as redshift from '@aws-cdk/aws-redshift-alpha';
 * import * as cdk from 'aws-cdk-lib';
 *
 * import { FlywayRunner } from 'aws-analytics-reference-architecture';
 *
 * const integTestApp = new cdk.App();
 * const stack = new cdk.Stack(integTestApp, 'fywayRunnerTest');
 *
 * const vpc = new ec2.Vpc(stack, 'Vpc');

 * const dbName = 'testdb';
 * const cluster = new redshift.Cluster(stack, 'Redshift', {
 *   removalPolicy: cdk.RemovalPolicy.DESTROY,
 *   masterUser: {
 *     masterUsername: 'admin',
 *   },
 *   vpc,
 *   defaultDatabaseName: dbName,
 * });

 * new FlywayRunner(stack, 'testMigration', {
 *   migrationScriptsFolderAbsolutePath: path.join(__dirname, './resources/sql'),
 *   cluster: cluster,
 *   vpc: vpc,
 *   databaseName: dbName,
 * });
 * ```
 */
class FlywayRunner extends constructs_1.Construct {
    /**
     * Constructs a new instance of the FlywayRunner construct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {FlywayRunnerProps} props the FlywayRunner [properties]{@link FlywayRunnerProps}
     */
    constructor(scope, id, props) {
        super(scope, id);
        const sqlFilesAsset = s3deploy.Source.asset(props.migrationScriptsFolderAbsolutePath);
        const migrationFilesBucket = new s3.Bucket(this, 'migrationFilesBucket', {
            versioned: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
        });
        const migrationFilesDeployment = new s3deploy.BucketDeployment(this, 'DeploySQLMigrationFiles', {
            sources: [sqlFilesAsset],
            destinationBucket: migrationFilesBucket,
        });
        let flywayLambdaPolicy = [
            new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                resources: [
                    '*',
                ],
                actions: [
                    'cloudformation:DescribeStacks',
                    'cloudformation:DescribeStackResource',
                ],
            }),
            new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                resources: [
                    '*',
                ],
                actions: [
                    'ec2:CreateNetworkInterface',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:DeleteNetworkInterface',
                ],
            }),
        ];
        const flywayLambda = new pre_bundled_function_1.PreBundledFunction(this, 'FlywayLambda', {
            codePath: path.join(__dirname.split('/').slice(-1)[0], './resources/flyway-lambda/flyway-all.jar'),
            lambdaPolicyStatements: flywayLambdaPolicy,
            handler: 'com.geekoosh.flyway.FlywayCustomResourceHandler::handleRequest',
            runtime: lambda.Runtime.JAVA_11,
            logRetention: props.logRetention ?? logs.RetentionDays.ONE_WEEK,
            memorySize: 2048,
            timeout: cdk.Duration.seconds(900),
            vpc: props.vpc,
            securityGroups: props.cluster.connections.securityGroups,
            environment: {
                S3_BUCKET: migrationFilesBucket.bucketName,
                DB_CONNECTION_STRING: `jdbc:redshift://${props.cluster.clusterEndpoint.socketAddress}/${props.databaseName}`,
                DB_SECRET: props.cluster.secret.secretFullArn,
            },
        });
        // Allowing connection to the cluster
        props.cluster.connections.allowDefaultPortInternally();
        props.cluster.secret?.grantRead(flywayLambda);
        migrationFilesBucket.grantRead(flywayLambda);
        const flywayCustomResourceProvider = new custom_resources_1.Provider(this, 'FlywayCustomResourceProvider', {
            onEventHandler: flywayLambda,
            logRetention: props.logRetention ?? logs.RetentionDays.ONE_WEEK,
            securityGroups: props.cluster.connections.securityGroups,
            vpc: props.vpc,
        });
        this.runner = new aws_cdk_lib_1.CustomResource(this, 'trigger', {
            serviceToken: flywayCustomResourceProvider.serviceToken,
            properties: {
                flywayMethod: 'migrate',
                placeholders: props.replaceDictionary,
                assetHash: migrationFilesDeployment.node.findChild('Asset1').assetHash,
            },
        });
        for (const subnet of props.vpc.privateSubnets) {
            flywayLambda.node.addDependency(subnet);
        }
        flywayCustomResourceProvider.node.addDependency(migrationFilesDeployment);
    }
}
_a = JSII_RTTI_SYMBOL_1;
FlywayRunner[_a] = { fqn: "aws-analytics-reference-architecture.FlywayRunner", version: "2.12.6" };
exports.FlywayRunner = FlywayRunner;
//# sourceMappingURL=data:application/json;base64,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