import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as logs from 'aws-cdk-lib/aws-logs';
import * as redshift from '@aws-cdk/aws-redshift-alpha';
import { CustomResource } from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * The properties of the FlywayRunner construct, needed to run flyway migration scripts.
 */
export interface FlywayRunnerProps {
    /**
     * The absolute path to the flyway migration scripts.
     * Those scripts needs to follow expected flyway naming convention.
     * @see https://flywaydb.org/documentation/concepts/migrations.html#sql-based-migrations for more details.
     */
    readonly migrationScriptsFolderAbsolutePath: string;
    /**
     * The cluster to run migration scripts against.
     */
    readonly cluster: redshift.Cluster;
    /**
     * The vpc hosting the cluster.
     */
    readonly vpc: ec2.Vpc;
    /**
     * The database name to run migration scripts against.
     */
    readonly databaseName: string;
    /**
     * Period to keep the logs around.
     * @default logs.RetentionDays.ONE_WEEK
     */
    readonly logRetention?: logs.RetentionDays;
    /**
     * A key-value map of string (encapsulated between `${` and `}`) to replace in the SQL files given.
     *
     * Example:
     *
     * * The SQL file:
     *
     *   ```sql
     *   SELECT * FROM ${TABLE_NAME};
     *   ```
     * * The replacement map:
     *
     *   ```typescript
     *   replaceDictionary = {
     *     TABLE_NAME: 'my_table'
     *   }
     *   ```
     * @default - No replacement is done
     */
    readonly replaceDictionary?: {
        [key: string]: string;
    };
}
/**
 * A CDK construct that runs flyway migration scripts against a redshift cluster.
 *
 * This construct is based on two main resource, an AWS Lambda hosting a flyway runner
 * and one custom resource invoking it when content of migrationScriptsFolderAbsolutePath changes.
 *
 * Usage example:
 *
 * *This example assume that migration SQL files are located in `resources/sql` of the cdk project.*
 * ```typescript
 * import * as path from 'path';
 * import * as ec2 from 'aws-cdk-lib/aws-ec2';
 * import * as redshift from '@aws-cdk/aws-redshift-alpha';
 * import * as cdk from 'aws-cdk-lib';
 *
 * import { FlywayRunner } from 'aws-analytics-reference-architecture';
 *
 * const integTestApp = new cdk.App();
 * const stack = new cdk.Stack(integTestApp, 'fywayRunnerTest');
 *
 * const vpc = new ec2.Vpc(stack, 'Vpc');

 * const dbName = 'testdb';
 * const cluster = new redshift.Cluster(stack, 'Redshift', {
 *   removalPolicy: cdk.RemovalPolicy.DESTROY,
 *   masterUser: {
 *     masterUsername: 'admin',
 *   },
 *   vpc,
 *   defaultDatabaseName: dbName,
 * });

 * new FlywayRunner(stack, 'testMigration', {
 *   migrationScriptsFolderAbsolutePath: path.join(__dirname, './resources/sql'),
 *   cluster: cluster,
 *   vpc: vpc,
 *   databaseName: dbName,
 * });
 * ```
 */
export declare class FlywayRunner extends Construct {
    readonly runner: CustomResource;
    /**
     * Constructs a new instance of the FlywayRunner construct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {FlywayRunnerProps} props the FlywayRunner [properties]{@link FlywayRunnerProps}
     */
    constructor(scope: Construct, id: string, props: FlywayRunnerProps);
}
