"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataDomain = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const constructs_1 = require("constructs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const central_governance_1 = require("./central-governance");
const data_lake_storage_1 = require("../data-lake-storage");
const data_domain_nrac_workflow_1 = require("./data-domain-nrac-workflow");
const data_domain_tbac_workflow_1 = require("./data-domain-tbac-workflow");
const data_domain_crawler_1 = require("./data-domain-crawler");
const data_mesh_workflow_role_1 = require("./data-mesh-workflow-role");
const lake_formation_1 = require("../lake-formation");
const s3_cross_account_1 = require("../s3-cross-account");
/**
 * This CDK Construct creates all required resources for data mesh in Data Domain account.
 *
 * It creates the following:
 * * A data lake with multiple layers (Raw, Cleaned, Transformed) using {@link DataLakeStorage} construct
 * * An mazon EventBridge Event Bus and Rules to enable Central Governance account to send events to Data Domain account
 * * An AWS Secret Manager secret encrypted via AWS KMS and used to share references with the central governance account
 * * A Data Domain Workflow {@link DataDomainWorkflow} responsible for creating resources in the data domain via a Step Functions state machine
 * * An optional Crawler workflow {@link DataDomainCrawler} responsible for updating the data product schema after registration via a Step Functions state machine
 *
 * Usage example:
 * ```typescript
 * import { App, Stack } from 'aws-cdk-lib';
 * import { Role } from 'aws-cdk-lib/aws-iam';
 * import { DataDomain } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new App();
 * const stack = new Stack(exampleApp, 'DataProductStack');
 *
 * new DataDomain(stack, 'myDataDomain', {
 *  centralAccountId: '1234567891011',
 *  crawlerWorkflow: true,
 *  domainName: 'domainName'
 * });
 * ```
 */
class DataDomain extends constructs_1.Construct {
    /**
     * Construct a new instance of DataDomain.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataDomainProps} props the DataDomainProps properties
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // The data lake used by the Data Domain
        this.dataLake = new data_lake_storage_1.DataLakeStorage(this, 'dataLakeStorage');
        this.centralAccountId = props.centralAccountId;
        // Add CDK execution role to LF admins
        lake_formation_1.LakeFormationAdmin.addCdkExecRole(scope, 'CdkLfAdmin');
        // Using the Bucket object and not the IBucket because CDK needs to change the bucket policy
        // KMS key is automatically discovered from the Bucket object and key policy is updated
        new s3_cross_account_1.S3CrossAccount(this, 'DataProductsPathCrossAccount', {
            accountId: this.centralAccountId,
            s3Bucket: this.dataLake.cleanBucket,
            s3ObjectKey: DataDomain.DATA_PRODUCTS_PREFIX,
        });
        // Workflow role used by state machine workflows
        const workflowRole = new data_mesh_workflow_role_1.DataMeshWorkflowRole(this, 'WorkflowRole');
        // Event Bridge event bus for data domain account
        this.eventBus = new aws_events_1.EventBus(this, 'dataDomainEventBus', {
            eventBusName: DataDomain.DOMAIN_BUS_NAME,
        });
        this.eventBus.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        // Cross-account policy to allow the central account to send events to data domain's bus
        const crossAccountBusPolicy = new aws_events_1.CfnEventBusPolicy(this, 'crossAccountBusPolicy', {
            eventBusName: this.eventBus.eventBusName,
            statementId: 'AllowCentralAccountToPutEvents',
            action: 'events:PutEvents',
            principal: this.centralAccountId,
        });
        crossAccountBusPolicy.node.addDependency(this.eventBus);
        // Create NRAC workflow
        const nracWorkflow = new data_domain_nrac_workflow_1.DataDomainNracWorkflow(this, 'nracWorkflow', {
            workflowRole: workflowRole.role,
            centralAccountId: this.centralAccountId,
            eventBus: this.eventBus,
        });
        // Event Bridge Rule to trigger NRAC worklfow upon event from the central account
        this.addBusRule('NRAC', central_governance_1.LfAccessControlMode.NRAC, nracWorkflow.stateMachine);
        // Create TBAC workflow
        const tbacWorkflow = new data_domain_tbac_workflow_1.DataDomainTbacWorkflow(this, 'tbacWorkflow', {
            domainName: props.domainName,
            workflowRole: workflowRole.role,
            centralAccountId: this.centralAccountId,
            eventBus: this.eventBus,
        });
        // Event Bridge Rule to trigger NRAC worklfow upon event from the central account
        this.addBusRule('TBAC', central_governance_1.LfAccessControlMode.TBAC, tbacWorkflow.stateMachine);
        // Allow the workflow role to send events to data domain event bus
        workflowRole.role.attachInlinePolicy(new aws_iam_1.Policy(this, 'SendEvents', {
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ['events:Put*'],
                    resources: [this.eventBus.eventBusArn],
                }),
            ],
        }));
        // create a workflow to update data products schemas on registration
        if (props.crawlerWorkflow) {
            const crawlerWorkflow = new data_domain_crawler_1.DataDomainCrawler(this, 'DataDomainCrawler', {
                domainName: props.domainName,
                workflowRole: workflowRole.role,
                dataProductsBucket: this.dataLake.cleanBucket,
                dataProductsPrefix: DataDomain.DATA_PRODUCTS_PREFIX,
                eventBus: this.eventBus,
            });
            // AWS Lambda function responsible to grant permissions on AWS Lake Formation tag to crawler role
            const tagPermissionsFn = new pre_bundled_function_1.PreBundledFunction(this, 'TagPermissionsFn', {
                runtime: aws_lambda_1.Runtime.PYTHON_3_9,
                codePath: 'data-mesh/resources/lambdas/crawler-tag-permission',
                handler: 'lambda.handler',
                logRetention: aws_logs_1.RetentionDays.ONE_DAY,
                timeout: aws_cdk_lib_1.Duration.seconds(20),
                role: workflowRole.role,
                environment: {
                    'CRAWLER_ROLE_ARN': crawlerWorkflow.crawlerRole.roleArn,
                    'CENTRAL_CATALOG_ID': props.centralAccountId,
                    'TAG_KEY': central_governance_1.CentralGovernance.DOMAIN_TAG_KEY,
                    'DOMAIN_TAG_VALUE': props.domainName,
                }
            });
            // add a rule to trigger the workflow from the event bus
            const triggerCrawlerRule = new aws_events_1.Rule(this, 'TriggerCrawler', {
                eventBus: this.eventBus,
                targets: [
                    crawlerWorkflow.stateMachine,
                ],
                eventPattern: {
                    source: ['com.central.stepfunction'],
                    detailType: ['triggerCrawler'],
                }
            });
            triggerCrawlerRule.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
            // add a rule to trigger the workflow from the event bus
            const grantCrawlerPermissionRule = new aws_events_1.Rule(this, 'grantCrawlerPermission', {
                eventBus: this.eventBus,
                targets: [
                    new targets.LambdaFunction(tagPermissionsFn),
                ],
                eventPattern: {
                    source: ['com.central.stepfunction'],
                    detailType: ['grantCrawlerPermission'],
                }
            });
            grantCrawlerPermissionRule.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
            // allow grantCrawlerPermissionRule to invoke Lambda fn
            targets.addLambdaPermission(grantCrawlerPermissionRule, tagPermissionsFn);
            // Construct central gov. account bus ARN
            const centralBusArn = `arn:aws:events:${aws_cdk_lib_1.Aws.REGION}:${props.centralAccountId}:event-bus/${central_governance_1.CentralGovernance.CENTRAL_BUS_NAME}`;
            // Event Bridge Rule to send event about crawler state to Central account
            const forwardCrawlerStateRule = new aws_events_1.Rule(this, 'CrawlerStateRule', {
                eventPattern: {
                    source: ['data-domain-state-change'],
                    detailType: ['data-domain-crawler-update'],
                },
                eventBus: this.eventBus,
            });
            forwardCrawlerStateRule.addTarget(new targets.EventBus(aws_events_1.EventBus.fromEventBusArn(this, '`${id}centralBus`', centralBusArn)));
            forwardCrawlerStateRule.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        }
        // create the data domain configuration object (in JSON) to be passed to the central governance account 
        var secretObject = {
            DomainName: aws_cdk_lib_1.SecretValue.unsafePlainText(props.domainName),
            BucketName: aws_cdk_lib_1.SecretValue.unsafePlainText(this.dataLake.cleanBucket.bucketName),
            Prefix: aws_cdk_lib_1.SecretValue.unsafePlainText(DataDomain.DATA_PRODUCTS_PREFIX),
        };
        // if the data product bucket is encrypted, add the key ID
        if (this.dataLake.cleanBucket.encryptionKey) {
            secretObject = {
                ...secretObject,
                ...{ KmsKeyId: aws_cdk_lib_1.SecretValue.unsafePlainText(this.dataLake.cleanBucket.encryptionKey.keyId) }
            };
        }
        const centralGovAccount = new aws_iam_1.AccountPrincipal(props.centralAccountId);
        // create a KMS key for encrypting the secret. It's required for cross account secret access
        const secretKey = new aws_kms_1.Key(this, 'SecretKey', {
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        secretKey.grantDecrypt(centralGovAccount);
        // create the secret containing the data domain configuration object
        const domainConfigSecret = new aws_secretsmanager_1.Secret(this, 'DomainBucketSecret', {
            secretObjectValue: secretObject,
            secretName: DataDomain.DOMAIN_CONFIG_SECRET,
            encryptionKey: secretKey,
        });
        domainConfigSecret.grantRead(centralGovAccount);
        // output the full ARN of the secret to be passed when registring the data domain
        new aws_cdk_lib_1.CfnOutput(this, 'DomainSecretArnOutput', {
            value: domainConfigSecret.secretArn,
            exportName: `${aws_cdk_lib_1.Aws.ACCOUNT_ID}SecretArn`,
        });
    }
    addBusRule(id, mode, workflow) {
        // Create a Rule in Data Domain Event Bus
        const rule = new aws_events_1.Rule(this, `${id}Rule`, {
            eventPattern: {
                source: ['com.central.stepfunction'],
                account: [this.centralAccountId],
                detailType: [`${aws_cdk_lib_1.Aws.ACCOUNT_ID}_createResourceLinks`],
                detail: {
                    'lf_access_mode': [mode],
                },
            },
            eventBus: this.eventBus,
        });
        rule.node.addDependency(this.eventBus);
        rule.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        // Add target for this Rule
        rule.addTarget(new targets.SfnStateMachine(workflow));
    }
}
_a = JSII_RTTI_SYMBOL_1;
DataDomain[_a] = { fqn: "aws-analytics-reference-architecture.DataDomain", version: "2.12.6" };
DataDomain.DATA_PRODUCTS_PREFIX = 'data-products';
DataDomain.DOMAIN_CONFIG_SECRET = 'domain-config';
DataDomain.DOMAIN_BUS_NAME = 'data-mesh-bus';
exports.DataDomain = DataDomain;
//# sourceMappingURL=data:application/json;base64,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