"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CentralGovernance = exports.LfAccessControlMode = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
;
const constructs_1 = require("constructs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const targets = require("aws-cdk-lib/aws-events-targets");
const lakeformation = require("aws-cdk-lib/aws-lakeformation");
const data_mesh_workflow_role_1 = require("./data-mesh-workflow-role");
const lake_formation_1 = require("../lake-formation");
const lake_formation_2 = require("../lake-formation");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const data_domain_1 = require("./data-domain");
/**
 * Enum to define access control mode in Lake Formation
 */
var LfAccessControlMode;
(function (LfAccessControlMode) {
    LfAccessControlMode["NRAC"] = "nrac";
    LfAccessControlMode["TBAC"] = "tbac";
})(LfAccessControlMode = exports.LfAccessControlMode || (exports.LfAccessControlMode = {}));
/**
 * This CDK Construct creates a Data Product registration workflow and resources for the Central Governance account.
 * It uses AWS Step Functions state machine to orchestrate the workflow:
 * * creates tables in AWS Glue Data Catalog
 * * shares tables to Data Product owner account (Producer)
 *
 * This construct also creates an Amazon EventBridge Event Bus to enable communication with Data Domain accounts (Producer/Consumer).
 *
 * This construct requires to use the default [CDK qualifier](https://docs.aws.amazon.com/cdk/v2/guide/bootstrapping.html) generated with the standard CDK bootstrap stack.
 * It ensures the right CDK execution role is used and granted Lake Formation administrator permissions so CDK can create Glue databases when registring a DataDomain.
 *
 * To register a DataDomain, the following information are required:
 * * The account Id of the DataDomain
 * * The secret ARN for the domain configuration available as a CloudFormation output when creating a {@link DataDomain}
 *
 * Usage example:
 * ```typescript
 * import { App, Stack } from 'aws-cdk-lib';
 * import { Role } from 'aws-cdk-lib/aws-iam';
 * import { CentralGovernance, LfTag } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new App();
 * const stack = new Stack(exampleApp, 'CentralGovStack');
 *
 * const tags: LfTag[] = [{key: 'tag1': values:['LfTagValue1', 'LfTagValue2']}]
 * const governance = new CentralGovernance(stack, 'myCentralGov', { tags });
 *
 * governance.registerDataDomain('Domain1', 'domain1Name', <DOMAIN_CONFIG_SECRET_ARN>);
 * ```
 */
class CentralGovernance extends constructs_1.Construct {
    /**
     * Construct a new instance of CentralGovernance.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {CentralGovernanceProps} props the CentralGovernance properties
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.lfTags = [];
        // Event Bridge event bus for the Central Governance account
        this.eventBus = new aws_events_1.EventBus(this, 'centralEventBus', {
            eventBusName: CentralGovernance.CENTRAL_BUS_NAME,
        });
        this.eventBus.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        this.cdkLfAdmin = lake_formation_2.LakeFormationAdmin.addCdkExecRole(scope, 'CdkLfAdmin');
        // Create LF tags in Central Governance account
        if (props) {
            if (props.lfTags) {
                this.lfTags = props.lfTags;
                this.lfTags.forEach(tag => new lakeformation.CfnTag(this, `CentralLfTag${tag.key}`, {
                    tagKey: tag.key,
                    tagValues: tag.values,
                }).node.addDependency(this.cdkLfAdmin));
            }
        }
        // Workflow role used by the state machine
        this.workflowRole = new data_mesh_workflow_role_1.DataMeshWorkflowRole(this, 'WorkflowRole').role;
        this.workflowRole.attachInlinePolicy(new aws_iam_1.Policy(this, 'sendEvents', {
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: ['events:Put*'],
                    resources: [this.eventBus.eventBusArn],
                }),
            ],
        }));
        // Task to create a table
        const createTable = new aws_stepfunctions_tasks_1.CallAwsService(this, 'createTable', {
            service: 'glue',
            action: 'createTable',
            iamResources: ['*'],
            parameters: {
                'DatabaseName.$': `States.Format('{}-${CentralGovernance.DOMAIN_DATABASE_PREFIX}-{}', $.lf_access_mode, $.producer_acc_id)`,
                'TableInput': {
                    'Name.$': '$.tables.name',
                    'Owner.$': '$.producer_acc_id',
                    'StorageDescriptor': {
                        'Location.$': '$.tables.location'
                    }
                },
            },
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        // Grant SUPER permissions (and grantable) on product database and tables to Data Domain account
        const grantTablePermissions = new aws_stepfunctions_tasks_1.CallAwsService(this, 'grantTablePermissionsToProducer', {
            service: 'lakeformation',
            action: 'grantPermissions',
            iamResources: ['*'],
            parameters: {
                'Permissions': [
                    "ALL"
                ],
                'PermissionsWithGrantOption': [
                    'ALL'
                ],
                'Principal': {
                    'DataLakePrincipalIdentifier.$': '$.producer_acc_id'
                },
                'Resource': {
                    'Table': {
                        'DatabaseName.$': `States.Format('{}-${CentralGovernance.DOMAIN_DATABASE_PREFIX}-{}', $.lf_access_mode ,$.producer_acc_id)`,
                        'Name.$': '$.tables.name',
                    },
                },
            },
            outputPath: '$.tables.name',
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        grantTablePermissions.addRetry({
            errors: ['LakeFormation.ConcurrentModificationException'],
            backoffRate: 2,
            interval: aws_cdk_lib_1.Duration.seconds(5),
            maxAttempts: 2
        });
        // Trigger workflow in Data Domain account via Event Bridge
        const triggerProducer = new aws_stepfunctions_tasks_1.EventBridgePutEvents(this, 'triggerCreateResourceLinks', {
            entries: [{
                    detail: aws_stepfunctions_1.TaskInput.fromObject({
                        'central_database_name': aws_stepfunctions_1.JsonPath.format("{}-{}-{}", aws_stepfunctions_1.JsonPath.stringAt("$.lf_access_mode"), CentralGovernance.DOMAIN_DATABASE_PREFIX, aws_stepfunctions_1.JsonPath.stringAt("$.producer_acc_id")),
                        'central_account_id': aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        'producer_acc_id': aws_stepfunctions_1.JsonPath.stringAt("$.producer_acc_id"),
                        'database_name': aws_stepfunctions_1.JsonPath.format("{}-{}", aws_stepfunctions_1.JsonPath.stringAt("$.lf_access_mode"), CentralGovernance.DOMAIN_DATABASE_PREFIX),
                        'table_names': aws_stepfunctions_1.JsonPath.stringAt("$.map_result.flatten"),
                        'lf_access_mode': aws_stepfunctions_1.JsonPath.stringAt("$.lf_access_mode"),
                    }),
                    detailType: aws_stepfunctions_1.JsonPath.format("{}_createResourceLinks", aws_stepfunctions_1.JsonPath.stringAt("$.producer_acc_id")),
                    eventBus: this.eventBus,
                    source: 'com.central.stepfunction'
                }]
        });
        // Iterate over multiple tables in parallel
        const tablesMapTask = new aws_stepfunctions_1.Map(this, 'forEachTable', {
            itemsPath: '$.tables',
            parameters: {
                'producer_acc_id.$': '$.producer_acc_id',
                'tables.$': '$$.Map.Item.Value',
                'lf_access_mode.$': '$.lf_access_mode',
            },
            resultSelector: {
                'flatten.$': '$[*]'
            },
            resultPath: '$.map_result',
        });
        // Check if LF access mode is NRAC
        const checkModeTask = new aws_stepfunctions_1.Choice(this, 'isModeNRAC')
            .when(aws_stepfunctions_1.Condition.stringEquals('$.lf_access_mode', LfAccessControlMode.NRAC), grantTablePermissions)
            .otherwise(new aws_stepfunctions_1.Pass(this, 'Pass', {
            outputPath: '$.tables.name',
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD
        }));
        tablesMapTask.iterator(createTable.addCatch(checkModeTask, {
            errors: ['Glue.AlreadyExistsException'],
            resultPath: '$.CreateTableException',
        }).next(checkModeTask));
        tablesMapTask.next(triggerProducer);
        // Create Log group for this state machine
        const logGroup = new aws_logs_1.LogGroup(this, 'centralGov-stateMachine', {
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
            logGroupName: '/aws/vendedlogs/data-mesh/workflow',
        });
        logGroup.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
        // State machine to register data product from Data Domain
        new aws_stepfunctions_1.StateMachine(this, 'RegisterDataProduct', {
            definition: tablesMapTask,
            role: this.workflowRole,
            logs: {
                destination: logGroup,
                level: aws_stepfunctions_1.LogLevel.ALL,
            },
        });
    }
    /**
     * Registers a new Data Domain account in Central Governance account.
     * Each Data Domain account {@link DataDomain} has to be registered in Central Gov. account before it can participate in a mesh.
     *
     * It creates:
     * * A cross-account policy for Amazon EventBridge Event Bus to enable Data Domain to send events to Central Gov. account
     * * A Lake Formation data access role scoped down to the data domain products bucket
     * * A Glue Catalog Database to hold Data Products for this Data Domain
     * * A Rule to forward events to target Data Domain account.
     *
     * Object references are passed from the DataDomain account to the CentralGovernance account via a AWS Secret Manager secret and cross account access.
     * It includes the following JSON object:
     * ```json
     * {
     *   BucketName: 'clean-<ACCOUNT_ID>-<REGION>',
     *   Prefix: 'data-products',
     *   KmsKeyId: '<KMS_ID>,
     * }
     * ```
     *
     * @param {string} id the ID of the CDK Construct
     * @param {string} domainId the account ID of the DataDomain to register
     * @param {string} domainName the name of the DataDomain, i.e. Line of Business name
     * @param {string} domainSecretArn the full ARN of the secret used by producers to share references with the central governance
     * @param {LfAccessControlMode} lfAccessControlMode Lake Formation Access Control mode for the DataDomain
     * @access public
     */
    registerDataDomain(id, domainId, domainName, domainSecretArn, lfAccessControlMode) {
        // Import the data domain secret from it's full ARN
        const domainSecret = aws_secretsmanager_1.Secret.fromSecretCompleteArn(this, `${id}DomainSecret`, domainSecretArn);
        // Extract data domain references
        const domainBucket = domainSecret.secretValueFromJson('BucketName').unsafeUnwrap();
        const domainPrefix = domainSecret.secretValueFromJson('Prefix').unsafeUnwrap();
        const domainKey = domainSecret.secretValueFromJson('KmsKeyId').unsafeUnwrap();
        // Construct domain event bus ARN
        const dataDomainBusArn = `arn:aws:events:${aws_cdk_lib_1.Aws.REGION}:${domainId}:event-bus/${data_domain_1.DataDomain.DOMAIN_BUS_NAME}`;
        const lfModes = lfAccessControlMode ? [lfAccessControlMode] : [LfAccessControlMode.NRAC, LfAccessControlMode.TBAC];
        lfModes.forEach(mode => {
            // Create the database in Glue with datadomain mode+prefix+bucket, and metadata parameters
            new aws_cdk_lib_1.aws_glue.CfnDatabase(this, `${id}DataDomainDatabase-${mode}`, {
                catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                databaseInput: {
                    description: `Database for data products in ${domainName} data domain. Account id: ${domainId}. LF Access Control mode: ${mode}`,
                    name: mode + '-' + CentralGovernance.DOMAIN_DATABASE_PREFIX + '-' + domainId,
                    locationUri: `s3://${domainBucket}/${domainPrefix}`,
                    parameters: {
                        'data_owner': domainId,
                        'data_owner_name': domainName,
                        'pii_flag': false,
                        'access_mode': mode,
                    }
                }
            }).node.addDependency(this.cdkLfAdmin);
            // Grant workflow role permissions to domain database
            new lakeformation.CfnPrincipalPermissions(this, `${id}WorkflowRoleDbAccess-${mode}`, {
                permissions: ['ALL'],
                permissionsWithGrantOption: [],
                principal: {
                    dataLakePrincipalIdentifier: this.workflowRole.roleArn,
                },
                resource: {
                    database: {
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        name: mode + '-' + CentralGovernance.DOMAIN_DATABASE_PREFIX + '-' + domainId,
                    }
                },
            }).node.addDependency(this.node.findChild(`${id}DataDomainDatabase-${mode}`));
        });
        // register the S3 location in Lake Formation and create data access role
        new lake_formation_1.LakeFormationS3Location(this, `${id}LFLocation`, {
            s3Location: {
                bucketName: domainBucket,
                objectKey: domainPrefix,
            },
            accountId: domainId,
            kmsKeyId: domainKey,
        });
        if (lfModes.includes(LfAccessControlMode.TBAC)) {
            const mode = LfAccessControlMode.TBAC;
            // Create LF tag for data domain
            new lakeformation.CfnTag(this, `${id}LfTag`, {
                tagKey: CentralGovernance.DOMAIN_TAG_KEY,
                tagValues: [domainName],
            }).node.addDependency(this.node.findChild(`${id}DataDomainDatabase-${mode}`));
            // Associate LF tag with data domain database
            new lakeformation.CfnTagAssociation(this, `${id}DbTagAssoc`, {
                resource: {
                    database: {
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        name: mode + '-' + CentralGovernance.DOMAIN_DATABASE_PREFIX + '-' + domainId,
                    }
                },
                lfTags: [
                    {
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        tagKey: CentralGovernance.DOMAIN_TAG_KEY,
                        tagValues: [domainName],
                    }
                ],
            }).node.addDependency(this.node.findChild(`${id}LfTag`));
            // share data domain tag with domainId account
            new lakeformation.CfnPrincipalPermissions(this, `grantDataDomainTag`, {
                permissions: ['ASSOCIATE'],
                permissionsWithGrantOption: ['ASSOCIATE'],
                principal: {
                    dataLakePrincipalIdentifier: domainId,
                },
                resource: {
                    lfTag: {
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        tagKey: CentralGovernance.DOMAIN_TAG_KEY,
                        tagValues: [domainName]
                    }
                },
            }).node.addDependency(this.node.findChild(`${id}LfTag`));
            // create LF tag policy for table resource
            new lakeformation.CfnPrincipalPermissions(this, `LFPolicyTable`, {
                permissions: ['ALL'],
                permissionsWithGrantOption: ['ALL'],
                principal: {
                    dataLakePrincipalIdentifier: domainId,
                },
                resource: {
                    lfTagPolicy: {
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        resourceType: 'TABLE',
                        expression: [{
                                tagKey: CentralGovernance.DOMAIN_TAG_KEY,
                                tagValues: [domainName]
                            }]
                    }
                },
            }).node.addDependency(this.node.findChild(`${id}LfTag`));
            // create LF tag policy for database resource
            new lakeformation.CfnPrincipalPermissions(this, `LFPolicyDatabase`, {
                permissions: ['CREATE_TABLE', 'DESCRIBE'],
                permissionsWithGrantOption: ['CREATE_TABLE', 'DESCRIBE'],
                principal: {
                    dataLakePrincipalIdentifier: domainId,
                },
                resource: {
                    lfTagPolicy: {
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        resourceType: 'DATABASE',
                        expression: [{
                                tagKey: CentralGovernance.DOMAIN_TAG_KEY,
                                tagValues: [domainName]
                            }]
                    }
                },
            }).node.addDependency(this.node.findChild(`${id}LfTag`));
            if (this.lfTags) {
                var shareTagsDependency = new constructs_1.DependencyGroup();
                this.lfTags.forEach(tag => shareTagsDependency.add(this.node.findChild(`CentralLfTag${tag.key}`)));
                // Share all tags with domainId account
                this.lfTags.forEach(tag => {
                    new lakeformation.CfnPrincipalPermissions(this, `grantDataDomainTag${tag.key}`, {
                        permissions: ['ASSOCIATE'],
                        permissionsWithGrantOption: ['ASSOCIATE'],
                        principal: {
                            dataLakePrincipalIdentifier: domainId,
                        },
                        resource: {
                            lfTag: {
                                catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                                tagKey: tag.key,
                                tagValues: tag.values
                            }
                        },
                    }).node.addDependency(shareTagsDependency);
                });
            }
        }
        // Cross-account policy to allow Data Domain account to send events to Central Gov. account event bus
        new aws_events_1.CfnEventBusPolicy(this, `${domainName}Policy`, {
            eventBusName: this.eventBus.eventBusName,
            statementId: `AllowDataDomainAccToPutEvents_${domainId}`,
            action: 'events:PutEvents',
            principal: domainId,
        });
        // Event Bridge Rule to trigger createResourceLinks workflow in target Data Domain account
        const rule = new aws_events_1.Rule(this, `${id}Rule`, {
            eventPattern: {
                source: ['com.central.stepfunction'],
                detailType: [`${domainId}_createResourceLinks`],
            },
            eventBus: this.eventBus,
        });
        rule.addTarget(new targets.EventBus(aws_events_1.EventBus.fromEventBusArn(this, `${id}DomainEventBus`, dataDomainBusArn)));
        rule.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.DESTROY);
    }
}
_a = JSII_RTTI_SYMBOL_1;
CentralGovernance[_a] = { fqn: "aws-analytics-reference-architecture.CentralGovernance", version: "2.12.6" };
CentralGovernance.CENTRAL_BUS_NAME = 'central-mesh-bus';
CentralGovernance.DOMAIN_DATABASE_PREFIX = 'data-domain';
CentralGovernance.DOMAIN_TAG_KEY = 'LoB';
exports.CentralGovernance = CentralGovernance;
//# sourceMappingURL=data:application/json;base64,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