"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataLakeStorage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const ara_bucket_1 = require("./ara-bucket");
const context_options_1 = require("./common/context-options");
const tracked_construct_1 = require("./common/tracked-construct");
/**
 * A CDK Construct that creates the storage layers of a data lake composed of Amazon S3 Buckets.
 *
 * This construct is based on 3 Amazon S3 buckets configured with AWS best practices:
 *  * S3 buckets for Raw/Cleaned/Transformed data,
 *  * data lifecycle optimization/transitioning to different Amazon S3 storage classes
 *  * server side buckets encryption managed by KMS customer key
 *  * Default single KMS key
 *  * SSL communication enforcement
 *  * access logged to an S3 bucket
 *  * All public access blocked
 *
 * By default the transitioning rules to Amazon S3 storage classes are configured as following:
 *  * Raw data is moved to Infrequent Access after 30 days and archived to Glacier after 90 days
 *  * Clean and Transformed data is moved to Infrequent Access after 90 days and is not archived
 *
 * Objects and buckets are automatically deleted when the CDK application is detroyed.
 *
 * For custom requirements, consider using {@link AraBucket}.
 *
 * Usage example:
 * ```typescript
 * import * as cdk from 'aws-cdk-lib';
 * import { DataLakeStorage } from 'aws-analytics-reference-architecture';
 *
 * const exampleApp = new cdk.App();
 * const stack = new cdk.Stack(exampleApp, 'DataLakeStorageStack');
 *
 * new DataLakeStorage(stack, 'MyDataLakeStorage', {
 *  rawInfrequentAccessDelay: 90,
 *  rawArchiveDelay: 180,
 *  cleanInfrequentAccessDelay: 180,
 *  cleanArchiveDelay: 360,
 *  transformInfrequentAccessDelay: 180,
 *  transformArchiveDelay: 360,
 * });
 * ```
 */
class DataLakeStorage extends tracked_construct_1.TrackedConstruct {
    /**
     * Construct a new instance of DataLakeStorage based on Amazon S3 buckets with best practices configuration
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataLakeStorageProps} props the DataLakeStorageProps properties
     * @access public
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingCode: context_options_1.ContextOptions.DATA_LAKE_ID,
        };
        super(scope, id, trackedConstructProps);
        var rawInfrequentAccessDelay = 30;
        var rawArchiveDelay = 90;
        var cleanInfrequentAccessDelay = 90;
        var cleanArchiveDelay = undefined;
        var transformInfrequentAccessDelay = 90;
        var transformArchiveDelay = undefined;
        if (props) {
            if (props.rawInfrequentAccessDelay) {
                if (props.rawInfrequentAccessDelay < 30) {
                    throw new Error('Transitioning to infrequent access storage class cannot be done before 30 days');
                }
                else {
                    rawInfrequentAccessDelay = props.rawInfrequentAccessDelay;
                }
            }
            if (props.rawArchiveDelay) {
                if (props.rawArchiveDelay < 90) {
                    throw new Error('Archiving to glacier storage class cannot be done before 90 days');
                }
                else {
                    rawArchiveDelay = props.rawArchiveDelay;
                }
            }
            if (props.cleanInfrequentAccessDelay) {
                if (props.cleanInfrequentAccessDelay < 30) {
                    throw new Error('Transitioning to infrequent access storage class cannot be done before 30 days');
                }
                else {
                    cleanInfrequentAccessDelay = props.cleanInfrequentAccessDelay;
                }
            }
            if (props.cleanArchiveDelay) {
                if (props.cleanArchiveDelay < 90) {
                    throw new Error('Archiving to glacier storage class cannot be done before 90 days');
                }
                else {
                    cleanArchiveDelay = props.cleanArchiveDelay;
                }
            }
            if (props.transformInfrequentAccessDelay) {
                if (props.transformInfrequentAccessDelay < 30) {
                    throw new Error('Transitioning to infrequent access storage class cannot be done before 30 days');
                }
                else {
                    transformInfrequentAccessDelay = props.transformInfrequentAccessDelay;
                }
            }
            if (props.transformArchiveDelay) {
                if (props.transformArchiveDelay < 90) {
                    throw new Error('Archiving to glacier storage class cannot be done before 90 days');
                }
                else {
                    transformArchiveDelay = props.transformArchiveDelay;
                }
            }
        }
        // Prepare Amazon S3 Lifecycle Rules for raw data
        const rawTransitions = [
            {
                storageClass: aws_s3_1.StorageClass.INFREQUENT_ACCESS,
                transitionAfter: aws_cdk_lib_1.Duration.days(rawInfrequentAccessDelay),
            },
            {
                storageClass: aws_s3_1.StorageClass.GLACIER,
                transitionAfter: aws_cdk_lib_1.Duration.days(rawArchiveDelay),
            },
        ];
        // Create the raw data bucket with the raw transitions
        this.rawBucket = ara_bucket_1.AraBucket.getOrCreate(this, {
            encryption: aws_s3_1.BucketEncryption.KMS,
            bucketName: 'raw',
            lifecycleRules: [
                {
                    transitions: rawTransitions,
                    abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(1),
                },
            ],
            serverAccessLogsPrefix: 'raw-bucket',
        });
        // Prepare Amazon S3 Lifecycle Rules for clean data
        const cleanTransitions = [
            {
                storageClass: aws_s3_1.StorageClass.INFREQUENT_ACCESS,
                transitionAfter: aws_cdk_lib_1.Duration.days(cleanInfrequentAccessDelay),
            },
        ];
        if (cleanArchiveDelay) {
            cleanTransitions.push({
                storageClass: aws_s3_1.StorageClass.GLACIER,
                transitionAfter: aws_cdk_lib_1.Duration.days(cleanArchiveDelay),
            });
        }
        // Create the clean data bucket
        this.cleanBucket = ara_bucket_1.AraBucket.getOrCreate(this, {
            encryption: aws_s3_1.BucketEncryption.KMS,
            bucketName: 'clean',
            lifecycleRules: [
                {
                    transitions: cleanTransitions,
                    abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(1),
                },
            ],
            serverAccessLogsPrefix: 'clean-bucket',
        });
        // Prepare Amazon S3 Lifecycle Rules for clean data
        const transformTransitions = [
            {
                storageClass: aws_s3_1.StorageClass.INFREQUENT_ACCESS,
                transitionAfter: aws_cdk_lib_1.Duration.days(transformInfrequentAccessDelay),
            },
        ];
        if (transformArchiveDelay) {
            transformTransitions.push({
                storageClass: aws_s3_1.StorageClass.GLACIER,
                transitionAfter: aws_cdk_lib_1.Duration.days(transformArchiveDelay),
            });
        }
        // Create the transform data bucket
        this.transformBucket = ara_bucket_1.AraBucket.getOrCreate(this, {
            encryption: aws_s3_1.BucketEncryption.KMS,
            bucketName: 'transform',
            lifecycleRules: [
                {
                    transitions: transformTransitions,
                    abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(1),
                },
            ],
            serverAccessLogsPrefix: 'transform-bucket',
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
DataLakeStorage[_a] = { fqn: "aws-analytics-reference-architecture.DataLakeStorage", version: "2.12.6" };
exports.DataLakeStorage = DataLakeStorage;
//# sourceMappingURL=data:application/json;base64,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