"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataLakeExporter = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kinesisfirehose_1 = require("aws-cdk-lib/aws-kinesisfirehose");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const context_options_1 = require("./common/context-options");
const tracked_construct_1 = require("./common/tracked-construct");
/**
 * DataLakeExporter Construct to export data from a stream to the data lake.
 * Source can be an Amazon Kinesis Data Stream.
 * Target can be an Amazon S3 bucket.
 */
class DataLakeExporter extends tracked_construct_1.TrackedConstruct {
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingCode: context_options_1.ContextOptions.DATA_LAKE_ID,
        };
        super(scope, id, trackedConstructProps);
        if (props.deliverySize || 128 > 128) {
            throw 'deliverySize cannot be more than 128MB';
        }
        if (props.deliveryInterval || 900 > 900) {
            throw 'deliveryInterval cannot be more than 900s';
        }
        // const stack = Stack.of(this);
        // Create log group for storing Amazon Kinesis Firehose logs.
        const logGroup = new aws_logs_1.LogGroup(this, 'dataLakeExporterLogGroup', {
            logGroupName: '/aws/data-lake-exporter/',
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            retention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        // Create the Kinesis Firehose log stream.
        const firehoseLogStream = new aws_logs_1.LogStream(this, 'dataLakeExporterLogStream', {
            logGroup: logGroup,
            logStreamName: 'firehose-stream',
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        const policyDocumentKinesisFirehose = new aws_iam_1.PolicyDocument({
            statements: [
                new aws_iam_1.PolicyStatement({
                    resources: [
                        `${logGroup.logGroupArn}:log-stream:${firehoseLogStream.logStreamName}`,
                    ],
                    actions: [
                        'logs:PutLogEvents',
                    ],
                }),
            ],
        });
        const managedPolicyKinesisFirehose = new aws_iam_1.ManagedPolicy(this, 'managedPolicyKinesisFirehose', {
            document: policyDocumentKinesisFirehose,
        });
        // Create an Amazon IAM Role used by Amazon Kinesis Firehose delivery stream
        const roleKinesisFirehose = new aws_iam_1.Role(this, 'dataLakeExporterRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('firehose.amazonaws.com'),
            managedPolicies: [managedPolicyKinesisFirehose],
        });
        roleKinesisFirehose.node.addDependency(managedPolicyKinesisFirehose);
        const grantSink = props.sinkBucket.grantWrite(roleKinesisFirehose);
        const grantSource = props.sourceKinesisDataStream.grantRead(roleKinesisFirehose);
        const grantTable = props.sourceGlueTable.grantRead(roleKinesisFirehose);
        const grantGlue = props.sourceGlueTable.grantToUnderlyingResources(roleKinesisFirehose, ['glue:GetTableVersions']);
        // Create the Delivery stream from Cfn because L2 Construct doesn't support conversion to parquet and custom partitioning
        this.cfnIngestionStream = new aws_kinesisfirehose_1.CfnDeliveryStream(this, 'dataLakeExporter', {
            deliveryStreamType: 'KinesisStreamAsSource',
            extendedS3DestinationConfiguration: {
                bucketArn: props.sinkBucket.bucketArn,
                bufferingHints: {
                    intervalInSeconds: props.deliveryInterval || 900,
                    sizeInMBs: props.deliverySize || 128,
                },
                cloudWatchLoggingOptions: {
                    logGroupName: logGroup.logGroupName,
                    logStreamName: firehoseLogStream.logStreamName,
                },
                roleArn: roleKinesisFirehose.roleArn,
                errorOutputPrefix: `${props.sinkObjectKey}-error`,
                prefix: props.sinkObjectKey,
                compressionFormat: 'UNCOMPRESSED',
                s3BackupMode: 'Disabled',
                dataFormatConversionConfiguration: {
                    enabled: true,
                    inputFormatConfiguration: {
                        deserializer: {
                            openXJsonSerDe: {},
                        },
                    },
                    outputFormatConfiguration: {
                        serializer: {
                            parquetSerDe: {},
                        },
                    },
                    schemaConfiguration: {
                        roleArn: roleKinesisFirehose.roleArn,
                        catalogId: aws_cdk_lib_1.Aws.ACCOUNT_ID,
                        region: aws_cdk_lib_1.Aws.REGION,
                        databaseName: props.sourceGlueDatabase.databaseName,
                        tableName: props.sourceGlueTable.tableName,
                    },
                },
            },
            kinesisStreamSourceConfiguration: {
                kinesisStreamArn: props.sourceKinesisDataStream.streamArn,
                roleArn: roleKinesisFirehose.roleArn,
            },
        });
        // Need to enforce a dependancy because the grant methods generate an IAM Policy without dependency on the Firehose
        this.cfnIngestionStream.node.addDependency(grantSink);
        this.cfnIngestionStream.node.addDependency(grantSource);
        this.cfnIngestionStream.node.addDependency(grantTable);
        this.cfnIngestionStream.node.addDependency(grantGlue);
    }
}
_a = JSII_RTTI_SYMBOL_1;
DataLakeExporter[_a] = { fqn: "aws-analytics-reference-architecture.DataLakeExporter", version: "2.12.6" };
exports.DataLakeExporter = DataLakeExporter;
//# sourceMappingURL=data:application/json;base64,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