"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomDataset = exports.CustomDatasetInputFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const pre_bundled_pyspark_job_executable_1 = require("../common/pre-bundled-pyspark-job-executable");
const aws_glue_alpha_1 = require("@aws-cdk/aws-glue-alpha");
const prepared_dataset_1 = require("./prepared-dataset");
const constructs_1 = require("constructs");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const ara_bucket_1 = require("../ara-bucket");
const synchronous_glue_job_1 = require("../synchronous-glue-job");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const custom_resources_1 = require("aws-cdk-lib/custom-resources");
const singleton_kms_key_1 = require("../singleton-kms-key");
var CustomDatasetInputFormat;
(function (CustomDatasetInputFormat) {
    CustomDatasetInputFormat["CSV"] = "csv";
    CustomDatasetInputFormat["PARQUET"] = "parquet";
    CustomDatasetInputFormat["JSON"] = "json";
})(CustomDatasetInputFormat = exports.CustomDatasetInputFormat || (exports.CustomDatasetInputFormat = {}));
/**
 * A CustomDataset is a dataset that you need to prepare for the [BatchReplayer](@link BatchReplayer) to generate data.
 * The dataset is transformed into a [PreparedDataset](@link PreparedDataset) by a Glue Job that runs synchronously during the CDK deploy.
 * The Glue job is sized based on the approximate size of the input data or uses autoscaling (max 100) if no data size is provided.
 *
 * The Glue job is applying the following transformations to the input dataset:
 * 1. Read the input dataset based on its format. Currently, it supports data in CSV, JSON and Parquet
 * 2. Group rows into tumbling windows based on the partition range parameter provided.
 * The partition range should be adapted to the data volume and the total dataset time range
 * 3. Convert dates from MM-dd-yyyy HH:mm:ss.SSS to MM-dd-yyyyTHH:mm:ss.SSSZ format and remove null values
 * 4. Write data into the output bucket partitioned by the tumbling window time.
 * For example, one partition for every 5 minutes.
 * 5. Generate a manifest file based on the previous output to be used by the BatchReplayer for generating data
 *
 * The CloudWatch log group is stored as an object parameter to help check any error with the Glue job.
 *
 * Usage example:
 * ```typescript
 * import { CustomDataset, CustomDatasetInputFormat } from './data-generator/custom-dataset';
 *
 * const app = new App();
 * const stack = new Stack(app, 'CustomDatasetStack');
 *
 * const custom = new CustomDataset(stack, 'CustomDataset', {
 *   s3Location: {
 *     bucketName: 'aws-analytics-reference-architecture',
 *     objectKey: 'datasets/custom',
 *   },
 *   inputFormat: CustomDatasetInputFormat.CSV,
 *   datetimeColumn: 'tpep_pickup_datetime',
 *   datetimeColumnsToAdjust: ['tpep_pickup_datetime'],
 *   partitionRange: Duration.minutes(5),
 *   approximateDataSize: 1,
 * });
 *
 * new CfnOutput(this, 'LogGroupName', {
 *   exportName: 'logGroupName,
 *   value: custom.glueJobLogGroup,
 * });
 * ```
 *
 * An example of a custom dataset that can be processed by this construct is available in s3://aws-analytics-reference-architecture/datasets/custom
 */
class CustomDataset extends constructs_1.Construct {
    /**
     * Constructs a new instance of a CustomDataset construct that extends a PreparedDataset
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {CustomDatasetProps} props the CustomDataset [properties]{@link CustomDatasetProps}
     */
    constructor(scope, id, props) {
        super(scope, id);
        // the source bucket to read data from
        const sourceBucket = aws_s3_1.Bucket.fromBucketAttributes(scope, 'InputBucket', {
            bucketName: props.s3Location.bucketName,
        });
        // the sink bucket used by the Glue job to write the prepared dataset
        const outputBucket = ara_bucket_1.AraBucket.getOrCreate(scope, {
            bucketName: 'custom-dataset',
            serverAccessLogsPrefix: 'custom-dataset',
        });
        const glueRole = new aws_iam_1.Role(scope, 'CustomDatasetRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
            // add inline policy to encrypt logs
            inlinePolicies: {
                SecurityConfig: new aws_iam_1.PolicyDocument({
                    statements: [
                        new aws_iam_1.PolicyStatement({
                            actions: [
                                "logs:AssociateKmsKey"
                            ],
                            resources: [
                                `arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group:/aws-glue/jobs/*`
                            ],
                        })
                    ]
                })
            }
        });
        // grant permissions on the eS3 buckets to the glue job role
        sourceBucket.grantRead(glueRole, props.s3Location.objectKey + '*');
        outputBucket.grantReadWrite(glueRole, props.s3Location.objectKey + '*');
        // the SSM parameter used to stored the output of the Glue job
        const ssm = new aws_ssm_1.StringParameter(this, 'JobOutputParameter', {
            stringValue: 'minDatetime',
        });
        ssm.grantWrite(glueRole);
        glueRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSGlueServiceRole'));
        // create a KMS key for the Glue security configureation
        const glueKey = singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey');
        // We add a resource policy so the key can be used in Cloudwatch logs
        glueKey.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            principals: [
                new aws_iam_1.ServicePrincipal(`logs.${aws_cdk_lib_1.Aws.REGION}.amazonaws.com`)
            ],
            actions: [
                "kms:Encrypt*",
                "kms:Decrypt*",
                "kms:ReEncrypt*",
                "kms:GenerateDataKey*",
                "kms:Describe*"
            ],
            resources: ["*"],
            conditions: { ArnLike: { "kms:EncryptionContext:aws:logs:arn": `arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*` } },
        }));
        // the Glue job security configuration following best practices
        const conf = new aws_glue_alpha_1.SecurityConfiguration(this, 'SecurityConfiguration', {
            securityConfigurationName: id,
            cloudWatchEncryption: {
                mode: aws_glue_alpha_1.CloudWatchEncryptionMode.KMS,
                kmsKey: glueKey,
            },
            jobBookmarksEncryption: {
                mode: aws_glue_alpha_1.JobBookmarksEncryptionMode.CLIENT_SIDE_KMS,
                kmsKey: glueKey,
            },
            s3Encryption: {
                mode: aws_glue_alpha_1.S3EncryptionMode.S3_MANAGED,
            }
        });
        // calculate the size of the Glue job based on input data size (1 + 1 DPU per 2GB)
        var workerNum = 9;
        if (props.approximateDataSize !== undefined && props.approximateDataSize > 16) {
            workerNum = Math.ceil(props.approximateDataSize / 2) + 1;
        }
        // Glue job to prepare the dataset
        const glueJob = new synchronous_glue_job_1.SynchronousGlueJob(scope, 'CustomDatasetJob', {
            executable: pre_bundled_pyspark_job_executable_1.PreBundledPysparkJobExecutable.pythonEtl({
                glueVersion: aws_glue_alpha_1.GlueVersion.V3_0,
                pythonVersion: aws_glue_alpha_1.PythonVersion.THREE,
                codePath: 'data-generator/resources/glue/custom-dataset/script.py',
            }),
            defaultArguments: {
                '--s3_input_path': `s3://${props.s3Location.bucketName}/${props.s3Location.objectKey}`,
                '--s3_output_path': `s3://${outputBucket.bucketName}/${props.s3Location.objectKey}`,
                '--input_format': props.inputFormat,
                '--datetime_column': props.datetimeColumn,
                '--partition_range': props.partitionRange.toMinutes().toString(),
                '--ssm_parameter': ssm.parameterName,
                '--enable-auto-scaling': 'true',
            },
            role: glueRole,
            workerCount: props.approximateDataSize ? workerNum : 100,
            workerType: aws_glue_alpha_1.WorkerType.G_1X,
            continuousLogging: {
                enabled: true,
            },
            securityConfiguration: conf,
        });
        this.glueJobLogGroup = glueJob.glueJobLogStream;
        // Get the offset value calculated in the SynchronousGlueJob
        // We cannot rely on the SSM parameter resource created previously because the offset is generated during deploy time
        const getParameter = new custom_resources_1.AwsCustomResource(this, 'GetParameter', {
            onCreate: {
                service: 'SSM',
                action: 'getParameter',
                parameters: {
                    Name: ssm.parameterName,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.of(Date.now().toString()),
            },
            policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({
                resources: [ssm.parameterArn],
            })
        });
        // Add a dependency on the synchronous glue job to only get the value after processing
        getParameter.node.addDependency(glueJob);
        // Create a prepared dataset based on the output of the Glue job
        this.preparedDataset = new prepared_dataset_1.PreparedDataset({
            location: {
                bucketName: outputBucket.bucketName,
                objectKey: props.s3Location.objectKey,
            },
            offset: getParameter.getResponseField('Parameter.Value'),
            manifestLocation: {
                bucketName: outputBucket.bucketName,
                objectKey: props.s3Location.objectKey + '-manifest.csv',
            },
            dateTimeColumnToFilter: props.datetimeColumn,
            dateTimeColumnsToAdjust: props.datetimeColumnsToAdjust,
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
CustomDataset[_a] = { fqn: "aws-analytics-reference-architecture.CustomDataset", version: "2.12.6" };
exports.CustomDataset = CustomDataset;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3VzdG9tLWRhdGFzZXQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi9zcmMvZGF0YS1nZW5lcmF0b3IvY3VzdG9tLWRhdGFzZXQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxxRUFBcUU7QUFDckUsaUNBQWlDO0FBRWpDLHFHQUE4RjtBQUM5Riw0REFBZ0w7QUFDaEwseURBQXFEO0FBQ3JELDJDQUF1QztBQUN2QyxpREFBNkc7QUFDN0csK0NBQXNEO0FBQ3RELDhDQUEwQztBQUMxQyxrRUFBNkQ7QUFDN0QsaURBQXNEO0FBQ3RELDZDQUE0QztBQUM1QyxtRUFBOEc7QUFDOUcsNERBQW9EO0FBa0NwRCxJQUFZLHdCQUlYO0FBSkQsV0FBWSx3QkFBd0I7SUFDbEMsdUNBQVcsQ0FBQTtJQUNYLCtDQUFtQixDQUFBO0lBQ25CLHlDQUFhLENBQUE7QUFDZixDQUFDLEVBSlcsd0JBQXdCLEdBQXhCLGdDQUF3QixLQUF4QixnQ0FBd0IsUUFJbkM7QUFFRDs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O0dBMENHO0FBQ0gsTUFBYSxhQUFjLFNBQVEsc0JBQVM7SUFXMUM7Ozs7O09BS0c7SUFDSCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXlCO1FBRWpFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsc0NBQXNDO1FBQ3RDLE1BQU0sWUFBWSxHQUFHLGVBQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsYUFBYSxFQUFFO1lBQ3JFLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVSxDQUFDLFVBQVU7U0FDeEMsQ0FBQyxDQUFBO1FBRUYscUVBQXFFO1FBQ3JFLE1BQU0sWUFBWSxHQUFHLHNCQUFTLENBQUMsV0FBVyxDQUFDLEtBQUssRUFBRTtZQUNoRCxVQUFVLEVBQUUsZ0JBQWdCO1lBQzVCLHNCQUFzQixFQUFFLGdCQUFnQjtTQUN6QyxDQUFDLENBQUM7UUFFSCxNQUFNLFFBQVEsR0FBRyxJQUFJLGNBQUksQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUU7WUFDcEQsU0FBUyxFQUFFLElBQUksMEJBQWdCLENBQUMsb0JBQW9CLENBQUM7WUFDckQsb0NBQW9DO1lBQ3BDLGNBQWMsRUFBRTtnQkFDZCxjQUFjLEVBQUUsSUFBSSx3QkFBYyxDQUFDO29CQUNqQyxVQUFVLEVBQUU7d0JBQ1YsSUFBSSx5QkFBZSxDQUFDOzRCQUNsQixPQUFPLEVBQUU7Z0NBQ1Asc0JBQXNCOzZCQUN2Qjs0QkFDRCxTQUFTLEVBQUU7Z0NBQ1QsZ0JBQWdCLGlCQUFHLENBQUMsTUFBTSxJQUFJLGlCQUFHLENBQUMsVUFBVSw2QkFBNkI7NkJBQzFFO3lCQUNGLENBQUM7cUJBQ0g7aUJBQ0YsQ0FBQzthQUNIO1NBQ0YsQ0FBQyxDQUFDO1FBRUgsNERBQTREO1FBQzVELFlBQVksQ0FBQyxTQUFTLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsU0FBUyxHQUFHLEdBQUcsQ0FBQyxDQUFDO1FBQ25FLFlBQVksQ0FBQyxjQUFjLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsU0FBUyxHQUFHLEdBQUcsQ0FBQyxDQUFDO1FBRXhFLDhEQUE4RDtRQUM5RCxNQUFNLEdBQUcsR0FBRyxJQUFJLHlCQUFlLENBQUMsSUFBSSxFQUFFLG9CQUFvQixFQUFFO1lBQzFELFdBQVcsRUFBRSxhQUFhO1NBQzNCLENBQUMsQ0FBQztRQUNILEdBQUcsQ0FBQyxVQUFVLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFekIsUUFBUSxDQUFDLGdCQUFnQixDQUFDLHVCQUFhLENBQUMsd0JBQXdCLENBQUMsaUNBQWlDLENBQUMsQ0FBQyxDQUFDO1FBRXJHLHdEQUF3RDtRQUN4RCxNQUFNLE9BQU8sR0FBSSxnQ0FBWSxDQUFDLFdBQVcsQ0FBQyxLQUFLLEVBQUUsZUFBZSxDQUFDLENBQUE7UUFFakUscUVBQXFFO1FBQ3JFLE9BQU8sQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLHlCQUFlLENBQUM7WUFDOUMsVUFBVSxFQUFFO2dCQUNWLElBQUksMEJBQWdCLENBQUMsUUFBUSxpQkFBRyxDQUFDLE1BQU0sZ0JBQWdCLENBQUM7YUFDekQ7WUFDRCxPQUFPLEVBQUU7Z0JBQ1AsY0FBYztnQkFDZCxjQUFjO2dCQUNkLGdCQUFnQjtnQkFDaEIsc0JBQXNCO2dCQUN0QixlQUFlO2FBQ2hCO1lBQ0QsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1lBQ2hCLFVBQVUsRUFBRSxFQUFFLE9BQU8sRUFBRSxFQUFFLG9DQUFvQyxFQUFFLGdCQUFnQixpQkFBRyxDQUFDLE1BQU0sSUFBSSxpQkFBRyxDQUFDLFVBQVUsSUFBSSxFQUFFLEVBQUU7U0FDcEgsQ0FBQyxDQUFDLENBQUE7UUFFSCwrREFBK0Q7UUFDL0QsTUFBTSxJQUFJLEdBQUcsSUFBSSxzQ0FBcUIsQ0FBQyxJQUFJLEVBQUUsdUJBQXVCLEVBQUU7WUFDcEUseUJBQXlCLEVBQUUsRUFBRTtZQUM3QixvQkFBb0IsRUFBRTtnQkFDcEIsSUFBSSxFQUFFLHlDQUF3QixDQUFDLEdBQUc7Z0JBQ2xDLE1BQU0sRUFBRSxPQUFPO2FBQ2hCO1lBQ0Qsc0JBQXNCLEVBQUU7Z0JBQ3RCLElBQUksRUFBRSwyQ0FBMEIsQ0FBQyxlQUFlO2dCQUNoRCxNQUFNLEVBQUUsT0FBTzthQUNoQjtZQUNELFlBQVksRUFBRTtnQkFDWixJQUFJLEVBQUUsaUNBQWdCLENBQUMsVUFBVTthQUNsQztTQUNGLENBQUMsQ0FBQztRQUVILGtGQUFrRjtRQUNsRixJQUFJLFNBQVMsR0FBRyxDQUFDLENBQUM7UUFDbEIsSUFBSSxLQUFLLENBQUMsbUJBQW1CLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxtQkFBbUIsR0FBRyxFQUFFLEVBQUM7WUFDNUUsU0FBUyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLG1CQUFtQixHQUFHLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUMxRDtRQUVELGtDQUFrQztRQUNsQyxNQUFNLE9BQU8sR0FBRyxJQUFJLHlDQUFrQixDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTtZQUNoRSxVQUFVLEVBQUUsbUVBQThCLENBQUMsU0FBUyxDQUFDO2dCQUNuRCxXQUFXLEVBQUUsNEJBQVcsQ0FBQyxJQUFJO2dCQUM3QixhQUFhLEVBQUUsOEJBQWEsQ0FBQyxLQUFLO2dCQUNsQyxRQUFRLEVBQUUsd0RBQXdEO2FBQ25FLENBQUM7WUFDRixnQkFBZ0IsRUFBRTtnQkFDaEIsaUJBQWlCLEVBQUUsUUFBUSxLQUFLLENBQUMsVUFBVSxDQUFDLFVBQVUsSUFBSSxLQUFLLENBQUMsVUFBVSxDQUFDLFNBQVMsRUFBRTtnQkFDdEYsa0JBQWtCLEVBQUUsUUFBUSxZQUFZLENBQUMsVUFBVSxJQUFJLEtBQUssQ0FBQyxVQUFVLENBQUMsU0FBUyxFQUFFO2dCQUNuRixnQkFBZ0IsRUFBRSxLQUFLLENBQUMsV0FBVztnQkFDbkMsbUJBQW1CLEVBQUUsS0FBSyxDQUFDLGNBQWM7Z0JBQ3pDLG1CQUFtQixFQUFFLEtBQUssQ0FBQyxjQUFjLENBQUMsU0FBUyxFQUFFLENBQUMsUUFBUSxFQUFFO2dCQUNoRSxpQkFBaUIsRUFBRSxHQUFHLENBQUMsYUFBYTtnQkFDcEMsdUJBQXVCLEVBQUUsTUFBTTthQUNoQztZQUNELElBQUksRUFBRSxRQUFRO1lBQ2QsV0FBVyxFQUFFLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxHQUFHO1lBQ3hELFVBQVUsRUFBRSwyQkFBVSxDQUFDLElBQUk7WUFDM0IsaUJBQWlCLEVBQUM7Z0JBQ2hCLE9BQU8sRUFBRSxJQUFJO2FBQ2Q7WUFDRCxxQkFBcUIsRUFBRSxJQUFJO1NBQzVCLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxlQUFlLEdBQUcsT0FBTyxDQUFDLGdCQUFnQixDQUFDO1FBRWhELDREQUE0RDtRQUM1RCxxSEFBcUg7UUFDckgsTUFBTSxZQUFZLEdBQUcsSUFBSSxvQ0FBaUIsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQy9ELFFBQVEsRUFBRTtnQkFDUixPQUFPLEVBQUUsS0FBSztnQkFDZCxNQUFNLEVBQUUsY0FBYztnQkFDdEIsVUFBVSxFQUFFO29CQUNWLElBQUksRUFBRSxHQUFHLENBQUMsYUFBYTtpQkFDeEI7Z0JBQ0Qsa0JBQWtCLEVBQUUscUNBQWtCLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQyxRQUFRLEVBQUUsQ0FBQzthQUNqRTtZQUNELE1BQU0sRUFBRSwwQ0FBdUIsQ0FBQyxZQUFZLENBQUM7Z0JBQzNDLFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUM7YUFDOUIsQ0FBQztTQUNILENBQUMsQ0FBQztRQUNILHNGQUFzRjtRQUN0RixZQUFZLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUV6QyxnRUFBZ0U7UUFDaEUsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLGtDQUFlLENBQUM7WUFDekMsUUFBUSxFQUFFO2dCQUNSLFVBQVUsRUFBRSxZQUFZLENBQUMsVUFBVTtnQkFDbkMsU0FBUyxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsU0FBUzthQUN0QztZQUNELE1BQU0sRUFBRSxZQUFZLENBQUMsZ0JBQWdCLENBQUMsaUJBQWlCLENBQUM7WUFDeEQsZ0JBQWdCLEVBQUU7Z0JBQ2hCLFVBQVUsRUFBRSxZQUFZLENBQUMsVUFBVTtnQkFDbkMsU0FBUyxFQUFFLEtBQUssQ0FBQyxVQUFVLENBQUMsU0FBUyxHQUFDLGVBQWU7YUFDdEQ7WUFDRCxzQkFBc0IsRUFBRSxLQUFLLENBQUMsY0FBYztZQUM1Qyx1QkFBdUIsRUFBRSxLQUFLLENBQUMsdUJBQXVCO1NBQ3ZELENBQUMsQ0FBQztJQUNMLENBQUM7Ozs7QUFuS1Usc0NBQWEiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbi8vIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBNSVQtMFxuXG5pbXBvcnQgeyBQcmVCdW5kbGVkUHlzcGFya0pvYkV4ZWN1dGFibGUgfSBmcm9tICcuLi9jb21tb24vcHJlLWJ1bmRsZWQtcHlzcGFyay1qb2ItZXhlY3V0YWJsZSc7XG5pbXBvcnQgeyBDbG91ZFdhdGNoRW5jcnlwdGlvbk1vZGUsIEdsdWVWZXJzaW9uLCBKb2JCb29rbWFya3NFbmNyeXB0aW9uTW9kZSwgUHl0aG9uVmVyc2lvbiwgUzNFbmNyeXB0aW9uTW9kZSwgU2VjdXJpdHlDb25maWd1cmF0aW9uLCBXb3JrZXJUeXBlIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWdsdWUtYWxwaGEnO1xuaW1wb3J0IHsgUHJlcGFyZWREYXRhc2V0IH0gZnJvbSAnLi9wcmVwYXJlZC1kYXRhc2V0JztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgTWFuYWdlZFBvbGljeSwgUG9saWN5RG9jdW1lbnQsIFBvbGljeVN0YXRlbWVudCwgUm9sZSwgU2VydmljZVByaW5jaXBhbCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0IHsgQnVja2V0LCBMb2NhdGlvbiB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zMyc7XG5pbXBvcnQgeyBBcmFCdWNrZXQgfSBmcm9tICcuLi9hcmEtYnVja2V0JztcbmltcG9ydCB7IFN5bmNocm9ub3VzR2x1ZUpvYiB9IGZyb20gJy4uL3N5bmNocm9ub3VzLWdsdWUtam9iJztcbmltcG9ydCB7IFN0cmluZ1BhcmFtZXRlciB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zc20nO1xuaW1wb3J0IHsgQXdzLCBEdXJhdGlvbiB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IEF3c0N1c3RvbVJlc291cmNlLCBBd3NDdXN0b21SZXNvdXJjZVBvbGljeSwgUGh5c2ljYWxSZXNvdXJjZUlkIH0gZnJvbSAnYXdzLWNkay1saWIvY3VzdG9tLXJlc291cmNlcyc7XG5pbXBvcnQgeyBTaW5nbGV0b25LZXkgfSBmcm9tICcuLi9zaW5nbGV0b24ta21zLWtleSc7XG5cblxuLyoqXG4gKiBUaGUgcHJvcGVydGllcyBmb3IgdGhlIEJyaW5nIFlvdXIgT3duIERhdGEgZ2VuZXJhdG9yXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQ3VzdG9tRGF0YXNldFByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBTMyBsb2NhdGlvbiBvZiB0aGUgaW5wdXQgZGF0YVxuICAgKi9cbiAgcmVhZG9ubHkgczNMb2NhdGlvbjogTG9jYXRpb247XG4gIC8qKlxuICAgKiBUaGUgZm9ybWF0IG9mIHRoZSBpbnB1dCBkYXRhXG4gICAqL1xuICByZWFkb25seSBpbnB1dEZvcm1hdDogQ3VzdG9tRGF0YXNldElucHV0Rm9ybWF0O1xuICAvKipcbiAgICogVGhlIGRhdGV0aW1lIGNvbHVtbiB0byB1c2UgZm9yIGRhdGEgZ2VuZXJhdGlvbiBhcyB0aGUgdGltZSByZWZlcmVuY2VcbiAgICovXG4gIHJlYWRvbmx5IGRhdGV0aW1lQ29sdW1uOiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgZGF0ZXRpbWUgY29sdW1ucyB0byB1c2UgZm9yIGRhdGEgZ2VuZXJhdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgZGF0ZXRpbWVDb2x1bW5zVG9BZGp1c3Q6IHN0cmluZ1tdO1xuICAvKipcbiAgICogVGhlIGludGVydmFsIHRvIHBhcnRpdGlvbiBkYXRhIGFuZCBvcHRpbWl6ZSB0aGUgZGF0YSBnZW5lcmF0aW9uIGluIE1pbnV0ZXNcbiAgICovXG4gIHJlYWRvbmx5IHBhcnRpdGlvblJhbmdlOiBEdXJhdGlvbjtcbiAgLyoqXG4gICAqIEFwcHJveGltYXRlIGRhdGEgc2l6ZSAoaW4gR0IpIG9mIHRoZSBjdXN0b20gZGF0YXNldC4gXG4gICAqIEBkZWZhdWx0IC0gVGhlIEdsdWUgam9iIHJlc3BvbnNpYmxlIGZvciBwcmVwYXJpbmcgdGhlIGRhdGEgdXNlcyBhdXRvc2NhbGluZyB3aXRoIGEgbWF4aW11bSBvZiAxMDAgd29ya2Vyc1xuICAgKi9cbiAgcmVhZG9ubHkgYXBwcm94aW1hdGVEYXRhU2l6ZT86IG51bWJlcjtcbn1cblxuZXhwb3J0IGVudW0gQ3VzdG9tRGF0YXNldElucHV0Rm9ybWF0IHtcbiAgQ1NWID0gJ2NzdicsXG4gIFBBUlFVRVQgPSAncGFycXVldCcsXG4gIEpTT04gPSAnanNvbicsXG59XG5cbi8qKlxuICogQSBDdXN0b21EYXRhc2V0IGlzIGEgZGF0YXNldCB0aGF0IHlvdSBuZWVkIHRvIHByZXBhcmUgZm9yIHRoZSBbQmF0Y2hSZXBsYXllcl0oQGxpbmsgQmF0Y2hSZXBsYXllcikgdG8gZ2VuZXJhdGUgZGF0YS5cbiAqIFRoZSBkYXRhc2V0IGlzIHRyYW5zZm9ybWVkIGludG8gYSBbUHJlcGFyZWREYXRhc2V0XShAbGluayBQcmVwYXJlZERhdGFzZXQpIGJ5IGEgR2x1ZSBKb2IgdGhhdCBydW5zIHN5bmNocm9ub3VzbHkgZHVyaW5nIHRoZSBDREsgZGVwbG95LlxuICogVGhlIEdsdWUgam9iIGlzIHNpemVkIGJhc2VkIG9uIHRoZSBhcHByb3hpbWF0ZSBzaXplIG9mIHRoZSBpbnB1dCBkYXRhIG9yIHVzZXMgYXV0b3NjYWxpbmcgKG1heCAxMDApIGlmIG5vIGRhdGEgc2l6ZSBpcyBwcm92aWRlZC5cbiAqIFxuICogVGhlIEdsdWUgam9iIGlzIGFwcGx5aW5nIHRoZSBmb2xsb3dpbmcgdHJhbnNmb3JtYXRpb25zIHRvIHRoZSBpbnB1dCBkYXRhc2V0OlxuICogMS4gUmVhZCB0aGUgaW5wdXQgZGF0YXNldCBiYXNlZCBvbiBpdHMgZm9ybWF0LiBDdXJyZW50bHksIGl0IHN1cHBvcnRzIGRhdGEgaW4gQ1NWLCBKU09OIGFuZCBQYXJxdWV0XG4gKiAyLiBHcm91cCByb3dzIGludG8gdHVtYmxpbmcgd2luZG93cyBiYXNlZCBvbiB0aGUgcGFydGl0aW9uIHJhbmdlIHBhcmFtZXRlciBwcm92aWRlZC4gXG4gKiBUaGUgcGFydGl0aW9uIHJhbmdlIHNob3VsZCBiZSBhZGFwdGVkIHRvIHRoZSBkYXRhIHZvbHVtZSBhbmQgdGhlIHRvdGFsIGRhdGFzZXQgdGltZSByYW5nZVxuICogMy4gQ29udmVydCBkYXRlcyBmcm9tIE1NLWRkLXl5eXkgSEg6bW06c3MuU1NTIHRvIE1NLWRkLXl5eXlUSEg6bW06c3MuU1NTWiBmb3JtYXQgYW5kIHJlbW92ZSBudWxsIHZhbHVlc1xuICogNC4gV3JpdGUgZGF0YSBpbnRvIHRoZSBvdXRwdXQgYnVja2V0IHBhcnRpdGlvbmVkIGJ5IHRoZSB0dW1ibGluZyB3aW5kb3cgdGltZS4gXG4gKiBGb3IgZXhhbXBsZSwgb25lIHBhcnRpdGlvbiBmb3IgZXZlcnkgNSBtaW51dGVzLiBcbiAqIDUuIEdlbmVyYXRlIGEgbWFuaWZlc3QgZmlsZSBiYXNlZCBvbiB0aGUgcHJldmlvdXMgb3V0cHV0IHRvIGJlIHVzZWQgYnkgdGhlIEJhdGNoUmVwbGF5ZXIgZm9yIGdlbmVyYXRpbmcgZGF0YVxuICogXG4gKiBUaGUgQ2xvdWRXYXRjaCBsb2cgZ3JvdXAgaXMgc3RvcmVkIGFzIGFuIG9iamVjdCBwYXJhbWV0ZXIgdG8gaGVscCBjaGVjayBhbnkgZXJyb3Igd2l0aCB0aGUgR2x1ZSBqb2IuXG4gKiBcbiAqIFVzYWdlIGV4YW1wbGU6XG4gKiBgYGB0eXBlc2NyaXB0XG4gKiBpbXBvcnQgeyBDdXN0b21EYXRhc2V0LCBDdXN0b21EYXRhc2V0SW5wdXRGb3JtYXQgfSBmcm9tICcuL2RhdGEtZ2VuZXJhdG9yL2N1c3RvbS1kYXRhc2V0JztcbiAqIFxuICogY29uc3QgYXBwID0gbmV3IEFwcCgpO1xuICogY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soYXBwLCAnQ3VzdG9tRGF0YXNldFN0YWNrJyk7XG4gKiBcbiAqIGNvbnN0IGN1c3RvbSA9IG5ldyBDdXN0b21EYXRhc2V0KHN0YWNrLCAnQ3VzdG9tRGF0YXNldCcsIHtcbiAqICAgczNMb2NhdGlvbjoge1xuICogICAgIGJ1Y2tldE5hbWU6ICdhd3MtYW5hbHl0aWNzLXJlZmVyZW5jZS1hcmNoaXRlY3R1cmUnLFxuICogICAgIG9iamVjdEtleTogJ2RhdGFzZXRzL2N1c3RvbScsXG4gKiAgIH0sXG4gKiAgIGlucHV0Rm9ybWF0OiBDdXN0b21EYXRhc2V0SW5wdXRGb3JtYXQuQ1NWLFxuICogICBkYXRldGltZUNvbHVtbjogJ3RwZXBfcGlja3VwX2RhdGV0aW1lJyxcbiAqICAgZGF0ZXRpbWVDb2x1bW5zVG9BZGp1c3Q6IFsndHBlcF9waWNrdXBfZGF0ZXRpbWUnXSxcbiAqICAgcGFydGl0aW9uUmFuZ2U6IER1cmF0aW9uLm1pbnV0ZXMoNSksXG4gKiAgIGFwcHJveGltYXRlRGF0YVNpemU6IDEsXG4gKiB9KTtcbiAqIFxuICogbmV3IENmbk91dHB1dCh0aGlzLCAnTG9nR3JvdXBOYW1lJywge1xuICogICBleHBvcnROYW1lOiAnbG9nR3JvdXBOYW1lLFxuICogICB2YWx1ZTogY3VzdG9tLmdsdWVKb2JMb2dHcm91cCxcbiAqIH0pO1xuICogYGBgXG4gKiBcbiAqIEFuIGV4YW1wbGUgb2YgYSBjdXN0b20gZGF0YXNldCB0aGF0IGNhbiBiZSBwcm9jZXNzZWQgYnkgdGhpcyBjb25zdHJ1Y3QgaXMgYXZhaWxhYmxlIGluIHMzOi8vYXdzLWFuYWx5dGljcy1yZWZlcmVuY2UtYXJjaGl0ZWN0dXJlL2RhdGFzZXRzL2N1c3RvbVxuICovXG5leHBvcnQgY2xhc3MgQ3VzdG9tRGF0YXNldCBleHRlbmRzIENvbnN0cnVjdCB7XG5cbiAgLyoqXG4gICAqIFRoZSBwcmVwYXJlZCBkYXRhc2V0IGdlbmVyYXRlZCBmcm9tIHRoZSBjdXN0b20gZGF0YXNldFxuICAgKi9cbiAgcmVhZG9ubHkgcHJlcGFyZWREYXRhc2V0OiBQcmVwYXJlZERhdGFzZXQ7XG4gIC8qKlxuICAgKiBUaGUgbG9jYXRpb24gb2YgdGhlIGxvZ3MgdG8gYW5hbHl6ZSBwb3RlbnRpYWwgZXJyb3JzIGluIHRoZSBHbHVlIGpvYlxuICAgKi9cbiAgcmVhZG9ubHkgZ2x1ZUpvYkxvZ0dyb3VwOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIENvbnN0cnVjdHMgYSBuZXcgaW5zdGFuY2Ugb2YgYSBDdXN0b21EYXRhc2V0IGNvbnN0cnVjdCB0aGF0IGV4dGVuZHMgYSBQcmVwYXJlZERhdGFzZXRcbiAgICogQHBhcmFtIHtDb25zdHJ1Y3R9IHNjb3BlIHRoZSBTY29wZSBvZiB0aGUgQ0RLIENvbnN0cnVjdFxuICAgKiBAcGFyYW0ge3N0cmluZ30gaWQgdGhlIElEIG9mIHRoZSBDREsgQ29uc3RydWN0XG4gICAqIEBwYXJhbSB7Q3VzdG9tRGF0YXNldFByb3BzfSBwcm9wcyB0aGUgQ3VzdG9tRGF0YXNldCBbcHJvcGVydGllc117QGxpbmsgQ3VzdG9tRGF0YXNldFByb3BzfVxuICAgKi9cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEN1c3RvbURhdGFzZXRQcm9wcykge1xuICAgIFxuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICAvLyB0aGUgc291cmNlIGJ1Y2tldCB0byByZWFkIGRhdGEgZnJvbVxuICAgIGNvbnN0IHNvdXJjZUJ1Y2tldCA9IEJ1Y2tldC5mcm9tQnVja2V0QXR0cmlidXRlcyhzY29wZSwgJ0lucHV0QnVja2V0Jywge1xuICAgICAgYnVja2V0TmFtZTogcHJvcHMuczNMb2NhdGlvbi5idWNrZXROYW1lLFxuICAgIH0pXG5cbiAgICAvLyB0aGUgc2luayBidWNrZXQgdXNlZCBieSB0aGUgR2x1ZSBqb2IgdG8gd3JpdGUgdGhlIHByZXBhcmVkIGRhdGFzZXRcbiAgICBjb25zdCBvdXRwdXRCdWNrZXQgPSBBcmFCdWNrZXQuZ2V0T3JDcmVhdGUoc2NvcGUsIHtcbiAgICAgIGJ1Y2tldE5hbWU6ICdjdXN0b20tZGF0YXNldCcsXG4gICAgICBzZXJ2ZXJBY2Nlc3NMb2dzUHJlZml4OiAnY3VzdG9tLWRhdGFzZXQnLFxuICAgIH0pO1xuXG4gICAgY29uc3QgZ2x1ZVJvbGUgPSBuZXcgUm9sZShzY29wZSwgJ0N1c3RvbURhdGFzZXRSb2xlJywge1xuICAgICAgYXNzdW1lZEJ5OiBuZXcgU2VydmljZVByaW5jaXBhbCgnZ2x1ZS5hbWF6b25hd3MuY29tJyksXG4gICAgICAvLyBhZGQgaW5saW5lIHBvbGljeSB0byBlbmNyeXB0IGxvZ3NcbiAgICAgIGlubGluZVBvbGljaWVzOiB7XG4gICAgICAgIFNlY3VyaXR5Q29uZmlnOiBuZXcgUG9saWN5RG9jdW1lbnQoe1xuICAgICAgICAgIHN0YXRlbWVudHM6IFtcbiAgICAgICAgICAgIG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgICAgXCJsb2dzOkFzc29jaWF0ZUttc0tleVwiXG4gICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgICAgICAgIGBhcm46YXdzOmxvZ3M6JHtBd3MuUkVHSU9OfToke0F3cy5BQ0NPVU5UX0lEfTpsb2ctZ3JvdXA6L2F3cy1nbHVlL2pvYnMvKmBcbiAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH0pXG4gICAgICAgICAgXVxuICAgICAgICB9KVxuICAgICAgfVxuICAgIH0pO1xuICBcbiAgICAvLyBncmFudCBwZXJtaXNzaW9ucyBvbiB0aGUgZVMzIGJ1Y2tldHMgdG8gdGhlIGdsdWUgam9iIHJvbGVcbiAgICBzb3VyY2VCdWNrZXQuZ3JhbnRSZWFkKGdsdWVSb2xlLCBwcm9wcy5zM0xvY2F0aW9uLm9iamVjdEtleSArICcqJyk7XG4gICAgb3V0cHV0QnVja2V0LmdyYW50UmVhZFdyaXRlKGdsdWVSb2xlLCBwcm9wcy5zM0xvY2F0aW9uLm9iamVjdEtleSArICcqJyk7XG5cbiAgICAvLyB0aGUgU1NNIHBhcmFtZXRlciB1c2VkIHRvIHN0b3JlZCB0aGUgb3V0cHV0IG9mIHRoZSBHbHVlIGpvYlxuICAgIGNvbnN0IHNzbSA9IG5ldyBTdHJpbmdQYXJhbWV0ZXIodGhpcywgJ0pvYk91dHB1dFBhcmFtZXRlcicsIHtcbiAgICAgIHN0cmluZ1ZhbHVlOiAnbWluRGF0ZXRpbWUnLCAgICBcbiAgICB9KTtcbiAgICBzc20uZ3JhbnRXcml0ZShnbHVlUm9sZSk7XG5cbiAgICBnbHVlUm9sZS5hZGRNYW5hZ2VkUG9saWN5KE1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQVdTR2x1ZVNlcnZpY2VSb2xlJykpO1xuXG4gICAgLy8gY3JlYXRlIGEgS01TIGtleSBmb3IgdGhlIEdsdWUgc2VjdXJpdHkgY29uZmlndXJlYXRpb25cbiAgICBjb25zdCBnbHVlS2V5ID0gIFNpbmdsZXRvbktleS5nZXRPckNyZWF0ZShzY29wZSwgJ0RlZmF1bHRLbXNLZXknKVxuICAgIFxuICAgIC8vIFdlIGFkZCBhIHJlc291cmNlIHBvbGljeSBzbyB0aGUga2V5IGNhbiBiZSB1c2VkIGluIENsb3Vkd2F0Y2ggbG9nc1xuICAgIGdsdWVLZXkuYWRkVG9SZXNvdXJjZVBvbGljeShuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIHByaW5jaXBhbHM6IFtcbiAgICAgICAgbmV3IFNlcnZpY2VQcmluY2lwYWwoYGxvZ3MuJHtBd3MuUkVHSU9OfS5hbWF6b25hd3MuY29tYClcbiAgICAgIF0sXG4gICAgICBhY3Rpb25zOiBbXG4gICAgICAgIFwia21zOkVuY3J5cHQqXCIsXG4gICAgICAgIFwia21zOkRlY3J5cHQqXCIsXG4gICAgICAgIFwia21zOlJlRW5jcnlwdCpcIixcbiAgICAgICAgXCJrbXM6R2VuZXJhdGVEYXRhS2V5KlwiLFxuICAgICAgICBcImttczpEZXNjcmliZSpcIlxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogW1wiKlwiXSxcbiAgICAgIGNvbmRpdGlvbnM6IHsgQXJuTGlrZTogeyBcImttczpFbmNyeXB0aW9uQ29udGV4dDphd3M6bG9nczphcm5cIjogYGFybjphd3M6bG9nczoke0F3cy5SRUdJT059OiR7QXdzLkFDQ09VTlRfSUR9OipgIH0gfSxcbiAgICB9KSlcblxuICAgIC8vIHRoZSBHbHVlIGpvYiBzZWN1cml0eSBjb25maWd1cmF0aW9uIGZvbGxvd2luZyBiZXN0IHByYWN0aWNlc1xuICAgIGNvbnN0IGNvbmYgPSBuZXcgU2VjdXJpdHlDb25maWd1cmF0aW9uKHRoaXMsICdTZWN1cml0eUNvbmZpZ3VyYXRpb24nLCB7XG4gICAgICBzZWN1cml0eUNvbmZpZ3VyYXRpb25OYW1lOiBpZCxcbiAgICAgIGNsb3VkV2F0Y2hFbmNyeXB0aW9uOiB7XG4gICAgICAgIG1vZGU6IENsb3VkV2F0Y2hFbmNyeXB0aW9uTW9kZS5LTVMsXG4gICAgICAgIGttc0tleTogZ2x1ZUtleSxcbiAgICAgIH0sXG4gICAgICBqb2JCb29rbWFya3NFbmNyeXB0aW9uOiB7XG4gICAgICAgIG1vZGU6IEpvYkJvb2ttYXJrc0VuY3J5cHRpb25Nb2RlLkNMSUVOVF9TSURFX0tNUyxcbiAgICAgICAga21zS2V5OiBnbHVlS2V5LFxuICAgICAgfSxcbiAgICAgIHMzRW5jcnlwdGlvbjoge1xuICAgICAgICBtb2RlOiBTM0VuY3J5cHRpb25Nb2RlLlMzX01BTkFHRUQsXG4gICAgICB9XG4gICAgfSk7XG5cbiAgICAvLyBjYWxjdWxhdGUgdGhlIHNpemUgb2YgdGhlIEdsdWUgam9iIGJhc2VkIG9uIGlucHV0IGRhdGEgc2l6ZSAoMSArIDEgRFBVIHBlciAyR0IpXG4gICAgdmFyIHdvcmtlck51bSA9IDk7XG4gICAgaWYgKHByb3BzLmFwcHJveGltYXRlRGF0YVNpemUgIT09IHVuZGVmaW5lZCAmJiBwcm9wcy5hcHByb3hpbWF0ZURhdGFTaXplID4gMTYpe1xuICAgICAgd29ya2VyTnVtID0gTWF0aC5jZWlsKHByb3BzLmFwcHJveGltYXRlRGF0YVNpemUgLyAyKSArIDE7XG4gICAgfVxuXG4gICAgLy8gR2x1ZSBqb2IgdG8gcHJlcGFyZSB0aGUgZGF0YXNldFxuICAgIGNvbnN0IGdsdWVKb2IgPSBuZXcgU3luY2hyb25vdXNHbHVlSm9iKHNjb3BlLCAnQ3VzdG9tRGF0YXNldEpvYicsIHtcbiAgICAgIGV4ZWN1dGFibGU6IFByZUJ1bmRsZWRQeXNwYXJrSm9iRXhlY3V0YWJsZS5weXRob25FdGwoe1xuICAgICAgICBnbHVlVmVyc2lvbjogR2x1ZVZlcnNpb24uVjNfMCxcbiAgICAgICAgcHl0aG9uVmVyc2lvbjogUHl0aG9uVmVyc2lvbi5USFJFRSxcbiAgICAgICAgY29kZVBhdGg6ICdkYXRhLWdlbmVyYXRvci9yZXNvdXJjZXMvZ2x1ZS9jdXN0b20tZGF0YXNldC9zY3JpcHQucHknLFxuICAgICAgfSksXG4gICAgICBkZWZhdWx0QXJndW1lbnRzOiB7XG4gICAgICAgICctLXMzX2lucHV0X3BhdGgnOiBgczM6Ly8ke3Byb3BzLnMzTG9jYXRpb24uYnVja2V0TmFtZX0vJHtwcm9wcy5zM0xvY2F0aW9uLm9iamVjdEtleX1gLFxuICAgICAgICAnLS1zM19vdXRwdXRfcGF0aCc6IGBzMzovLyR7b3V0cHV0QnVja2V0LmJ1Y2tldE5hbWV9LyR7cHJvcHMuczNMb2NhdGlvbi5vYmplY3RLZXl9YCxcbiAgICAgICAgJy0taW5wdXRfZm9ybWF0JzogcHJvcHMuaW5wdXRGb3JtYXQsXG4gICAgICAgICctLWRhdGV0aW1lX2NvbHVtbic6IHByb3BzLmRhdGV0aW1lQ29sdW1uLFxuICAgICAgICAnLS1wYXJ0aXRpb25fcmFuZ2UnOiBwcm9wcy5wYXJ0aXRpb25SYW5nZS50b01pbnV0ZXMoKS50b1N0cmluZygpLFxuICAgICAgICAnLS1zc21fcGFyYW1ldGVyJzogc3NtLnBhcmFtZXRlck5hbWUsXG4gICAgICAgICctLWVuYWJsZS1hdXRvLXNjYWxpbmcnOiAndHJ1ZScsXG4gICAgICB9LFxuICAgICAgcm9sZTogZ2x1ZVJvbGUsXG4gICAgICB3b3JrZXJDb3VudDogcHJvcHMuYXBwcm94aW1hdGVEYXRhU2l6ZSA/IHdvcmtlck51bSA6IDEwMCxcbiAgICAgIHdvcmtlclR5cGU6IFdvcmtlclR5cGUuR18xWCxcbiAgICAgIGNvbnRpbnVvdXNMb2dnaW5nOntcbiAgICAgICAgZW5hYmxlZDogdHJ1ZSxcbiAgICAgIH0sXG4gICAgICBzZWN1cml0eUNvbmZpZ3VyYXRpb246IGNvbmYsXG4gICAgfSk7XG5cbiAgICB0aGlzLmdsdWVKb2JMb2dHcm91cCA9IGdsdWVKb2IuZ2x1ZUpvYkxvZ1N0cmVhbTtcblxuICAgIC8vIEdldCB0aGUgb2Zmc2V0IHZhbHVlIGNhbGN1bGF0ZWQgaW4gdGhlIFN5bmNocm9ub3VzR2x1ZUpvYlxuICAgIC8vIFdlIGNhbm5vdCByZWx5IG9uIHRoZSBTU00gcGFyYW1ldGVyIHJlc291cmNlIGNyZWF0ZWQgcHJldmlvdXNseSBiZWNhdXNlIHRoZSBvZmZzZXQgaXMgZ2VuZXJhdGVkIGR1cmluZyBkZXBsb3kgdGltZVxuICAgIGNvbnN0IGdldFBhcmFtZXRlciA9IG5ldyBBd3NDdXN0b21SZXNvdXJjZSh0aGlzLCAnR2V0UGFyYW1ldGVyJywge1xuICAgICAgb25DcmVhdGU6IHtcbiAgICAgICAgc2VydmljZTogJ1NTTScsXG4gICAgICAgIGFjdGlvbjogJ2dldFBhcmFtZXRlcicsXG4gICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICBOYW1lOiBzc20ucGFyYW1ldGVyTmFtZSwgICAgICAgICAgXG4gICAgICAgIH0sXG4gICAgICAgIHBoeXNpY2FsUmVzb3VyY2VJZDogUGh5c2ljYWxSZXNvdXJjZUlkLm9mKERhdGUubm93KCkudG9TdHJpbmcoKSksXG4gICAgICB9LFxuICAgICAgcG9saWN5OiBBd3NDdXN0b21SZXNvdXJjZVBvbGljeS5mcm9tU2RrQ2FsbHMoe1xuICAgICAgICByZXNvdXJjZXM6IFtzc20ucGFyYW1ldGVyQXJuXSxcbiAgICAgIH0pXG4gICAgfSk7XG4gICAgLy8gQWRkIGEgZGVwZW5kZW5jeSBvbiB0aGUgc3luY2hyb25vdXMgZ2x1ZSBqb2IgdG8gb25seSBnZXQgdGhlIHZhbHVlIGFmdGVyIHByb2Nlc3NpbmdcbiAgICBnZXRQYXJhbWV0ZXIubm9kZS5hZGREZXBlbmRlbmN5KGdsdWVKb2IpO1xuXG4gICAgLy8gQ3JlYXRlIGEgcHJlcGFyZWQgZGF0YXNldCBiYXNlZCBvbiB0aGUgb3V0cHV0IG9mIHRoZSBHbHVlIGpvYlxuICAgIHRoaXMucHJlcGFyZWREYXRhc2V0ID0gbmV3IFByZXBhcmVkRGF0YXNldCh7XG4gICAgICBsb2NhdGlvbjoge1xuICAgICAgICBidWNrZXROYW1lOiBvdXRwdXRCdWNrZXQuYnVja2V0TmFtZSxcbiAgICAgICAgb2JqZWN0S2V5OiBwcm9wcy5zM0xvY2F0aW9uLm9iamVjdEtleSxcbiAgICAgIH0sXG4gICAgICBvZmZzZXQ6IGdldFBhcmFtZXRlci5nZXRSZXNwb25zZUZpZWxkKCdQYXJhbWV0ZXIuVmFsdWUnKSxcbiAgICAgIG1hbmlmZXN0TG9jYXRpb246IHtcbiAgICAgICAgYnVja2V0TmFtZTogb3V0cHV0QnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgIG9iamVjdEtleTogcHJvcHMuczNMb2NhdGlvbi5vYmplY3RLZXkrJy1tYW5pZmVzdC5jc3YnLFxuICAgICAgfSxcbiAgICAgIGRhdGVUaW1lQ29sdW1uVG9GaWx0ZXI6IHByb3BzLmRhdGV0aW1lQ29sdW1uLFxuICAgICAgZGF0ZVRpbWVDb2x1bW5zVG9BZGp1c3Q6IHByb3BzLmRhdGV0aW1lQ29sdW1uc1RvQWRqdXN0LFxuICAgIH0pO1xuICB9XG59XG5cbiJdfQ==