"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BatchReplayer = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const constructs_1 = require("constructs");
const pre_bundled_function_1 = require("../common/pre-bundled-function");
const batch_replayer_helpers_1 = require("./batch-replayer-helpers");
/**
 * Replay the data in the given PartitionedDataset.
 *
 * It will dump files into the target based on the given `frequency`.
 * The computation is in a Step Function with two Lambda steps.
 *
 * 1. resources/lambdas/find-file-paths
 * Read the manifest file and output a list of S3 file paths within that batch time range
 *
 * 2. resources/lambdas/write-in-batch
 * Take a file path, filter only records within given time range, adjust the time with offset to
 * make it looks like just being generated. Then write the output to the target
 *
 * Usage example:
 * ```typescript
 *
 * const myBucket = new Bucket(stack, "MyBucket")
 *
 * let myProps: S3Sink = {
 *  sinkBucket: myBucket,
 *  sinkObjectKey: 'some-prefix',
 *  outputFileMaxSizeInBytes: 10000000,
 * }
 *
 * new BatchReplayer(stack, "WebSalesReplayer", {
 *   dataset: PreparedDataset.RETAIL_1_GB_WEB_SALE,
 *   s3Props: myProps,
 *   frequency: 120,
 * });
 * ```
 *
 * :warning: **If the Bucket is encrypted with KMS, the Key must be managed by this stack.
 */
class BatchReplayer extends constructs_1.Construct {
    /**
     * Constructs a new instance of the BatchReplayer construct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {BatchReplayerProps} props the BatchReplayer [properties]{@link BatchReplayerProps}
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.dataset = props.dataset;
        this.frequency = props.frequency?.toSeconds() || 60;
        this.additionalStepFunctionTasks = props.additionalStepFunctionTasks;
        const manifestBucketName = this.dataset.manifestLocation.bucketName;
        const manifestObjectKey = this.dataset.manifestLocation.objectKey;
        const dataBucketName = this.dataset.location.bucketName;
        const dataObjectKey = this.dataset.location.objectKey;
        // Properties for DynamoDB target
        this.ddbProps = props.ddbProps ? props.ddbProps : undefined;
        // Properties for Redshift target
        this.redshiftProps = (props.redshiftProps && props.secGroup && props.vpc) ? props.redshiftProps : undefined;
        // Properties for Aurora target
        this.auroraProps = (props.auroraProps && props.secGroup && props.vpc) ? props.auroraProps : undefined;
        // Properties for RDS target
        this.rdsProps = (props.rdsProps && props.secGroup && props.vpc) ? props.rdsProps : undefined;
        const dataWranglerLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'PandasLayer', `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:336392948345:layer:AWSDataWrangler-Python39:1`);
        const findFilePathsFnPolicy = [
            new aws_iam_1.PolicyStatement({
                actions: [
                    's3:GetObject',
                    's3:ListBucket',
                ],
                resources: [
                    `arn:aws:s3:::${dataBucketName}/${dataObjectKey}/*`,
                    `arn:aws:s3:::${dataBucketName}/${dataObjectKey}-manifest.csv`,
                    `arn:aws:s3:::${dataBucketName}`,
                ],
            }),
        ];
        /**
         * Find all paths within the time range from the manifest file
         */
        const findFilePathsFn = new pre_bundled_function_1.PreBundledFunction(this, 'FindFilePath', {
            memorySize: 1024,
            codePath: 'data-generator/resources/lambdas/find-file-paths',
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            handler: 'find-file-paths.handler',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            layers: [dataWranglerLayer],
            lambdaPolicyStatements: findFilePathsFnPolicy,
        });
        const findFilePathsFnTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'FindFilePathFnTask', {
            lambdaFunction: findFilePathsFn,
            payload: aws_stepfunctions_1.TaskInput.fromObject({
                frequency: this.frequency,
                manifestFileBucket: manifestBucketName,
                manifestFileKey: manifestObjectKey,
                triggerTime: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Input.time'),
                offset: this.dataset.offset,
            }),
            // Retry on 500 error on invocation with an interval of 2 sec with back-off rate 2, for 6 times
            retryOnServiceExceptions: true,
            outputPath: '$.Payload',
        });
        const writeInBatchFnPolicy = [];
        let taskInputObj = {
            // Array from the last step to be mapped
            outputFileIndex: aws_stepfunctions_1.JsonPath.stringAt('$.index'),
            filePath: aws_stepfunctions_1.JsonPath.stringAt('$.filePath'),
            // For calculating the start/end time
            frequency: this.frequency,
            triggerTime: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Input.time'),
            offset: this.dataset.offset,
            // For file processing
            dateTimeColumnToFilter: this.dataset.dateTimeColumnToFilter,
            dateTimeColumnsToAdjust: this.dataset.dateTimeColumnsToAdjust,
        };
        // S3 target is selected
        if (props.s3Props) {
            // Used to force S3 bucket auto cleaning after deletion of this
            this.node.addDependency(props.s3Props.sinkBucket);
            var modS3Props = { ...props.s3Props };
            if (!props.s3Props.outputFileMaxSizeInBytes) {
                modS3Props.outputFileMaxSizeInBytes = 100 * 1024 * 1024; //Default to 100 MB
            }
            modS3Props.sinkObjectKey = props.s3Props.sinkObjectKey ?
                `${props.s3Props.sinkObjectKey}/${this.dataset.tableName}` : this.dataset.tableName;
            this.s3Props = modS3Props;
            const { policy, taskInputParams } = (0, batch_replayer_helpers_1.prepareS3Target)(this.s3Props, dataBucketName, dataObjectKey);
            writeInBatchFnPolicy.push(policy);
            taskInputObj = Object.assign(taskInputObj, taskInputParams);
        }
        // DynamoDB target is selected
        if (this.ddbProps) {
            const { policy, taskInputParams } = (0, batch_replayer_helpers_1.prepareDdbTarget)(this.ddbProps);
            writeInBatchFnPolicy.push(policy);
            taskInputObj = Object.assign(taskInputObj, taskInputParams);
        }
        /**
         * Redshift, Aurora and RDS databases require the Lambda to have VPC access.
         */
        if (this.secGroup && this.vpc) {
            // Lambda requires these actions to have access to all resources in order to connect to a VPC
            writeInBatchFnPolicy.push(new aws_iam_1.PolicyStatement({
                actions: [
                    'ec2:CreateNetworkInterface',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:DeleteNetworkInterface',
                ],
                resources: ['*'],
            }));
            // Redshift target is selected
            if (this.redshiftProps) {
                const { policy, taskInputParams } = (0, batch_replayer_helpers_1.prepareRedshiftTarget)(this.redshiftProps);
                writeInBatchFnPolicy.push(policy);
                taskInputObj = Object.assign(taskInputObj, taskInputParams);
            }
            // Aurora target is selected
            if (this.auroraProps) {
                const { policy, taskInputParams } = (0, batch_replayer_helpers_1.prepareAuroraTarget)(this.auroraProps);
                writeInBatchFnPolicy.push(policy);
                taskInputObj = Object.assign(taskInputObj, taskInputParams);
            }
            // RDS target is selected
            if (this.rdsProps) {
                const { policy, taskInputParams } = (0, batch_replayer_helpers_1.prepareRdsTarget)(this.rdsProps);
                writeInBatchFnPolicy.push(policy);
                taskInputObj = Object.assign(taskInputObj, taskInputParams);
            }
        }
        /**
         * Rewrite data
         */
        const writeInBatchFn = new pre_bundled_function_1.PreBundledFunction(this, 'WriteInBatch', {
            memorySize: 3008,
            codePath: 'data-generator/resources/lambdas/write-in-batch',
            runtime: aws_lambda_1.Runtime.PYTHON_3_9,
            handler: 'write-in-batch.handler',
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
            timeout: aws_cdk_lib_1.Duration.minutes(15),
            layers: [dataWranglerLayer],
            lambdaPolicyStatements: writeInBatchFnPolicy,
            vpc: this.vpc ? this.vpc : undefined,
            vpcSubnets: this.vpc ? this.vpc.selectSubnets({
                subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_NAT,
            }) : undefined,
            securityGroups: this.secGroup ? [this.secGroup] : undefined,
        });
        if (this.s3Props) {
            // grant permissions to write to the bucket and to use the KMS key
            this.s3Props.sinkBucket.grantWrite(writeInBatchFn, `${this.s3Props.sinkObjectKey}/*`);
        }
        const writeInBatchFnTask = new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'WriteInBatchFnTask', {
            lambdaFunction: writeInBatchFn,
            payload: aws_stepfunctions_1.TaskInput.fromObject(taskInputObj),
            // Retry on 500 error on invocation with an interval of 2 sec with back-off rate 2, for 6 times
            retryOnServiceExceptions: true,
            outputPath: '$.Payload',
        });
        // Use "Map" step to write each filePath parallelly
        const writeInBatchMapTask = new aws_stepfunctions_1.Map(this, 'WriteInBatchMapTask', {
            itemsPath: aws_stepfunctions_1.JsonPath.stringAt('$.filePaths'),
            parameters: {
                index: aws_stepfunctions_1.JsonPath.stringAt('$$.Map.Item.Index'),
                filePath: aws_stepfunctions_1.JsonPath.stringAt('$$.Map.Item.Value'),
            },
        });
        writeInBatchMapTask.iterator(writeInBatchFnTask);
        // Overarching Step Function StateMachine
        const batchReplayStepFn = new aws_stepfunctions_1.StateMachine(this, 'BatchReplayStepFn', {
            definition: this.chainStepFunctionTasks(findFilePathsFnTask.next(writeInBatchMapTask)),
            timeout: aws_cdk_lib_1.Duration.minutes(20),
            logs: {
                destination: new aws_logs_1.LogGroup(this, 'LogGroup', {
                    retention: aws_logs_1.RetentionDays.ONE_WEEK,
                    logGroupName: `/aws/batch-replayer/${this.dataset.tableName}`,
                    removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
                }),
                level: aws_stepfunctions_1.LogLevel.ALL,
            },
        });
        new aws_events_1.Rule(this, 'BatchReplayStepFnTrigger', {
            schedule: aws_events_1.Schedule.cron({ minute: `0/${Math.ceil(this.frequency / 60)}` }),
            targets: [new aws_events_targets_1.SfnStateMachine(batchReplayStepFn, {})],
        });
    }
    chainStepFunctionTasks(requiredTasks) {
        let base = requiredTasks;
        if (this.additionalStepFunctionTasks) {
            this.additionalStepFunctionTasks.forEach(newTask => {
                base = base.next(newTask);
            });
        }
        return base;
    }
}
_a = JSII_RTTI_SYMBOL_1;
BatchReplayer[_a] = { fqn: "aws-analytics-reference-architecture.BatchReplayer", version: "2.12.6" };
exports.BatchReplayer = BatchReplayer;
//# sourceMappingURL=data:application/json;base64,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