"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.prepareRdsTarget = exports.prepareAuroraTarget = exports.prepareRedshiftTarget = exports.prepareDdbTarget = exports.prepareS3Target = void 0;
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
/**
 * Set up permissions to write to S3 target and params required for WriteInBatch Lambda
 *
 * @param S3Props
 * @param dataBucketName
 * @param dataObjectKey
 */
function prepareS3Target(S3Props, dataBucketName, dataObjectKey) {
    // Add policy to allow access to bucket
    const policy = new aws_iam_1.PolicyStatement({
        actions: [
            's3:GetObject',
            's3:ListBucket',
        ],
        resources: [
            `arn:aws:s3:::${dataBucketName}/${dataObjectKey}/*`,
            `arn:aws:s3:::${dataBucketName}`,
        ],
    });
    // Add params to allow data to be output to S3 target
    const taskInputParams = {
        sinkPath: S3Props.sinkBucket.s3UrlForObject(S3Props.sinkObjectKey),
        outputFileMaxSizeInBytes: S3Props.outputFileMaxSizeInBytes,
    };
    return { policy, taskInputParams };
}
exports.prepareS3Target = prepareS3Target;
/**
 * Set up permissions to write to DynamoDB target and params required for WriteInBatch Lambda
 *
 * @param ddbProps
 */
function prepareDdbTarget(ddbProps) {
    // Add policy to allow access to table
    const policy = new aws_iam_1.PolicyStatement({
        actions: [
            'dynamodb:DescribeTable',
            'dynamodb:PutItem',
            'dynamodb:BatchWriteItem',
        ],
        resources: [ddbProps.table.tableArn],
    });
    // Add params to allow data to be output to DynamoDB target
    const taskInputParams = {
        ddbTableName: ddbProps.table.tableName,
    };
    return { policy, taskInputParams };
}
exports.prepareDdbTarget = prepareDdbTarget;
/**
 * Set up permissions to write to Redshift target and params required for WriteInBatch Lambda
 *
 * @param redshiftProps
 */
function prepareRedshiftTarget(redshiftProps) {
    // Add policy to allow access to table
    const policy = new aws_iam_1.PolicyStatement({
        actions: [
            'secretsmanager:GetSecretValue',
        ],
        resources: [redshiftProps.connection],
    });
    // Add params to allow data to be output to Redshift target
    const taskInputParams = {
        redshiftTableName: redshiftProps.table,
        redshiftConnection: redshiftProps.connection,
        redshiftSchema: redshiftProps.schema,
    };
    return { policy, taskInputParams };
}
exports.prepareRedshiftTarget = prepareRedshiftTarget;
/**
 * Set up permissions to write to S3 target and params required for WriteInBatch Lambda
 *
 * @param auroraProps
 */
function prepareAuroraTarget(auroraProps) {
    // Add policy to allow access to table
    const policy = new aws_iam_1.PolicyStatement({
        actions: [
            'secretsmanager:GetSecretValue',
        ],
        resources: [auroraProps.connection],
    });
    // Add params to allow data to be output to Aurora MySQL target
    const taskInputParams = {
        auroraTableName: auroraProps.table,
        auroraConnection: auroraProps.connection,
        auroraSchema: auroraProps.schema,
        dbType: auroraProps.type,
    };
    return { policy, taskInputParams };
}
exports.prepareAuroraTarget = prepareAuroraTarget;
/**
 * Set up permissions to write to RDS target and params required for WriteInBatch Lambda
 *
 * @param rdsProps
 */
function prepareRdsTarget(rdsProps) {
    // Add policy to allow access to table
    const policy = new aws_iam_1.PolicyStatement({
        actions: [
            'secretsmanager:GetSecretValue',
        ],
        resources: [rdsProps.connection],
    });
    // Add params to allow data to be output to RDS target
    const taskInputParams = {
        rdsTableName: rdsProps.table,
        rdsConnection: rdsProps.connection,
        rdsSchema: rdsProps.schema,
        dbType: rdsProps.type,
    };
    return { policy, taskInputParams };
}
exports.prepareRdsTarget = prepareRdsTarget;
//# sourceMappingURL=data:application/json;base64,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