import { PolicyStatement } from 'aws-cdk-lib/aws-iam';
import { ITable } from 'aws-cdk-lib/aws-dynamodb';
import { Bucket } from 'aws-cdk-lib/aws-s3';
export interface S3Sink {
    /**
     * The S3 Bucket sink where the BatchReplayer writes data.
     * :warning: **If the Bucket is encrypted with KMS, the Key must be managed by this stack.
     */
    readonly sinkBucket: Bucket;
    /**
     * The S3 object key sink where the BatchReplayer writes data.
     * @default - No object key is used and the BatchReplayer writes the dataset in s3://<BUCKET_NAME>/<TABLE_NAME>
     */
    readonly sinkObjectKey?: string;
    /**
     * The maximum file size in Bytes written by the BatchReplayer
     * @default - The BatchReplayer writes 100MB files maximum
     */
    readonly outputFileMaxSizeInBytes?: number;
}
export interface DbSink {
    /**
     * The name of the table to write to
     */
    readonly table: string;
    /**
     * Secret ARN of the database connection
     */
    readonly connection: string;
    /**
     * The name of the database schema if required
     */
    readonly schema?: string;
    /**
     * Database engine if applicable
     */
    readonly type?: 'mysql' | 'postgresql';
}
export interface DynamoDbSink {
    /**
     * DynamoDB table
     */
    readonly table: ITable;
}
/**
 * Set up permissions to write to S3 target and params required for WriteInBatch Lambda
 *
 * @param S3Props
 * @param dataBucketName
 * @param dataObjectKey
 */
export declare function prepareS3Target(S3Props: S3Sink, dataBucketName: string, dataObjectKey: string): {
    policy: PolicyStatement;
    taskInputParams: Object;
};
/**
 * Set up permissions to write to DynamoDB target and params required for WriteInBatch Lambda
 *
 * @param ddbProps
 */
export declare function prepareDdbTarget(ddbProps: DynamoDbSink): {
    policy: PolicyStatement;
    taskInputParams: Object;
};
/**
 * Set up permissions to write to Redshift target and params required for WriteInBatch Lambda
 *
 * @param redshiftProps
 */
export declare function prepareRedshiftTarget(redshiftProps: DbSink): {
    policy: PolicyStatement;
    taskInputParams: Object;
};
/**
 * Set up permissions to write to S3 target and params required for WriteInBatch Lambda
 *
 * @param auroraProps
 */
export declare function prepareAuroraTarget(auroraProps: DbSink): {
    policy: PolicyStatement;
    taskInputParams: Object;
};
/**
 * Set up permissions to write to RDS target and params required for WriteInBatch Lambda
 *
 * @param rdsProps
 */
export declare function prepareRdsTarget(rdsProps: DbSink): {
    policy: PolicyStatement;
    taskInputParams: Object;
};
