"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.reportBuild = exports.startBuild = void 0;
const defaultDestroyBuildSpec = `
version: 0.2
env:
  variables:
    CFN_RESPONSE_URL: CFN_RESPONSE_URL_NOT_SET
    CFN_STACK_ID: CFN_STACK_ID_NOT_SET
    CFN_REQUEST_ID: CFN_REQUEST_ID_NOT_SET
    CFN_LOGICAL_RESOURCE_ID: CFN_LOGICAL_RESOURCE_ID_NOT_SET
phases:
  pre_build:
    on-failure: ABORT
    commands:
      - echo "Default destroy buildspec"
      - cd $CODEBUILD_SRC_DIR/$CDK_APP_LOCATION
      - npm install -g aws-cdk && sudo apt-get install python3 && python -m
        ensurepip --upgrade && python -m pip install --upgrade pip && python -m
        pip install -r requirements.txt
      - \"export AWS_ACCOUNT_ID=$(echo $CODEBUILD_BUILD_ARN | cut -d: -f5)\"
      - 'echo \"AWS_ACCOUNT_ID: $AWS_ACCOUNT_ID\"'
      - cdk bootstrap aws://$AWS_ACCOUNT_ID/$AWS_REGION
  build:
    on-failure: ABORT
    commands:
      - \"export AWS_ACCOUNT_ID=$(echo $CODEBUILD_BUILD_ARN | cut -d: -f5)\"
      - 'echo \"AWS_ACCOUNT_ID: $AWS_ACCOUNT_ID\"'
      - cdk destroy --force --all --require-approval never
`;
const defaultDeployBuildSpec = `
version: 0.2
env:
  variables:
    CFN_RESPONSE_URL: CFN_RESPONSE_URL_NOT_SET
    CFN_STACK_ID: CFN_STACK_ID_NOT_SET
    CFN_REQUEST_ID: CFN_REQUEST_ID_NOT_SET
    CFN_LOGICAL_RESOURCE_ID: CFN_LOGICAL_RESOURCE_ID_NOT_SET
    PARAMETERS: PARAMETERS_NOT_SET
    STACKNAME: STACKNAME_NOT_SET
phases:
  pre_build:
    on-failure: ABORT
    commands:
      - echo "Default deploy buildspec"
      - cd $CODEBUILD_SRC_DIR/$CDK_APP_LOCATION
      - npm install -g aws-cdk && sudo apt-get install python3 && python -m
        ensurepip --upgrade && python -m pip install --upgrade pip && python -m
        pip install -r requirements.txt
      - \"export AWS_ACCOUNT_ID=$(echo $CODEBUILD_BUILD_ARN | cut -d: -f5)\"
      - 'echo \"AWS_ACCOUNT_ID: $AWS_ACCOUNT_ID\"'
      - cdk bootstrap aws://$AWS_ACCOUNT_ID/$AWS_REGION
  build:
    on-failure: ABORT
    commands:
      - \"export AWS_ACCOUNT_ID=$(echo $CODEBUILD_BUILD_ARN | cut -d: -f5)\"
      - 'echo \"AWS_ACCOUNT_ID: $AWS_ACCOUNT_ID\"'
      - cdk deploy $STACKNAME $PARAMETERS --require-approval=never
`;
// workaround to get a Lambda function with inline code and packaged into the ARA library
// We need inline code to ensure it's deployable via a CloudFormation template
// TODO modify the PreBundledFunction to allow for inline Lambda in addtion to asset based Lambda
const startBuild = (deployBuildSpec, destroyBuildSpec) => {
    return `
const respond = async function(event, context, responseStatus, responseData, physicalResourceId, noEcho) {
  return new Promise((resolve, reject) => {
    var responseBody = JSON.stringify({
      Status: responseStatus,
      Reason: \"See the details in CloudWatch Log Stream: \" + context.logGroupName + \" \" + context.logStreamName,
      PhysicalResourceId: physicalResourceId || context.logStreamName,
      StackId: event.StackId,
      RequestId: event.RequestId,
      LogicalResourceId: event.LogicalResourceId,
      NoEcho: noEcho || false,
      Data: responseData
    });
    
    console.log(\"Response body:\", responseBody);
    
    var https = require(\"https\");
    var url = require(\"url\");
    
    var parsedUrl = url.parse(event.ResponseURL);
    var options = {
      hostname: parsedUrl.hostname,
      port: 443,
      path: parsedUrl.path,
      method: \"PUT\",
      headers: {
        \"content-type\": \"\",
        \"content-length\": responseBody.length
      }
    };
    
    var request = https.request(options, function(response) {
      console.log(\"Status code: \" + response.statusCode);
      console.log(\"Status message: \" + response.statusMessage);
      resolve();
    });
    
    request.on(\"error\", function(error) {
      console.log(\"respond(..) failed executing https.request(..): \" + error);
      resolve();
    });
    
    request.write(responseBody);
    request.end();
  });
};

const AWS = require('aws-sdk');

exports.handler = async function (event, context) {
  console.log(JSON.stringify(event, null, 4));
  try {
    const projectName = event.ResourceProperties.ProjectName;
    const codebuild = new AWS.CodeBuild();
    
    console.log(\`Starting new build of project \${projectName}\`);
    
    const { build } = await codebuild.startBuild({
      projectName,
      // Pass CFN related parameters through the build for extraction by the
      // completion handler.
      buildspecOverride: event.RequestType === 'Delete' ? \`${destroyBuildSpec ? `${destroyBuildSpec.toBuildSpec()}` : defaultDestroyBuildSpec}\` : \`${deployBuildSpec ? `${deployBuildSpec.toBuildSpec()}` : defaultDeployBuildSpec}\`,
      environmentVariablesOverride: [
        {
          name: 'CFN_RESPONSE_URL',
          value: event.ResponseURL
        },
        {
          name: 'CFN_STACK_ID',
          value: event.StackId
        },
        {
          name: 'CFN_REQUEST_ID',
          value: event.RequestId
        },
        {
          name: 'CFN_LOGICAL_RESOURCE_ID',
          value: event.LogicalResourceId
        },
        {
          name: 'BUILD_ROLE_ARN',
          value: event.ResourceProperties.BuildRoleArn
        }
      ]
    }).promise();
    console.log(\`Build id \${build.id} started - resource completion handled by EventBridge\`);
  } catch(error) {
    console.error(error);
    await respond(event, context, 'FAILED', { Error: error });
  }
};
`;
};
exports.startBuild = startBuild;
exports.reportBuild = `
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0

const respond = async function(event, context, responseStatus, responseData, physicalResourceId, noEcho) {
  return new Promise((resolve, reject) => {
    var responseBody = JSON.stringify({
      Status: responseStatus,
      Reason: "See the details in CloudWatch Log Stream: " + context.logGroupName + " " + context.logStreamName,
      PhysicalResourceId: physicalResourceId || context.logStreamName,
      StackId: event.StackId,
      RequestId: event.RequestId,
      LogicalResourceId: event.LogicalResourceId,
      NoEcho: noEcho || false,
      Data: responseData
    });
    
    console.log("Response body:\
    ", responseBody);
    
    var https = require("https");
    var url = require("url");
    
    var parsedUrl = url.parse(event.ResponseURL);
    var options = {
      hostname: parsedUrl.hostname,
      port: 443,
      path: parsedUrl.path,
      method: "PUT",
      headers: {
        "content-type": "",
        "content-length": responseBody.length
      }
    };
    
    var request = https.request(options, function(response) {
      console.log("Status code: " + response.statusCode);
      console.log("Status message: " + response.statusMessage);
      resolve();
    });
    
    request.on("error", function(error) {
      console.log("respond(..) failed executing https.request(..): " + error);
      resolve();
    });
    
    request.write(responseBody);
    request.end();
  });
};

const AWS = require('aws-sdk');

exports.handler = async function (event, context) {
  console.log(JSON.stringify(event, null, 4));
  
  const projectName = event['detail']['project-name'];
  
  const codebuild = new AWS.CodeBuild();
  
  const buildId = event['detail']['build-id'];
  const { builds } = await codebuild.batchGetBuilds({
    ids: [ buildId ]
  }).promise();
  
  console.log(JSON.stringify(builds, null, 4));
  
  const build = builds[0];
  // Fetch the CFN resource and response parameters from the build environment.
  const environment = {};
  build.environment.environmentVariables.forEach(e => environment[e.name] = e.value);
  
  const response = {
    ResponseURL: environment.CFN_RESPONSE_URL,
    StackId: environment.CFN_STACK_ID,
    LogicalResourceId: environment.CFN_LOGICAL_RESOURCE_ID,
    RequestId: environment.CFN_REQUEST_ID
  };
  
  if (event['detail']['build-status'] === 'SUCCEEDED') {
    await respond(response, context, 'SUCCESS', { BuildStatus: 'SUCCESS'}, 'build');
  } else {
    await respond(response, context, 'FAILED', { Error: 'Build failed' });
  }
};
`;
//# sourceMappingURL=data:application/json;base64,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