"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AraBucket = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const singleton_kms_key_1 = require("./singleton-kms-key");
/**
* An Amazon S3 Bucket following best practices for the AWS Analytics Reference Architecture.
* The bucket name is mandatory and is used as the CDK id.
* The bucket name is postfixed with the AWS account ID and the AWS region.
*
* The bucket has the following default properties:
*  * the encryption mode is KMS managed by AWS
*  * if the encryption mode is KMS customer managed, the encryption key is a default and unique KMS key for ARA
*  * the KMS key is used as a bucket key
*  * the SSL is enforced
*  * the objects are automatically deleted when the bucket is deleted
*  * the access are logged in a default and unique S3 bucket for ARA if serverAccessLogsPrefix is provided
*  * the access are not logged if serverAccessLogsPrefix is  not provided
*  * the public access is blocked and no bucket policy or object permission can grant public access
*
* All standard S3 Bucket properties can be provided to not use the defaults.
* Usage example:
* ```typescript
* import * as cdk from 'aws-cdk-lib';
* import { AraBucket } from 'aws-analytics-reference-architecture';
*
* const exampleApp = new cdk.App();
* const stack = new cdk.Stack(exampleApp, 'AraBucketStack');
*
* new AraBucket(stack, {
*  bucketName: 'test-bucket',
*  serverAccessLogsPrefix: 'test-bucket',
* });
* ```
*/
class AraBucket extends aws_s3_1.Bucket {
    /**
    * Get the Amazon S3 Bucket from the AWS CDK Stack based on the provided name.
    * If no bucket exists, it creates a new one based on the provided properties.
    */
    static getOrCreate(scope, props) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = `${props.bucketName}`;
        const stackBucket = stack.node.tryFindChild(id) ?? (stack.nestedStackParent ? stack.nestedStackParent.node.tryFindChild(id) : undefined);
        return stackBucket || new AraBucket(stack, props);
    }
    /**
    * Constructs a new instance of the AraBucket class
    * @param {Construct} scope the Scope of the CDK Construct
    * @param {AraBucketProps} props the AraBucketProps [properties]{@link AraBucketProps}
    * @access private
    */
    constructor(scope, props) {
        if (scope.node.tryGetContext('@aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy') === true) {
            throw new Error("Using @aws-cdk/aws-s3:serverAccessLogsUseBucketPolicy is not supported please switch it to false in your cdk.json");
        }
        var serverAccessLogsBucket = undefined;
        if (props.serverAccessLogsPrefix) {
            serverAccessLogsBucket = props.serverAccessLogsBucket || AraBucket.getOrCreate(scope, {
                bucketName: 's3-access-logs',
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                objectOwnership: aws_s3_1.ObjectOwnership.BUCKET_OWNER_PREFERRED,
            });
        }
        // If using KMS encryption then use a customer managed key, if not set the key to undefined
        let bucketEncryptionKey = aws_s3_1.BucketEncryption.KMS == props.encryption ? props.encryptionKey || singleton_kms_key_1.SingletonKey.getOrCreate(scope, 'DefaultKmsKey') : undefined;
        // If the bucket is for s3 access logs, we remove the bucketname to ensure uniqueness across stacks
        let bucketName = (props.bucketName == 's3-access-logs') ? undefined : `${props.bucketName}-${aws_cdk_lib_1.Aws.ACCOUNT_ID}-${aws_cdk_lib_1.Aws.REGION}`;
        // set the right default parameters in the S3 bucket props
        const bucketProps = {
            ...props,
            ...{
                bucketName: bucketName,
                encryption: props.encryption ? props.encryption : aws_s3_1.BucketEncryption.KMS_MANAGED,
                encryptionKey: bucketEncryptionKey,
                bucketKeyEnabled: aws_s3_1.BucketEncryption.KMS == props.encryption ? true : false,
                enforceSSL: props.enforceSSL || true,
                removalPolicy: props.removalPolicy || aws_cdk_lib_1.RemovalPolicy.DESTROY,
                autoDeleteObjects: props.autoDeleteObjects || true,
                serverAccessLogsBucket: serverAccessLogsBucket,
                serverAccessLogsPrefix: props.serverAccessLogsPrefix,
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                lifecycleRules: props.lifecycleRules || [{ abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(1) }],
            },
        };
        // build the S3 bucket
        super(scope, props.bucketName, bucketProps);
        // Used to force the cleaning of the access logs bucket after deletion of this 
        if (serverAccessLogsBucket) {
            this.node.addDependency(serverAccessLogsBucket);
        }
    }
}
_a = JSII_RTTI_SYMBOL_1;
AraBucket[_a] = { fqn: "aws-analytics-reference-architecture.AraBucket", version: "2.12.6" };
exports.AraBucket = AraBucket;
//# sourceMappingURL=data:application/json;base64,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