import json
import time
import http.client
from types import SimpleNamespace

class CodeAuth:
    _Endpoint = None
    _ProjectID = None
    _UseCache = None
    _CacheDuration = None
    _CacheSession = None
    _CacheTimestamp = None
    _HasInitialized = False

    @staticmethod
    def Initialize(project_endpoint, project_id, use_cache=True, cache_duration=5):
        if CodeAuth._HasInitialized:
            raise Exception("CodeAuth has already been Initialized.")
        CodeAuth._HasInitialized = True

        CodeAuth._Endpoint = project_endpoint
        CodeAuth._ProjectID = project_id
        CodeAuth._UseCache = use_cache
        CodeAuth._CacheDuration = cache_duration * 1000  # ms to ms (same as JS)
        CodeAuth._CacheSession = {}
        CodeAuth._CacheTimestamp = int(time.time() * 1000)

    # ----------------------------------------------------------------------
    # Email sign-in
    # ----------------------------------------------------------------------
    @staticmethod
    def SignInEmail(email):
        if CodeAuth._HasInitialized == False:
            raise Exception("CodeAuth have not been Initialized.")
        
        res = CodeAuth.request(
            CodeAuth._Endpoint, "/signin/email",
            {"project_id": CodeAuth._ProjectID, "email": email}
        )
        return SimpleNamespace(**res)

    @staticmethod
    def SignInEmailVerify(email, code):
        if CodeAuth._HasInitialized == False:
            raise Exception("CodeAuth have not been Initialized.")
        
        return SimpleNamespace(**CodeAuth.request(
            CodeAuth._Endpoint, "/signin/emailverify",
            {
                "project_id": CodeAuth._ProjectID,
                "email": email,
                "code": code,
            }
        ))

    # ----------------------------------------------------------------------
    # Social login
    # ----------------------------------------------------------------------
    @staticmethod
    def SignInSocial(social_type):
        if CodeAuth._HasInitialized == False:
            raise Exception("CodeAuth have not been Initialized.")
        
        return SimpleNamespace(**CodeAuth.request(
            CodeAuth._Endpoint, "/signin/social",
            {"project_id": CodeAuth._ProjectID, "social_type": social_type}
        ))

    @staticmethod
    def SignInSocialVerify(social_type, authorization_code):
        if CodeAuth._HasInitialized == False:
            raise Exception("CodeAuth have not been Initialized.")
        
        return SimpleNamespace(**CodeAuth.request(
            CodeAuth._Endpoint, "/signin/socialverify",
            {
                "project_id": CodeAuth._ProjectID,
                "social_type": social_type,
                "authorization_code": authorization_code,
            }
        ))

    # ----------------------------------------------------------------------
    # Session Info w/ cache
    # ----------------------------------------------------------------------
    @staticmethod
    def SessionInfo(session_token):
        if CodeAuth._HasInitialized == False:
            raise Exception("CodeAuth have not been Initialized.")
        
        now = int(time.time() * 1000)

        if CodeAuth._UseCache:
            if CodeAuth._CacheTimestamp + CodeAuth._CacheDuration > now:
                cached = CodeAuth._CacheSession.get(session_token)
                if cached:
                    return SimpleNamespace(**cached)
            else:
                CodeAuth._CacheTimestamp = now
                CodeAuth._CacheSession.clear()

        result = CodeAuth.request(
            CodeAuth._Endpoint, "/session/info",
            {"project_id": CodeAuth._ProjectID, "session_token": session_token},
        )

        if CodeAuth._UseCache:
            CodeAuth._CacheSession[session_token] = result

        return SimpleNamespace(**result)

    # ----------------------------------------------------------------------
    # Session Refresh
    # ----------------------------------------------------------------------
    @staticmethod
    def SessionRefresh(session_token):
        if CodeAuth._HasInitialized == False:
            raise Exception("CodeAuth have not been Initialized.")
        
        result = CodeAuth.request(
            CodeAuth._Endpoint, "/session/refresh",
            {"project_id": CodeAuth._ProjectID, "session_token": session_token},
        )

        now = int(time.time() * 1000)

        if CodeAuth._UseCache and result.get("error") != "no_error":
            if CodeAuth._CacheTimestamp + CodeAuth._CacheDuration < now:
                CodeAuth._CacheTimestamp = now
                CodeAuth._CacheSession.clear()
            else:
                CodeAuth._CacheSession.pop(session_token, None)
                CodeAuth._CacheSession[result["session_token"]] = {
                    "email": result["email"],
                    "expiration": result["expiration"],
                    "refresh_left": result["refresh_left"],
                }

        return SimpleNamespace(**result)

    # ----------------------------------------------------------------------
    # Session Invalidate
    # ----------------------------------------------------------------------
    @staticmethod
    def SessionInvalidate(session_token, invalidate_type):
        if CodeAuth._HasInitialized == False:
            raise Exception("CodeAuth have not been Initialized.")
        
        result = CodeAuth.request(
            CodeAuth._Endpoint, "/session/invalidate",
            {
                "project_id": CodeAuth._ProjectID,
                "session_token": session_token,
                "invalidate_type": invalidate_type,
            },
        )

        now = int(time.time() * 1000)

        if CodeAuth._UseCache and result.get("error") != "no_error":
            if CodeAuth._CacheTimestamp + CodeAuth._CacheDuration < now:
                CodeAuth._CacheTimestamp = now
                CodeAuth._CacheSession.clear()
            else:
                CodeAuth._CacheSession.pop(session_token, None)

        return SimpleNamespace(**result)

    # ----------------------------------------------------------------------
    # Internal request function
    # ----------------------------------------------------------------------
    @staticmethod
    def request(host, path, data, port=443, use_https=True):
        try:
            body = json.dumps(data)

            conn = http.client.HTTPSConnection(host, port)

            conn.request(
                "POST",
                path,
                body=body,
                headers={
                    "Content-Type": "application/json",
                    "Content-Length": str(len(body)),
                },
            )

            res = conn.getresponse()
            raw = res.read().decode("utf-8")
            conn.close()

            try:
                json_data = json.loads(raw)
            except:
                return {"error": "connection_error"}

            if res.status == 200:
                json_data["error"] = "no_error"

            return json_data

        except:
            return {"error": "connection_error"}
        