"""
Incremental AST Cache.

[20251216_FEATURE] Cache parsed ASTs and invalidate only affected files on changes.

This module extends the base AnalysisCache to provide AST-specific caching
with dependency tracking and cascading invalidation.
"""

from __future__ import annotations

import hashlib
import json
import logging
import pickle
from dataclasses import dataclass, field
from pathlib import Path
from typing import Any, Optional, Set

logger = logging.getLogger(__name__)


@dataclass
class CacheMetadata:
    """
    Metadata for cached AST entries.

    [20251216_FEATURE] Track file hash and dependencies
    """

    file_hash: str
    dependencies: Set[str] = field(default_factory=set)
    timestamp: float = 0.0
    language: str = "python"


class IncrementalASTCache:
    """
    [20251216_FEATURE] Incremental AST cache with dependency tracking.

    Purpose: Cache parsed ASTs and invalidate only affected files on changes.

    Features:
    - Disk-based caching with file hash keys
    - Dependency graph tracking for cascading invalidation
    - Cache persistence across server restarts
    - 40%+ reduction in re-parse time for unchanged files

    Example:
        >>> cache = IncrementalASTCache()
        >>> ast = cache.get_or_parse("file.py", "python")
        >>> cache.invalidate("file.py")  # Returns affected files
    """

    def __init__(self, cache_dir: str | Path = ".scalpel_ast_cache"):
        """
        Initialize the incremental AST cache.

        Args:
            cache_dir: Directory to store cache files

        [20251216_FEATURE] Cache initialization with dependency tracking
        """
        self.cache_dir = Path(cache_dir)
        self.cache_dir.mkdir(exist_ok=True, parents=True)

        # In-memory caches
        self.file_hashes: dict[str, str] = {}  # file -> hash
        self.ast_cache: dict[str, Any] = {}  # file -> AST
        self.dependency_graph: dict[str, Set[str]] = {}  # file -> dependencies

        # Load metadata from disk
        self._load_metadata()

    def _hash_file(self, file_path: str | Path) -> str:
        """
        Compute SHA256 hash of file contents.

        Args:
            file_path: Path to file

        Returns:
            SHA256 hash hex digest

        [20251216_FEATURE] File hash-based cache keys
        """
        path = Path(file_path)
        if not path.exists():
            return ""

        with open(path, "rb") as f:
            return hashlib.sha256(f.read()).hexdigest()

    def _cache_path(self, file_hash: str, language: str) -> Path:
        """
        Get cache file path for a given file hash.

        Args:
            file_hash: SHA256 hash of file contents
            language: Programming language

        Returns:
            Path to cache file

        [20251216_FEATURE] Hash-based cache file naming
        """
        return self.cache_dir / f"{file_hash}_{language}.ast"

    def _metadata_path(self) -> Path:
        """
        Get path to metadata file.

        Returns:
            Path to metadata JSON file

        [20251216_FEATURE] Metadata persistence
        """
        return self.cache_dir / "metadata.json"

    def _load_metadata(self) -> None:
        """
        Load cache metadata from disk.

        [20251216_FEATURE] Cache survives server restarts
        """
        metadata_path = self._metadata_path()
        if not metadata_path.exists():
            return

        try:
            with open(metadata_path, "r", encoding="utf-8") as f:
                metadata = json.load(f)

            self.file_hashes = metadata.get("file_hashes", {})

            # Reconstruct dependency graph (sets stored as lists in JSON)
            dep_graph = metadata.get("dependency_graph", {})
            self.dependency_graph = {k: set(v) for k, v in dep_graph.items()}

        except (json.JSONDecodeError, FileNotFoundError) as e:
            logger.warning(f"Failed to load cache metadata: {e}")
            self.file_hashes = {}
            self.dependency_graph = {}

    def _save_metadata(self) -> None:
        """
        Save cache metadata to disk.

        [20251216_FEATURE] Persist metadata for cache survival
        """
        try:
            # Convert sets to lists for JSON serialization
            dep_graph = {k: list(v) for k, v in self.dependency_graph.items()}

            metadata = {
                "file_hashes": self.file_hashes,
                "dependency_graph": dep_graph,
            }

            with open(self._metadata_path(), "w", encoding="utf-8") as f:
                json.dump(metadata, f, indent=2)

        except Exception as e:
            # [20240613_BUGFIX] Log full stack trace for cache metadata save failures
            logger.exception(f"Failed to save cache metadata: {e}")

    def get_or_parse(
        self, file_path: str | Path, language: str, parse_fn: Optional[Any] = None
    ) -> Any:
        """
        Get cached AST or parse fresh.

        Args:
            file_path: Path to source file
            language: Programming language
            parse_fn: Optional function to parse file if not cached

        Returns:
            Parsed AST

        [20251216_FEATURE] Cache-first AST retrieval
        """
        path = Path(file_path).resolve()
        path_str = str(path)

        # Compute file hash
        file_hash = self._hash_file(path)

        # Check memory cache first
        if path_str in self.ast_cache:
            cached_hash = self.file_hashes.get(path_str)
            if cached_hash == file_hash:
                return self.ast_cache[path_str]

        # Check disk cache
        cache_path = self._cache_path(file_hash, language)
        if cache_path.exists() and self.file_hashes.get(path_str) == file_hash:
            try:
                ast = self._load_ast(cache_path)
                self.ast_cache[path_str] = ast
                return ast
            except (pickle.UnpicklingError, EOFError, OSError) as e:
                logger.exception(f"Failed to load cached AST from {cache_path}: {e}")

        # Parse fresh
        if parse_fn is None:
            # Default parser based on language
            ast = self._parse_file(path, language)
        else:
            ast = parse_fn(path)

        # Cache the result
        self._save_ast(cache_path, ast)
        self.ast_cache[path_str] = ast
        self.file_hashes[path_str] = file_hash
        self._save_metadata()

        return ast

    def _parse_file(self, file_path: Path, language: str) -> Any:
        """
        Parse a file (default implementation).

        Args:
            file_path: Path to file
            language: Programming language

        Returns:
            Parsed AST

        [20251216_FEATURE] Default parsing logic
        """
        if language == "python":
            import ast

            with open(file_path, "r", encoding="utf-8") as f:
                return ast.parse(f.read(), filename=str(file_path))
        else:
            # For other languages, return a placeholder
            # Real implementation would use appropriate parsers
            return {"type": "Module", "language": language, "file": str(file_path)}

    def _load_ast(self, cache_path: Path) -> Any:
        """
        Load AST from cache file.

        Args:
            cache_path: Path to cache file

        Returns:
            Loaded AST

        [20251216_FEATURE] Pickle-based AST deserialization
        """
        with open(cache_path, "rb") as f:
            # [20251218_SECURITY] pickle.load safe here: internal cache only, no untrusted data (B301)
            return pickle.load(f)  # nosec B301

    def _save_ast(self, cache_path: Path, ast: Any) -> None:
        """
        Save AST to cache file.

        Args:
            cache_path: Path to cache file
            ast: AST to save

        [20251216_FEATURE] Pickle-based AST serialization
        """
        try:
            with open(cache_path, "wb") as f:
                pickle.dump(ast, f)
        except (pickle.PicklingError, OSError, TypeError) as e:
            # [20240613_BUGFIX] Restrict exception handler to expected pickle/file errors and log full traceback
            logger.exception(f"Failed to save AST to {cache_path}: {e}")

    def invalidate(self, file_path: str | Path) -> Set[str]:
        """
        Invalidate cache and return affected files.

        Args:
            file_path: Path to file that changed

        Returns:
            Set of file paths affected by the change

        [20251216_FEATURE] Cascading invalidation with dependency tracking
        """
        path = Path(file_path).resolve()
        path_str = str(path)

        # Invalidate this file
        self.file_hashes.pop(path_str, None)
        self.ast_cache.pop(path_str, None)

        # Find dependents
        affected = self._find_dependents(path_str)

        # Invalidate dependents
        for dep in affected:
            self.file_hashes.pop(dep, None)
            self.ast_cache.pop(dep, None)

        self._save_metadata()

        return affected

    def _find_dependents(self, file_path: str) -> Set[str]:
        """
        Find all files that depend on the given file.

        Args:
            file_path: Path to file

        Returns:
            Set of dependent file paths

        [20251216_FEATURE] Dependency graph traversal
        """
        dependents: Set[str] = set()

        # Find direct dependents
        for source, deps in self.dependency_graph.items():
            if file_path in deps:
                dependents.add(source)

        # Recursively find transitive dependents
        to_process = list(dependents)
        while to_process:
            current = to_process.pop()
            for source, deps in self.dependency_graph.items():
                if current in deps and source not in dependents:
                    dependents.add(source)
                    to_process.append(source)

        return dependents

    def record_dependency(self, source: str | Path, depends_on: str | Path) -> None:
        """
        Record a dependency between files.

        Args:
            source: Source file path
            depends_on: Dependency file path

        [20251216_FEATURE] Track dependency graph for cascading invalidation
        """
        source_path = str(Path(source).resolve())
        dep_path = str(Path(depends_on).resolve())

        if source_path not in self.dependency_graph:
            self.dependency_graph[source_path] = set()

        self.dependency_graph[source_path].add(dep_path)
        self._save_metadata()

    def get_cache_stats(self) -> dict[str, Any]:
        """
        Get cache statistics.

        Returns:
            Dictionary with cache stats

        [20251216_FEATURE] Cache observability
        """
        total_files = len(self.file_hashes)
        memory_cached = len(self.ast_cache)

        # Count disk cache files
        disk_files = len(list(self.cache_dir.glob("*.ast")))

        return {
            "total_tracked_files": total_files,
            "memory_cached_asts": memory_cached,
            "disk_cached_files": disk_files,
            "dependency_edges": sum(
                len(deps) for deps in self.dependency_graph.values()
            ),
            "cache_dir": str(self.cache_dir),
        }

    def clear_cache(self) -> None:
        """
        Clear all cache data.

        [20251216_FEATURE] Cache reset capability
        """
        self.file_hashes.clear()
        self.ast_cache.clear()
        self.dependency_graph.clear()

        # Remove cache files
        for cache_file in self.cache_dir.glob("*.ast"):
            cache_file.unlink()

        # Remove metadata
        metadata_path = self._metadata_path()
        if metadata_path.exists():
            metadata_path.unlink()


# [20251216_FEATURE] Global cache instance
_global_ast_cache: Optional[IncrementalASTCache] = None


def get_ast_cache(cache_dir: str | Path = ".scalpel_ast_cache") -> IncrementalASTCache:
    """
    Get the global AST cache instance.

    Args:
        cache_dir: Directory to store cache files

    Returns:
        Global IncrementalASTCache instance

    [20251216_FEATURE] Singleton cache accessor
    """
    global _global_ast_cache
    if _global_ast_cache is None:
        _global_ast_cache = IncrementalASTCache(cache_dir)
    return _global_ast_cache
