"""
Command-line interface for pycallgraph-visualizer
"""

import argparse
import os
import sys
import webbrowser
from pathlib import Path
from .analyzer import analyze_directory, generate_html_graph


def main():
    """Main entry point for the CLI"""
    parser = argparse.ArgumentParser(
        prog='pycallgraph',
        description='Visualize Python code dependencies with interactive call graphs',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  # Analyze current directory
  pycallgraph
  
  # Analyze specific directory
  pycallgraph /path/to/project
  
  # Specify output file
  pycallgraph -o my_graph.html
  
  # Don't open browser automatically
  pycallgraph --no-browser
  
  # Increase complexity threshold
  pycallgraph --complexity-threshold 15
        """
    )
    
    parser.add_argument(
        'directory',
        nargs='?',
        default='.',
        help='Directory to analyze (default: current directory)'
    )
    
    parser.add_argument(
        '-o', '--output',
        default='call_graph.html',
        help='Output HTML file name (default: call_graph.html)'
    )
    
    parser.add_argument(
        '--no-browser',
        action='store_true',
        help="Don't open the browser automatically"
    )
    
    parser.add_argument(
        '--complexity-threshold',
        type=int,
        default=10,
        help='Complexity threshold for flagging functions (default: 10)'
    )
    
    parser.add_argument(
        '--include-tests',
        action='store_true',
        help='Include test files in analysis'
    )
    
    parser.add_argument(
        '-v', '--verbose',
        action='store_true',
        help='Verbose output'
    )
    
    parser.add_argument(
        '--version',
        action='version',
        version='%(prog)s 1.0.0'
    )
    
    args = parser.parse_args()
    
    # Resolve directory path
    directory = os.path.abspath(args.directory)
    
    if not os.path.exists(directory):
        print(f"❌ Error: Directory not found: {directory}", file=sys.stderr)
        sys.exit(1)
    
    if not os.path.isdir(directory):
        print(f"❌ Error: Path is not a directory: {directory}", file=sys.stderr)
        sys.exit(1)
    
    # Check if there are any Python files
    has_python_files = any(
        f.endswith('.py') 
        for root, _, files in os.walk(directory) 
        for f in files
    )
    
    if not has_python_files:
        print(f"❌ Error: No Python files found in: {directory}", file=sys.stderr)
        sys.exit(1)
    
    print("🔍 PyCallGraph Visualizer")
    print("=" * 60)
    print(f"📁 Analyzing: {directory}")
    print()
    
    try:
        # Analyze the directory
        analysis_result = analyze_directory(
            directory
        )
        
        if analysis_result["total_functions"] == 0:
            print("⚠️  No Python functions found in the directory.")
            print("   Make sure the directory contains .py files with function definitions.")
            sys.exit(1)
        
        # Generate output file path
        if os.path.isabs(args.output):
            output_file = args.output
        else:
            output_file = os.path.join(directory, args.output)
        
        # Generate HTML graph
        generate_html_graph(analysis_result, output_file, directory)
        
        # Print summary
        print()
        print("=" * 60)
        print("📊 Analysis Complete!")
        print("=" * 60)
        print(f"📈 Total Functions: {analysis_result['total_functions']}")
        print(f"📁 Total Files: {analysis_result['total_files']}")
        print(f"🔗 Total Connections: {sum(len(v) for v in analysis_result['call_graph'].values())}")
        print()
        
        if analysis_result['circular_dependencies']:
            print(f"🔴 Circular Dependencies: {len(analysis_result['circular_dependencies'])}")
        else:
            print("✅ No circular dependencies found")
        
        if analysis_result['dead_functions']:
            print(f"💀 Dead Functions: {len(analysis_result['dead_functions'])}")
        else:
            print("✅ No dead code found")
        
        if analysis_result['high_complexity_functions']:
            print(f"📈 High Complexity Functions: {len(analysis_result['high_complexity_functions'])}")
        else:
            print("✅ All functions have reasonable complexity")

        print(f"🚪 Entry Points: {len(analysis_result['entry_points'])}")
        print()
        print(f"📄 Output: {output_file}")
        
        # Open browser
        if not args.no_browser:
            print("🌐 Opening in browser...")
            webbrowser.open('file://' + os.path.abspath(output_file))
        
        print()
        print("💡 Tips:")
        print("   - Use search to filter functions")
        print("   - Click function names to focus on their dependencies")
        print("   - Hover over nodes for detailed metrics")
        print("   - Red = Circular deps, Gray = Dead code, Orange = Complex, Green = Entry points")
        
    except Exception as e:
        print(f"❌ Error during analysis: {e}", file=sys.stderr)
        if args.verbose:
            import traceback
            traceback.print_exc()
        sys.exit(1)


if __name__ == '__main__':
    main()