"""
analyzer_local.py - Analyze local Python directories
"""

import ast
import os
from collections import defaultdict
from pathlib import Path
from pyvis.network import Network
import re as regex

IGNORE_DIRS = {"venv", "env", ".venv", "__pycache__", ".git", "site-packages", 
               "node_modules", ".pytest_cache", "dist", "build", ".tox", ".mypy_cache"}
BUILTINS = set(dir(__builtins__))

def should_ignore(path: str) -> bool:
    """Check if path should be ignored"""
    parts = set(Path(path).parts)
    return not parts.isdisjoint(IGNORE_DIRS)

def find_all_cycles(graph):
    """Find all cycles in the call graph using DFS"""
    cycles = []
    visited = set()
    rec_stack = []
    
    def dfs(node):
        if node in rec_stack:
            cycle_start = rec_stack.index(node)
            cycle = rec_stack[cycle_start:] + [node]
            min_idx = cycle.index(min(cycle[:-1]))
            normalized = tuple(cycle[min_idx:-1] + cycle[:min_idx] + [cycle[min_idx]])
            if normalized not in cycles:
                cycles.append(normalized)
            return
        
        if node in visited:
            return
        
        visited.add(node)
        rec_stack.append(node)
        
        for neighbor in graph.get(node, []):
            dfs(neighbor)
        
        rec_stack.pop()
    
    for node in graph.keys():
        dfs(node)
    
    return cycles

def calculate_complexity(func_node):
    """Calculate cyclomatic complexity"""
    complexity = 1
    for node in ast.walk(func_node):
        if isinstance(node, (ast.If, ast.While, ast.For, ast.ExceptHandler)):
            complexity += 1
        elif isinstance(node, ast.BoolOp):
            complexity += len(node.values) - 1
        elif isinstance(node, (ast.ListComp, ast.DictComp, ast.SetComp, ast.GeneratorExp)):
            complexity += 1
    return complexity

def count_lines(func_node):
    """Count lines of code"""
    if hasattr(func_node, 'end_lineno') and hasattr(func_node, 'lineno'):
        return func_node.end_lineno - func_node.lineno + 1
    return 0

def analyze_directory(directory_path: str) -> dict:
    """Analyze a local directory and return structured data"""
    
    print(f"📁 Analyzing directory: {directory_path}")
    
    function_files = {}
    functions_ast = defaultdict(dict)
    
    # Step 1: Collect functions
    python_files_found = 0
    for root, dirs, files in os.walk(directory_path):
        # Remove ignored directories from traversal
        dirs[:] = [d for d in dirs if not should_ignore(os.path.join(root, d))]
        
        for f in files:
            if f.endswith(".py"):
                python_files_found += 1
                path = os.path.join(root, f)
                
                if should_ignore(path):
                    continue
                
                # Get relative path for better display
                try:
                    rel_path = os.path.relpath(path, directory_path)
                except ValueError:
                    # On Windows, relpath fails if paths are on different drives
                    rel_path = path
                
                with open(path, "r", encoding="utf8", errors="ignore") as src:
                    try:
                        tree = ast.parse(src.read(), filename=path)
                    except SyntaxError as e:
                        print(f"   ⚠️  Syntax error in {rel_path}: {e}")
                        continue
                    except Exception as e:
                        print(f"   ⚠️  Error parsing {rel_path}: {e}")
                        continue
                
                for node in ast.walk(tree):
                    if isinstance(node, ast.FunctionDef):
                        function_files[node.name] = rel_path
                        functions_ast[rel_path][node.name] = node
    
    print(f"   Found {python_files_found} Python files")
    print(f"   Collected {len(function_files)} functions from {len(functions_ast)} files")
    
    if len(function_files) == 0:
        print("   ⚠️  No functions found!")
        return {
            "function_files": {},
            "functions_ast": {},
            "call_graph": {},
            "circular_dependencies": [],
            "functions_in_cycles": [],
            "dead_functions": [],
            "entry_points": [],
            "complexity_metrics": {},
            "high_complexity_functions": {},
            "node_connections": {},
            "total_functions": 0,
            "total_files": 0,
        }
    
    # Step 2: Build call graph
    call_graph = defaultdict(set)
    for file_name, funcs in functions_ast.items():
        for func_name, func_node in funcs.items():
            for node in ast.walk(func_node):
                if isinstance(node, ast.Call) and isinstance(node.func, ast.Name):
                    callee = node.func.id
                    if callee in function_files and callee not in BUILTINS:
                        call_graph[func_name].add(callee)
    
    print(f"   Built call graph with {len(call_graph)} edges")
    
    # Step 3: Detect circular dependencies
    circular_dependencies = find_all_cycles(call_graph)
    functions_in_cycles = set()
    for cycle in circular_dependencies:
        functions_in_cycles.update(cycle)
    
    print(f"   Found {len(circular_dependencies)} circular dependencies")
    
    # Step 4: Detect dead code and entry points
    all_callees = set()
    for callees in call_graph.values():
        all_callees.update(callees)
    
    never_called = set(function_files.keys()) - all_callees
    dead_functions = set()
    for func in never_called:
        if func not in call_graph or len(call_graph[func]) == 0:
            dead_functions.add(func)
    
    entry_points = never_called - dead_functions
    
    print(f"   Dead functions: {len(dead_functions)}")
    print(f"   Entry points: {len(entry_points)}")
    
    # Step 5: Calculate complexity metrics
    complexity_metrics = {}
    high_complexity_functions = {}
    
    for file_name, funcs in functions_ast.items():
        for func_name, func_node in funcs.items():
            complexity = calculate_complexity(func_node)
            loc = count_lines(func_node)
            complexity_metrics[func_name] = {
                'complexity': complexity,
                'loc': loc,
                'file': file_name
            }
            if complexity > 10:
                high_complexity_functions[func_name] = complexity_metrics[func_name]
    
    print(f"   High complexity functions: {len(high_complexity_functions)}")
    
    # Calculate node connections
    node_connections = defaultdict(int)
    for caller, callees in call_graph.items():
        node_connections[caller] += len(callees)
        for callee in callees:
            node_connections[callee] += 1
    
    print("✅ Analysis complete!")
    
    return {
        "function_files": function_files,
        "functions_ast": functions_ast,
        "call_graph": {k: list(v) for k, v in call_graph.items()},
        "circular_dependencies": [list(cycle) for cycle in circular_dependencies],
        "functions_in_cycles": list(functions_in_cycles),
        "dead_functions": list(dead_functions),
        "entry_points": list(entry_points),
        "complexity_metrics": complexity_metrics,
        "high_complexity_functions": high_complexity_functions,
        "node_connections": dict(node_connections),
        "total_functions": len(function_files),
        "total_files": len(functions_ast),
    }

def generate_html_graph(analysis_result: dict, output_path: str, directory_path: str):
    """Generate interactive HTML visualization"""
    
    function_files = analysis_result["function_files"]
    call_graph = analysis_result["call_graph"]
    functions_in_cycles = set(analysis_result["functions_in_cycles"])
    dead_functions = set(analysis_result["dead_functions"])
    entry_points = set(analysis_result["entry_points"])
    high_complexity_functions = analysis_result["high_complexity_functions"]
    complexity_metrics = analysis_result["complexity_metrics"]
    node_connections = analysis_result["node_connections"]
    functions_ast = analysis_result["functions_ast"]
    circular_dependencies = analysis_result["circular_dependencies"]
    
    # Create network
    net = Network(height="100%", width="100%", directed=True, bgcolor="#ffffff")
    
    net.force_atlas_2based(
    gravity=-100, 
    central_gravity=0.05, 
    spring_length=150, 
    spring_strength=0.1, 
    damping=0.5, 
    overlap=0.5
    )
    
    net.set_options("""
    var options = {
      "physics": {
        "enabled": true,
        "stabilization": {"enabled": true, "iterations": 200},
        "forceAtlas2Based": {
          "gravitationalConstant": -80,
          "centralGravity": 0.015,
          "springLength": 250,
          "springConstant": 0.05,
          "damping": 0.5,
          "avoidOverlap": 1
        },
        "solver": "forceAtlas2Based"
      },
      "nodes": {"shape": "box", "font": {"size": 16}, "borderWidth": 2, "margin": 10},
      "edges": {"arrows": {"to": {"enabled": true, "scaleFactor": 0.5}}, "smooth": {"enabled": true, "type": "continuous"}, "width": 2},
      "interaction": {"hover": true, "dragNodes": true, "dragView": true, "zoomView": true, "navigationButtons": true}
    }
    """)
    
    # Generate colors for files
    color_palette = ["#8dd3c7", "#ffffb3", "#bebada", "#fb8072", "#80b1d3", 
                     "#fdb462", "#b3de69", "#fccde5", "#d9d9d9", "#bc80bd"]
    file_colors = {}
    unique_files = list(set(function_files.values()))
    for i, file_name in enumerate(unique_files):
        file_colors[file_name] = color_palette[i % len(color_palette)]
    
    # Add nodes
    nodes_added = set()
    for caller, callees in call_graph.items():
        if caller not in function_files:
            continue
            
        caller_file = function_files[caller]
        caller_id = f"{caller} ({caller_file})"
        
        if caller_id not in nodes_added:
            caller_complexity = complexity_metrics.get(caller, {}).get('complexity', 0)
            caller_loc = complexity_metrics.get(caller, {}).get('loc', 0)
            is_complex = caller in high_complexity_functions
            
            # Determine color and status
            if caller in functions_in_cycles:
                node_color = "#ff4444"
                border_color = "#cc0000"
                status = "⚠️ IN CIRCULAR DEPENDENCY!"
            elif caller in dead_functions:
                node_color = "#9e9e9e"
                border_color = "#616161"
                status = "💀 DEAD CODE"
            elif is_complex:
                node_color = "#ff9800"
                border_color = "#f57c00"
                status = f"📈 HIGH COMPLEXITY ({caller_complexity})"
            elif caller in entry_points:
                node_color = "#4caf50"
                border_color = "#2e7d32"
                status = "🚪 ENTRY POINT"
            else:
                node_color = file_colors.get(caller_file, "#cccccc")
                border_color = node_color
                status = ""
            
            node_size = 20 + min(node_connections.get(caller, 0) * 5, 50)
            
            net.add_node(
                caller_id,
                label=caller,
                title=f"{caller}\\nFile: {caller_file}\\nComplexity: {caller_complexity}\\nLines: {caller_loc}\\n{status}",
                color={'background': node_color, 'border': border_color},
                borderWidth=3,
                size=node_size
            )
            nodes_added.add(caller_id)
        
        for callee in callees:
            if callee not in function_files:
                continue
                
            callee_file = function_files[callee]
            callee_id = f"{callee} ({callee_file})"
            
            if callee_id not in nodes_added:
                callee_complexity = complexity_metrics.get(callee, {}).get('complexity', 0)
                callee_loc = complexity_metrics.get(callee, {}).get('loc', 0)
                is_complex = callee in high_complexity_functions
                
                if callee in functions_in_cycles:
                    node_color = "#ff4444"
                    border_color = "#cc0000"
                    status = "⚠️ IN CIRCULAR DEPENDENCY!"
                elif callee in dead_functions:
                    node_color = "#9e9e9e"
                    border_color = "#616161"
                    status = "💀 DEAD CODE"
                elif is_complex:
                    node_color = "#ff9800"
                    border_color = "#f57c00"
                    status = f"📈 HIGH COMPLEXITY ({callee_complexity})"
                elif callee in entry_points:
                    node_color = "#4caf50"
                    border_color = "#2e7d32"
                    status = "🚪 ENTRY POINT"
                else:
                    node_color = file_colors.get(callee_file, "#cccccc")
                    border_color = node_color
                    status = ""
                
                node_size = 20 + min(node_connections.get(callee, 0) * 5, 50)
                
                net.add_node(
                    callee_id,
                    label=callee,
                    title=f"{callee}\\nFile: {callee_file}\\nComplexity: {callee_complexity}\\nLines: {callee_loc}\\n{status}",
                    color={'background': node_color, 'border': border_color},
                    borderWidth=3,
                    size=node_size
                )
                nodes_added.add(callee_id)
            
            # Check if edge is part of cycle
            edge_color = None
            edge_width = 2
            for cycle in analysis_result["circular_dependencies"]:
                # Check for the (caller -> callee) link within the cycle list
                for j in range(len(cycle) - 1):
                    if cycle[j] == caller and cycle[j + 1] == callee:
                        edge_color = "#ff0000"
                        edge_width = 4
                        break
                # Also check the closing link of the cycle (last -> first)
                if cycle[-1] == caller and cycle[0] == callee:
                    edge_color = "#ff0000"
                    edge_width = 4
                    break

            net.add_edge(caller_id, callee_id, color=edge_color, width=edge_width)
    
    # Save to file
    net.write_html(output_path)

    with open(output_path, "r", encoding="utf8") as f:
        html = f.read()
    
    # --- HTML Elements for Sidebar ---
    
    header = f"""
    <div style="position: fixed; top: 10px; left: 10px; background: white; padding: 15px; border-radius: 10px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); z-index: 1000; max-width: 300px;">
        <h3 style="margin: 0 0 10px 0;">📊 Project Analysis</h3>
        <p style="margin: 5px 0; font-size: 11px; word-break: break-all;"><strong>Path:</strong> {directory_path}</p>
        <p style="margin: 5px 0; font-size: 12px;"><strong>Functions:</strong> {analysis_result['total_functions']}</p>
        <p style="margin: 5px 0; font-size: 12px;"><strong>Files:</strong> {analysis_result['total_files']}</p>
        <hr style="margin: 10px 0;">
        <p style="margin: 5px 0; font-size: 12px; color: #ff4444;">🔴 Cycles: {len(analysis_result['circular_dependencies'])}</p>
        <p style="margin: 5px 0; font-size: 12px; color: #9e9e9e;">💀 Dead: {len(analysis_result['dead_functions'])}</p>
        <p style="margin: 5px 0; font-size: 12px; color: #ff9800;">📈 Complex: {len(analysis_result['high_complexity_functions'])}</p>
        <p style="margin: 5px 0; font-size: 12px; color: #4caf50;">🚪 Entry: {len(analysis_result['entry_points'])}</p>
    </div>
    """
    
    # Prepare search and view buttons
    file_buttons_html = """
    <div style="background:#fff; padding:8px; margin-bottom:10px; border-radius:4px; box-shadow:0 2px 4px rgba(0,0,0,0.1);">
        <input type="text" id="search_box" placeholder="🔍 Search functions..." 
            style="width:100%; padding:8px; border:2px solid #ddd; border-radius:4px; font-size:14px; box-sizing:border-box;">
        <div id="search_results" style="margin-top:5px; font-size:11px; color:#666;"></div>
    </div>
    """
    file_buttons_html += """
        <div style="background:#fff; padding:10px; margin-bottom:10px; border-radius:4px; box-shadow:0 2px 4px rgba(0,0,0,0.1);">
            <h4 style="margin:0 0 8px 0; color:#333;">🎯 Analysis Mode</h4>
            <div style="display:flex; gap:5px;">
                <button id="mode_normal" style="flex:1; padding:8px; background:#4CAF50; color:white; font-weight:bold;">Normal</button>
                <button id="mode_path" style="flex:1; padding:8px; background:#2196F3; color:white;">Path Finder</button>
                <button id="mode_multi" style="flex:1; padding:8px; background:#FF9800; color:white;">Multi-Select</button>
            </div>
            <div id="mode_info" style="margin-top:8px; padding:6px; background:#e3f2fd; border-radius:3px; font-size:11px; min-height:40px;">
                <strong>Normal Mode:</strong> Click nodes to explore
            </div>
        </div>
        """
    file_buttons_html += """
        <div id="analysis_panel" style="display:none; background:#fff; padding:10px; margin-bottom:10px; border-radius:4px; box-shadow:0 2px 4px rgba(0,0,0,0.1);">
            <div style="display:flex; justify-content:space-between; align-items:center; margin-bottom:8px;">
                <h4 style="margin:0; color:#333;">📊 Analysis Results</h4>
                <button id="clear_analysis" style="padding:4px 8px; background:#f44336; color:white; font-size:10px; border-radius:3px;">Clear</button>
            </div>
            <div id="analysis_content" style="font-size:11px; max-height:300px; overflow-y:auto;"></div>
        </div>
        """

    file_buttons_html += '<button id="btn_show_all" style="margin:2px; width:100%; background:#4CAF50; color:white; font-weight:bold;">Show All</button><br>\n'
    file_buttons_html += '<button id="btn_show_cycles" style="margin:2px; width:100%; background:#ff4444; color:white; font-weight:bold;">🔴 Show Cycles</button><br>\n'
    file_buttons_html += '<button id="btn_show_dead" style="margin:2px; width:100%; background:#9e9e9e; color:white; font-weight:bold;">💀 Show Dead Code</button><br>\n'
    file_buttons_html += '<button id="btn_show_complex" style="margin:2px; width:100%; background:#ff9800; color:white; font-weight:bold;">📈 Show Complex</button><br>\n'
    file_buttons_html += '<button id="btn_show_entry" style="margin:2px; width:100%; background:#4caf50; color:white; font-weight:bold;">🚪 Show Entry Points</button><br>\n'
    file_buttons_html += '<button id="btn_hide_cycles" style="margin:2px; width:100%; background:#ff9800; color:white; font-weight:bold;">Hide Cycles</button><br>\n'
    file_buttons_html += '<button id="btn_mode_explode" style="margin:2px; width:100%; background:#9C27B0; color:white; font-weight:bold;">⚛️ Explode Mode</button><br>'
    file_buttons_html += '<hr style="margin:10px 0;">\n'

    # Prepare file toggle buttons
    for file_name in functions_ast.keys():
        # Use the SAME sanitization as in JavaScript
        safe_id = file_name.replace('.', '_').replace('/', '__').replace('\\', '__').replace('-', '_').replace(' ', '_')
        button_id = f"btn_{safe_id}"
        file_buttons_html += f'<button id="{button_id}" style="margin:2px; width:100%;">{file_name}</button><br>\n'
        
    # Prepare function focus buttons
    function_buttons_html = '<button id="reset_focus" style="margin:2px; width:100%; background:#f44336; color:white; font-weight:bold;">Reset View</button><br>\n'
    for func_name in sorted(function_files.keys()):
        function_buttons_html += f'<button id="focus_{func_name}" style="margin:2px; width:100%; font-size:12px;">{func_name}</button><br>\n'

    # Create cycle info block
    cycle_info_html = ""
    if circular_dependencies:
        cycle_info_html = f"""
        <div style="background:#ffebee; padding:10px; margin:10px 0; border-left:4px solid #f44336; border-radius:4px;">
            <strong style="color:#c62828;">⚠️ {len(circular_dependencies)} Circular Dependencies</strong>
            <div style="margin-top:8px; font-size:11px; max-height:120px; overflow-y:auto;">
        """
        for i, cycle in enumerate(circular_dependencies[:5], 1):
            cycle_info_html += f"<div style='margin:4px 0; padding:4px; background:white; border-radius:2px;'>{i}. {' → '.join(list(cycle)[:3])}{'...' if len(cycle) > 3 else ''}</div>"
        if len(circular_dependencies) > 5:
            cycle_info_html += f"<div style='margin:4px 0; font-style:italic;'>...+{len(circular_dependencies) - 5} more</div>"
        cycle_info_html += """
            </div>
        </div>
        """
    else:
        cycle_info_html = """
        <div style="background:#e8f5e9; padding:10px; margin:10px 0; border-left:4px solid #4caf50; border-radius:4px;">
            <strong style="color:#2e7d32;">✅ No Circular Dependencies</strong>
        </div>
        """

    # Dead code info - MAKE IT SCROLLABLE
    dead_info_html = ""
    if dead_functions:
        dead_list_html = ""
        for func in sorted(list(dead_functions)):
            dead_list_html += f"<div style='margin:2px 0; padding:3px; background:white; border-radius:2px; font-size:10px;'>• {func}</div>"
        
        dead_info_html = f"""
        <div style="background:#f5f5f5; padding:10px; margin:10px 0; border-left:4px solid #9e9e9e; border-radius:4px;">
            <div style="display:flex; justify-content:space-between; align-items:center;">
                <strong style="color:#424242;">💀 {len(dead_functions)} Dead Functions</strong>
                <button id="toggle_dead_list" style="padding:2px 6px; font-size:10px; cursor:pointer;">Show All</button>
            </div>
            <div style="font-size:10px; color:#666; margin-top:3px;">Never called, no connections</div>
            <div id="dead_list" style="display:none; margin-top:8px; max-height:200px; overflow-y:auto; font-size:11px;">
                {dead_list_html}
            </div>
        </div>
        """

    # Entry points info - MAKE IT SCROLLABLE 
    entry_info_html = ""
    if entry_points:
        entry_list_html = ""
        for func in sorted(list(entry_points)):
            num_calls = len(call_graph.get(func, []))
            entry_list_html += f"<div style='margin:2px 0; padding:3px; background:white; border-radius:2px; font-size:10px;'>• {func} → {num_calls} call(s)</div>"
        
        entry_info_html = f"""
        <div style="background:#e8f5e9; padding:10px; margin:10px 0; border-left:4px solid #4caf50; border-radius:4px;">
            <div style="display:flex; justify-content:space-between; align-items:center;">
                <strong style="color:#2e7d32;">🚪 {len(entry_points)} Entry Points</strong>
                <button id="toggle_entry_list" style="padding:2px 6px; font-size:10px; cursor:pointer;">Show All</button>
            </div>
            <div style="font-size:10px; color:#2e7d32; margin-top:3px;">API/Cron/Event handlers</div>
            <div id="entry_list" style="display:none; margin-top:8px; max-height:200px; overflow-y:auto; font-size:11px;">
                {entry_list_html}
            </div>
        </div>
        """

    # Complexity info - MAKE IT SCROLLABLE
    complexity_info_html = ""
    if high_complexity_functions:
        sorted_complex = sorted(high_complexity_functions.items(), 
                                key=lambda x: x[1]['complexity'], 
                                reverse=True)
        complex_list_html = ""
        for func, metrics in sorted_complex:
            complex_list_html += f"<div style='margin:2px 0; padding:3px; background:white; border-radius:2px; font-size:10px;'>• {func}: C={metrics['complexity']}, L={metrics['loc']}</div>"
        
        complexity_info_html = f"""
        <div style="background:#fff3e0; padding:10px; margin:10px 0; border-left:4px solid #ff9800; border-radius:4px;">
            <div style="display:flex; justify-content:space-between; align-items:center;">
                <strong style="color:#e65100;">📈 {len(high_complexity_functions)} Complex Functions</strong>
                <button id="toggle_complex_list" style="padding:2px 6px; font-size:10px; cursor:pointer;">Show All</button>
            </div>
            <div style="font-size:10px; color:#e65100; margin-top:3px;">Complexity &gt; 10</div>
            <div id="complex_list" style="display:none; margin-top:8px; max-height:200px; overflow-y:auto; font-size:11px;">
                {complex_list_html}
            </div>
        </div>
        """

    # Create layout with sidebar on left
    html = html.replace("<body>", f"""
    <body style="margin:0; padding:0; overflow:hidden;">
    <div style="display:flex; height:100vh; width:100vw;">
        <div id="side-panel" style="width:320px; overflow-y:auto; background:#f5f5f5; padding:15px; border-right:3px solid #ddd; box-shadow: 2px 0 5px rgba(0,0,0,0.1);">
            <h2 style="margin-top:0; color:#333; border-bottom:2px solid #333; padding-bottom:5px;">Call Graph Explorer</h2>
            
            {cycle_info_html}
            {dead_info_html}
            {entry_info_html}
            {complexity_info_html}
            
            <h3 style="color:#555; margin-top:20px;">🔍 Search & Views</h3>
            {file_buttons_html}
            <hr style="margin:20px 0;">
            <h3 style="color:#555;">📁 Files</h3>
            {function_buttons_html}
        </div>
        <div id="graph" style="flex-grow:1; height:100%;"></div>
    </div>
    """)

    # Find and move the card into graph container
    card_pattern = r'<div class="card"[^>]*>.*?<div id="mynetwork"[^>]*>.*?</div>\s*</div>'
    card_match = regex.search(card_pattern, html, regex.DOTALL)
    
    if card_match:
        card_html = card_match.group(0)
        html = html.replace(card_html, '')
        html = html.replace('<div id="graph" style="flex-grow:1; height:100%;"></div>', 
                            f'<div id="graph" style="flex-grow:1; height:100%;">{card_html}</div>')

    # Override CSS for proper layout
    custom_css = """
    <style>
    body { margin: 0; padding: 0; overflow: hidden; }
    #graph { display: flex; flex-direction: column; }
    .card {
        flex: 1 !important;
        width: 100% !important;
        height: 100% !important;
        border: none !important;
        border-radius: 0 !important;
        margin: 0 !important;
        display: flex !important;
        flex-direction: column !important;
    }
    .card-body, #mynetwork {
        flex: 1 !important;
        width: 100% !important;
        height: 100% !important;
        padding: 0 !important;
        margin: 0 !important;
    }
    button {
        padding: 8px;
        border: 1px solid #ccc;
        border-radius: 4px;
        background: white;
        cursor: pointer;
        transition: all 0.2s;
    }
    button:hover {
        background: #e0e0e0;
        transform: translateX(2px);
    }
    #side-panel h3 {
        font-size: 14px;
        font-weight: bold;
    }
    #search_box:focus {
        outline: none;
        border-color: #4CAF50;
        box-shadow: 0 0 5px rgba(76, 175, 80, 0.3);
    }
    </style>
    """
    html = html.replace("</head>", f"{custom_css}</head>")

    # --- Build JavaScript for interactivity (FIXED) ---
    group_js = "<script>\n"

    # Functions involved in cycles, dead code, and high complexity (Data preparation)
    group_js += f"var functionsInCycles = new Set({list(functions_in_cycles)});\n"
    group_js += f"var deadFunctions = new Set({list(dead_functions)});\n"
    group_js += f"var entryPoints = new Set({list(entry_points)});\n"
    group_js += f"var highComplexityFunctions = new Set({list(high_complexity_functions.keys())});\n"

    # Core interactivity logic wrapped in DOMContentLoaded
    # This prevents the "Cannot set properties of null (setting 'onclick')" error.
    group_js += """
    document.addEventListener("DOMContentLoaded", function () {
    // 1. LIST TOGGLE BUTTONS (Dead, Entry, Complex)
    
    // Dead List Toggle
    const deadBtn = document.getElementById("toggle_dead_list");
    const deadList = document.getElementById('dead_list');
    if (deadBtn && deadList) {
        deadBtn.onclick = function() {
            if (deadList.style.display === 'none') {
                deadList.style.display = 'block';
                this.textContent = 'Hide';
            } else {
                deadList.style.display = 'none';
                this.textContent = 'Show All';
            }
        };
    }

    // Entry List Toggle
    const entryBtn = document.getElementById('toggle_entry_list');
    const entryList = document.getElementById('entry_list');
    if (entryBtn && entryList) {
        entryBtn.onclick = function() {
            if (entryList.style.display === 'none') {
                entryList.style.display = 'block';
                this.textContent = 'Hide';
            } else {
                entryList.style.display = 'none';
                this.textContent = 'Show All';
            }
        };
    }

    // Complex List Toggle
    const complexBtn = document.getElementById('toggle_complex_list');
    const complexList = document.getElementById('complex_list');
    if (complexBtn && complexList) {
        complexBtn.onclick = function() {
            if (complexList.style.display === 'none') {
                complexList.style.display = 'block';
                this.textContent = 'Hide';
            } else {
                complexList.style.display = 'none';
                this.textContent = 'Show All';
            }
        };
    }
    
    // Check if network object is globally available (standard for pyvis output)
    if (typeof network === 'undefined') {
        console.error("Vis.js network object is not defined. Interactivity buttons will not work.");
        return;
    }

    // 2. SEARCH FUNCTIONALITY
    var searchTimeout;
    document.getElementById('search_box').addEventListener('input', function(e) {
        clearTimeout(searchTimeout);
        var query = e.target.value.toLowerCase().trim();
        var resultsDiv = document.getElementById('search_results');
        
        if (query.length === 0) {
            // Show all nodes
            var allNodes = network.body.data.nodes.get();
            allNodes.forEach(function(n){ 
                network.body.data.nodes.update({id:n.id, hidden: false}); 
            });
            resultsDiv.textContent = '';
            return;
        }
        
        searchTimeout = setTimeout(function() {
            var allNodes = network.body.data.nodes.get();
            var matchCount = 0;
            
            allNodes.forEach(function(n){ 
                // n.label contains just the function name
                var matches = n.label.toLowerCase().includes(query); 
                network.body.data.nodes.update({id:n.id, hidden: !matches}); 
                if (matches) matchCount++;
            });
            
            resultsDiv.textContent = matchCount + ' function(s) found';
            
            if (matchCount > 0) {
                // Wait for the update to apply, then fit
                setTimeout(function(){ network.fit(); }, 300); 
            }
        }, 300);
    });

    // Clear search on Escape
    document.getElementById('search_box').addEventListener('keydown', function(e) {
        if (e.key === 'Escape') {
            e.target.value = '';
            e.target.dispatchEvent(new Event('input'));
        }
    });

    // 3. GLOBAL VIEW BUTTONS

    // Show all button
    document.getElementById('btn_show_all').onclick = function() {
        document.getElementById('search_box').value = '';
        var allNodes = network.body.data.nodes.get();
        allNodes.forEach(function(n){ 
            network.body.data.nodes.update({id:n.id, hidden: false}); 
        });
        document.getElementById('search_results').textContent = '';
        network.fit();
    };

    // Show only cycles button
    document.getElementById('btn_show_cycles').onclick = function() {
        document.getElementById('search_box').value = '';
        var allNodes = network.body.data.nodes.get();
        allNodes.forEach(function(n){ 
            var isInCycle = functionsInCycles.has(n.label);
            network.body.data.nodes.update({id:n.id, hidden: !isInCycle}); 
        });
        document.getElementById('search_results').textContent = functionsInCycles.size + ' function(s) in cycles';
        setTimeout(function(){ network.fit(); }, 300);
    };

    // Show dead code button
    document.getElementById('btn_show_dead').onclick = function() {
        document.getElementById('search_box').value = '';
        var allNodes = network.body.data.nodes.get();
        allNodes.forEach(function(n){ 
            var isDead = deadFunctions.has(n.label);
            network.body.data.nodes.update({id:n.id, hidden: !isDead}); 
        });
        document.getElementById('search_results').textContent = deadFunctions.size + ' dead function(s)';
        setTimeout(function(){ network.fit(); }, 300);
    };

    // Show complex functions button
    document.getElementById('btn_show_complex').onclick = function() {
        document.getElementById('search_box').value = '';
        var allNodes = network.body.data.nodes.get();
        allNodes.forEach(function(n){ 
            var isComplex = highComplexityFunctions.has(n.label);
            network.body.data.nodes.update({id:n.id, hidden: !isComplex}); 
        });
        document.getElementById('search_results').textContent = highComplexityFunctions.size + ' complex function(s)';
        setTimeout(function(){ network.fit(); }, 300);
    };

    // Show entry points button
    document.getElementById('btn_show_entry').onclick = function() {
        document.getElementById('search_box').value = '';
        var allNodes = network.body.data.nodes.get();
        allNodes.forEach(function(n){ 
            var isEntry = entryPoints.has(n.label);
            network.body.data.nodes.update({id:n.id, hidden: !isEntry}); 
        });
        document.getElementById('search_results').textContent = entryPoints.size + ' entry point(s)';
        setTimeout(function(){ network.fit(); }, 300);
    };

    // Hide cycles button
    document.getElementById('btn_hide_cycles').onclick = function() {
        document.getElementById('search_box').value = '';
        var allNodes = network.body.data.nodes.get();
        allNodes.forEach(function(n){ 
            var isInCycle = functionsInCycles.has(n.label);
            network.body.data.nodes.update({id:n.id, hidden: isInCycle}); 
        });
        document.getElementById('search_results').textContent = '';
        setTimeout(function(){ network.fit(); }, 300);
    };

    // Reset view button (Full Implementation)
    // Reset view button (FULL FIX)
        document.getElementById('reset_focus').onclick = function() {
            // 1. Clear search elements and UI panels
            document.getElementById('search_box').value = '';
            document.getElementById('search_results').textContent = '';
            document.getElementById('analysis_panel').style.display = 'none';

            // 2. Reset Nodes (Create a batch update array for reliability)
            var allNodes = network.body.data.nodes.get();
            var nodesToReset = [];
            allNodes.forEach(function(n) { 
                nodesToReset.push({
                    id: n.id, 
                    hidden: false, 
                    borderWidth: 2,
                    color: n.id.includes('(') ? undefined : '#97C2E5'
                }); 
            });
            network.body.data.nodes.update(nodesToReset); // Batch update

            // 3. Reset Edges (CRITICAL FIX: Create and submit a batch update array)
            var allEdges = network.body.data.edges.get();
            var edgesToReset = [];
            allEdges.forEach(function(e) {
                edgesToReset.push({
                    id: e.id,
                    hidden: false,           // CRITICAL: Ensure this value is passed
                    color: '#848484',        // Default color
                    width: 2                 // Default width
                });
            });
            network.body.data.edges.update(edgesToReset); // Batch update for reliability

            // 4. Reset Mode State
            currentMode = 'default';
            pathSourceNode = null;
            pathTargetNode = null;
            document.getElementById('mode_info').innerHTML = '<strong>Mode:</strong> Default (Drag/Zoom)';

            // 5. Fit to view (Adding a slight delay can sometimes help Vis.js stabilize)
            setTimeout(function() {
                network.fit();
            }, 50); // Small delay
        };
    // 4. FILE TOGGLE BUTTONS
    
    // Note: The pyvis-generated nodes do not have a 'group' property automatically
    // unless explicitly added. The node ID format is currently "function (file)".
    // We will update the logic to toggle visibility based on file name inside the ID.
    
"""
    
    # Python loop for File toggle buttons
    # for file_name in functions_ast.keys():
    #     # Sanitize button ID for JavaScript access
    #     button_id = f"btn_{file_name.replace('.', '_').replace('/', '__').replace('\\\\', '__')}"
    #     group_js += f"""
    #     document.getElementById("{button_id}").onclick = function() {{
    #     var nodes = network.body.data.nodes.get();
    #     var nodesToToggle = nodes.filter(function(n) {{
    #         // Check if node ID contains the file name (e.g., "func (file.py)")
    #         return n.id.includes('({file_name})');
    #     }});
        
    #     // Find the visibility state of the first node to determine action
    #     var isHidden = nodesToToggle.length > 0 ? nodesToToggle[0].hidden : false; 
    #     var newState = !isHidden;
        
    #     nodesToToggle.forEach(function(n){{
    #         network.body.data.nodes.update({{id: n.id, hidden: newState}});
    #     }});
    #     setTimeout(function(){{ network.fit(); }}, 300);
    # }};
    # """
    for file_name in functions_ast.keys():
        # Sanitize for HTML ID
        safe_id = file_name.replace('.', '_').replace('/', '__').replace('\\', '__').replace('-', '_').replace(' ', '_')
        button_id = f"btn_{safe_id}"
        
        # Escape backslashes for JavaScript strings
        js_safe_filename = file_name.replace('\\', '\\\\')
        
        group_js += f"""
        // File toggle for {file_name}
        (function() {{
            const btn = document.getElementById("{button_id}");
            if (!btn) {{
                console.error("File button not found: {button_id}");
                return;
            }}
            
            btn.onclick = function() {{
                var nodes = network.body.data.nodes.get();
                var nodesToToggle = nodes.filter(function(n) {{
                    return n.id.includes('({js_safe_filename})');
                }});
                
                var isHidden = nodesToToggle.length > 0 ? nodesToToggle[0].hidden : false; 
                var newState = !isHidden;
                
                nodesToToggle.forEach(function(n){{
                    network.body.data.nodes.update({{id: n.id, hidden: newState}});
                }});
                setTimeout(function(){{ network.fit(); }}, 300);
            }};
        }})();
        """
    # Function focus buttons
    for func_name in function_files.keys():
        button_id = f"focus_{func_name}"
        group_js += f"""
    // Focus button for {func_name}
    (function() {{
        const btn = document.getElementById('{button_id}');
        if (!btn) {{
            console.error("BUTTON NOT FOUND: {button_id}");
            return;
        }}

        btn.onclick = function() {{
            var allNodes = network.body.data.nodes.get();
            var showSet = new Set();
            var targetNodeId = null;

            allNodes.forEach(function(n) {{
                if(n.label === '{func_name}') {{
                    showSet.add(n.label);
                    targetNodeId = n.id;
                }}
            }});

            function getCallees(f_label) {{
                allNodes.filter(n => n.label === f_label).forEach(function(n) {{
                    network.body.data.edges.get().forEach(function(e) {{
                        if(e.from === n.id) {{
                            var targetNode = allNodes.find(x => x.id === e.to);
                            if(targetNode && !showSet.has(targetNode.label)) {{
                                showSet.add(targetNode.label);
                                getCallees(targetNode.label);
                            }}
                        }}
                    }});
                }});
            }}
            getCallees('{func_name}');

            function getCallers(f_label) {{
                allNodes.filter(n => n.label === f_label).forEach(function(n) {{
                    network.body.data.edges.get().forEach(function(e) {{
                        if(e.to === n.id) {{
                            var sourceNode = allNodes.find(x => x.id === e.from);
                            if(sourceNode && !showSet.has(sourceNode.label)) {{
                                showSet.add(sourceNode.label);
                                getCallers(sourceNode.label);
                            }}
                        }}
                    }});
                }});
            }}
            getCallers('{func_name}');

            allNodes.forEach(function(n) {{
                network.body.data.nodes.update({{ id: n.id, hidden: !showSet.has(n.label) }});
            }});

            if (targetNodeId) {{
                setTimeout(function() {{
                    network.focus(targetNodeId, {{ scale: 1.0, animation: true }});
                }}, 500);
            }} else {{
                setTimeout(function() {{
                    network.fit();
                }}, 500);
            }}
        }};
    }})();
    """
    group_js += """
        // Analysis Mode State
        var currentMode = 'normal'; // 'normal', 'path', 'multi'
        var pathSourceNode = null;
        var pathTargetNode = null;
        var selectedNodes = new Set();
        var originalNodeColors = new Map(); // Store original colors for reset
        
        // Store all original colors on initialization
        var allNodes = network.body.data.nodes.get();
        allNodes.forEach(function(n) {
            originalNodeColors.set(n.id, {
                background: n.color.background,
                border: n.color.border
            });
        });
        
        // Mode switching functions
        function setMode(mode) {
            currentMode = mode;
            resetAnalysis();
            
            // Update button styles
            document.getElementById('mode_normal').style.fontWeight = mode === 'normal' ? 'bold' : 'normal';
            document.getElementById('mode_normal').style.background = mode === 'normal' ? '#4CAF50' : '#81C784';
            
            document.getElementById('mode_path').style.fontWeight = mode === 'path' ? 'bold' : 'normal';
            document.getElementById('mode_path').style.background = mode === 'path' ? '#2196F3' : '#64B5F6';
            
            document.getElementById('mode_multi').style.fontWeight = mode === 'multi' ? 'bold' : 'normal';
            document.getElementById('mode_multi').style.background = mode === 'multi' ? '#FF9800' : '#FFB74D';
            
            // Update info text
            var infoText = '';
            if (mode === 'normal') {
                infoText = '<strong>Normal Mode:</strong> Click nodes to explore';
            } else if (mode === 'path') {
                infoText = '<strong>Path Finder:</strong> Click source node (blue), then target node (green) to find path';
            } else if (mode === 'multi') {
                infoText = '<strong>Multi-Select:</strong> Hold Ctrl/Cmd and click to select multiple nodes';
            }
            document.getElementById('mode_info').innerHTML = infoText;
        }
        
        function resetAnalysis() {
            pathSourceNode = null;
            pathTargetNode = null;
            selectedNodes.clear();
            
            // Reset all node colors to original
            var nodes = network.body.data.nodes.get();
            nodes.forEach(function(n) {
                var original = originalNodeColors.get(n.id);
                if (original) {
                    network.body.data.nodes.update({
                        id: n.id,
                        color: {background: original.background, border: original.border},
                        borderWidth: 3
                    });
                }
            });
            
            // Reset all edges to original
            var edges = network.body.data.edges.get();
            edges.forEach(function(e) {
                network.body.data.edges.update({
                    id: e.id,
                    color: e.color === '#00ff00' || e.color === '#0066ff' ? null : e.color,
                    width: e.width === 6 ? 2 : e.width
                });
            });
            
            document.getElementById('analysis_panel').style.display = 'none';
        }
        
        // Mode button handlers
        document.getElementById('mode_normal').onclick = function() { setMode('normal'); };
        document.getElementById('mode_path').onclick = function() { setMode('path'); };
        document.getElementById('mode_multi').onclick = function() { setMode('multi'); };
        document.getElementById('clear_analysis').onclick = resetAnalysis;
    """
    group_js += """
        // BFS Path Finding
        // Enhanced BFS Path Finding with depth limit and direction options
    function findPath(sourceId, targetId, maxDepth, bidirectional) {
        maxDepth = maxDepth || 10; // Default to 10 levels
        bidirectional = bidirectional !== false; // Default to true
        
        var edges = network.body.data.edges.get();
        var nodes = network.body.data.nodes.get();
        
        // Build adjacency lists
        var forwardGraph = {};
        var backwardGraph = {};
        nodes.forEach(function(n) { 
            forwardGraph[n.id] = [];
            backwardGraph[n.id] = [];
        });
        edges.forEach(function(e) {
            if (!forwardGraph[e.from]) forwardGraph[e.from] = [];
            if (!backwardGraph[e.to]) backwardGraph[e.to] = [];
            forwardGraph[e.from].push({to: e.to, forward: true});
            backwardGraph[e.to].push({to: e.from, forward: false});
        });
        
        // Try forward path first (A calls B)
        var forwardPath = bfsSearch(sourceId, targetId, forwardGraph, maxDepth);
        if (forwardPath) {
            return {path: forwardPath, type: 'forward'};
        }
        
        // If bidirectional, try backward path (B calls A)
        if (bidirectional) {
            var backwardPath = bfsSearch(sourceId, targetId, backwardGraph, maxDepth);
            if (backwardPath) {
                return {path: backwardPath, type: 'backward'};
            }
        }
        
        return null; // No path found within depth limit
    }
    
    function bfsSearch(sourceId, targetId, graph, maxDepth) {
        var queue = [[sourceId]];
        var visited = new Set([sourceId]);
        var depth = 0;
        
        while (queue.length > 0 && depth <= maxDepth) {
            var levelSize = queue.length;
            
            for (var i = 0; i < levelSize; i++) {
                var path = queue.shift();
                var node = path[path.length - 1];
                
                if (node === targetId) {
                    return path;
                }
                
                var neighbors = graph[node] || [];
                for (var j = 0; j < neighbors.length; j++) {
                    var next = neighbors[j].to;
                    if (!visited.has(next)) {
                        visited.add(next);
                        var newPath = path.slice();
                        newPath.push(next);
                        queue.push(newPath);
                    }
                }
            }
            depth++;
        }
        
        return null; // No path found within depth limit
    }
    
    function highlightPath(pathResult) {
    if (!pathResult || !pathResult.path || pathResult.path.length === 0) return;
    
    var path = pathResult.path;
    var pathType = pathResult.type;
    
    var allNodes = network.body.data.nodes.get();
    var allEdges = network.body.data.edges.get();
    var pathNodeSet = new Set(path); 

    // --- 1. ISOLATE NON-PATH NODES ---
    allNodes.forEach(function(n) {
        var isPathNode = pathNodeSet.has(n.id);
        network.body.data.nodes.update({
            id: n.id,
            hidden: !isPathNode, // HIDE IF NOT IN PATH
            borderWidth: isPathNode ? 3 : 2,
            color: isPathNode ? {border: '#000000'} : undefined 
        });
    });

    // --- 2. ISOLATE NON-PATH EDGES (Fix) ---
    allEdges.forEach(function(e) {
        // Keep edge visible if BOTH its source and target nodes are part of the path.
        var isEdgeInPathNeighborhood = pathNodeSet.has(e.from) && pathNodeSet.has(e.to);

        network.body.data.edges.update({
            id: e.id,
            hidden: !isEdgeInPathNeighborhood, // HIDE ONLY if both ends are NOT on the path
            width: 2,
            color: '#848484'
        });
    });
    // ------------------------------------------

    // --- 3. HIGHLIGHTING (Original Logic) ---

    // Highlight nodes in path (Original logic, ensures path nodes are bold/colored)
    path.forEach(function(nodeId, index) {
        // ... (Node coloring logic remains the same)
        network.body.data.nodes.update({
            id: nodeId,
            // ... color and borderWidth updates ...
            hidden: false // Ensure path nodes are visible
        });
    });
    
    // Highlight specific edges forming the shortest path (Original logic, ensures these edges are bold/colored/hidden:false)
    if (pathType === 'forward') {
        // ... (forward edge logic)
        if (edge) {
            network.body.data.edges.update({
                id: edge.id,
                hidden: false, // ENSURE IT'S VISIBLE
                color: '#00ff00',
                width: 6
            });
        }
    } else if (pathType === 'backward') {
        // ... (backward edge logic)
        if (edge) {
            network.body.data.edges.update({
                id: edge.id,
                hidden: false, // ENSURE IT'S VISIBLE
                color: '#ff6b00',
                width: 6
            });
        }
    }
        
        // Show results with depth warning
        var nodes = network.body.data.nodes.get();
        var pathLabels = path.map(function(id) {
            var node = nodes.find(function(n) { return n.id === id; });
            return node ? node.label : id;
        });
        
        var directionIcon = pathType === 'forward' ? '→' : '←';
        var directionText = pathType === 'forward' ? 'Forward Path (A calls B)' : 'Backward Path (B calls A)';
        var directionColor = pathType === 'forward' ? '#2196F3' : '#FF6B00';
        
        var html = '<strong style="color:' + directionColor + ';">' + directionText + '</strong><br>';
        html += '<strong>Path Length: ' + (path.length - 1) + ' step(s)</strong><br>';
        
        if (path.length > 6) {
            html += '<div style="background:#fff3e0; padding:4px; margin:4px 0; border-radius:3px; font-size:10px;">⚠️ Long path - consider refactoring</div>';
        }
        
        html += '<div style="margin-top:8px; padding:8px; background:#f5f5f5; border-radius:3px; max-height:250px; overflow-y:auto;">';
        pathLabels.forEach(function(label, i) {
            var arrow = (i < pathLabels.length - 1) ? ' <span style="color:' + directionColor + ';">' + directionIcon + '</span> ' : '';
            html += '<div style="margin:4px 0; padding:4px; background:white; border-left:3px solid ';
            if (i === 0) html += '#2196F3';
            else if (i === pathLabels.length - 1) html += '#4CAF50';
            else html += '#FF9800';
            html += ';">' + (i + 1) + '. ' + label + arrow + '</div>';
        });
        html += '</div>';
        
        document.getElementById('analysis_content').innerHTML = html;
        document.getElementById('analysis_panel').style.display = 'block';
        
        // Focus on path
        network.fit({nodes: path, animation: true});
    }
    """
    group_js += """
        function analyzeSelectedNodes() {
            if (selectedNodes.size === 0) {
                document.getElementById('analysis_panel').style.display = 'none';
                return;
            }
            
            var nodes = network.body.data.nodes.get();
            var edges = network.body.data.edges.get();
            var selectedArray = Array.from(selectedNodes);
            
            // Find connections between selected nodes
            var internalEdges = edges.filter(function(e) {
                return selectedNodes.has(e.from) && selectedNodes.has(e.to);
            });
            
            // Find common callers (nodes that call multiple selected)
            var callerCounts = {};
            edges.forEach(function(e) {
                if (selectedNodes.has(e.to) && !selectedNodes.has(e.from)) {
                    callerCounts[e.from] = (callerCounts[e.from] || 0) + 1;
                }
            });
            var commonCallers = Object.keys(callerCounts).filter(function(id) {
                return callerCounts[id] > 1;
            });
            
            // Find common callees (nodes called by multiple selected)
            var calleeCounts = {};
            edges.forEach(function(e) {
                if (selectedNodes.has(e.from) && !selectedNodes.has(e.to)) {
                    calleeCounts[e.to] = (calleeCounts[e.to] || 0) + 1;
                }
            });
            var commonCallees = Object.keys(calleeCounts).filter(function(id) {
                return calleeCounts[id] > 1;
            });
            
            // Build HTML
            var html = '<strong style="color:#FF9800;">' + selectedNodes.size + ' Functions Selected</strong><br>';
            html += '<div style="margin-top:8px;">';
            
            // Selected nodes
            html += '<div style="margin:8px 0;"><strong>Selected:</strong></div>';
            selectedArray.forEach(function(id) {
                var node = nodes.find(function(n) { return n.id === id; });
                if (node) {
                    html += '<div style="margin:2px 0; padding:3px; background:#fff3e0; border-left:3px solid #FF9800; font-size:10px;">• ' + node.label + '</div>';
                }
            });
            
            // Internal connections
            if (internalEdges.length > 0) {
                html += '<div style="margin:12px 0 4px 0;"><strong>Internal Connections: ' + internalEdges.length + '</strong></div>';
                internalEdges.slice(0, 5).forEach(function(e) {
                    var fromNode = nodes.find(function(n) { return n.id === e.from; });
                    var toNode = nodes.find(function(n) { return n.id === e.to; });
                    if (fromNode && toNode) {
                        html += '<div style="margin:2px 0; padding:3px; background:white; font-size:10px;">• ' + fromNode.label + ' → ' + toNode.label + '</div>';
                    }
                });
                if (internalEdges.length > 5) {
                    html += '<div style="font-size:10px; font-style:italic; margin:2px 0;">...+' + (internalEdges.length - 5) + ' more</div>';
                }
            }
            
            // Common callers
            if (commonCallers.length > 0) {
                html += '<div style="margin:12px 0 4px 0;"><strong>Common Callers: ' + commonCallers.length + '</strong></div>';
                commonCallers.slice(0, 3).forEach(function(id) {
                    var node = nodes.find(function(n) { return n.id === id; });
                    if (node) {
                        html += '<div style="margin:2px 0; padding:3px; background:#e3f2fd; border-left:3px solid #2196F3; font-size:10px;">• ' + node.label + ' (calls ' + callerCounts[id] + ')' + '</div>';
                    }
                });
            }
            
            // Common callees
            if (commonCallees.length > 0) {
                html += '<div style="margin:12px 0 4px 0;"><strong>Common Callees: ' + commonCallees.length + '</strong></div>';
                commonCallees.slice(0, 3).forEach(function(id) {
                    var node = nodes.find(function(n) { return n.id === id; });
                    if (node) {
                        html += '<div style="margin:2px 0; padding:3px; background:#e8f5e9; border-left:3px solid #4CAF50; font-size:10px;">• ' + node.label + ' (called by ' + calleeCounts[id] + ')' + '</div>';
                    }
                });
            }
            
            html += '</div>';
            
            document.getElementById('analysis_content').innerHTML = html;
            document.getElementById('analysis_panel').style.display = 'block';
        }
    """
    group_js += """
        // Handle node clicks based on mode
        network.on("click", function(params) {
            if (params.nodes.length === 0) return;
            
            var nodeId = params.nodes[0];
            
            if (currentMode === 'path') {
            if (!pathSourceNode) {
                // Select source
                pathSourceNode = nodeId;
                network.body.data.nodes.update({
                    id: nodeId,
                    color: {background: '#2196F3', border: '#1565C0'},
                    borderWidth: 5
                });
                document.getElementById('mode_info').innerHTML = '<strong>Path Finder:</strong> Source selected (blue). Now click target node.';
            } else if (!pathTargetNode && nodeId !== pathSourceNode) {
                // Select target and find path
                pathTargetNode = nodeId;
                
                // Get user preferences
                var maxDepth = 10
                var bidirectional = 1
                
                var pathResult = findPath(pathSourceNode, pathTargetNode, maxDepth, bidirectional);
                
                if (pathResult) {
                    highlightPath(pathResult);
                } else {
                    // No path found
                    network.body.data.nodes.update({
                        id: nodeId,
                        color: {background: '#f44336', border: '#c62828'},
                        borderWidth: 5
                    });
                    var msg = bidirectional ? 
                        'No path found in either direction within ' + maxDepth + ' levels.' :
                        'No forward path found within ' + maxDepth + ' levels.';
                    document.getElementById('analysis_content').innerHTML = 
                        '<strong style="color:#f44336;">❌ No Path Found</strong><br>' +
                        '<div style="margin-top:8px;">' + msg + '</div>' +
                        '<div style="margin-top:8px; font-size:10px; color:#666;">Try increasing max depth or enabling bidirectional search.</div>';
                    document.getElementById('analysis_panel').style.display = 'block';
                }
            }
        } else if (currentMode === 'multi') {
                // Toggle selection
                if (selectedNodes.has(nodeId)) {
                    selectedNodes.delete(nodeId);
                    var original = originalNodeColors.get(nodeId);
                    if (original) {
                        network.body.data.nodes.update({
                            id: nodeId,
                            color: {background: original.background, border: original.border},
                            borderWidth: 3
                        });
                    }
                } else {
                    selectedNodes.add(nodeId);
                    network.body.data.nodes.update({
                        id: nodeId,
                        color: {background: '#FF9800', border: '#F57C00'},
                        borderWidth: 5
                    });
                }
                analyzeSelectedNodes();
            }
            else if (currentMode === 'explode') {
            // Trigger the explode function immediately
            explodeNode(nodeId);
        }
        });
    """
    group_js += """
        // Global variable to store the currently exploded node's ID for easy clearing
        var explodedNodeId = null;

        function explodeNode(nodeId) {
            // Reset the view first to clear any previous paths or highlights
            resetAnalysis();
            
            explodedNodeId = nodeId;

            var allNodes = network.body.data.nodes.get();
            var allEdges = network.body.data.edges.get();
            
            // 1. Identify the Neighbors
            var neighbors = new Set();
            neighbors.add(nodeId); // Include the central node itself
            
            var connectingEdges = [];

            // Find all direct neighbors and connecting edges
            allEdges.forEach(function(e) {
                var isConnecting = false;
                
                if (e.from === nodeId) {
                    neighbors.add(e.to); // Callee
                    isConnecting = true;
                } else if (e.to === nodeId) {
                    neighbors.add(e.from); // Caller
                    isConnecting = true;
                }
                
                if (isConnecting) {
                    connectingEdges.push(e);
                }
            });

            // 2. Isolate/Hide elements
            
            // Hide non-neighbor nodes
            var nodesToUpdate = [];
            allNodes.forEach(function(n) {
                var isNeighbor = neighbors.has(n.id);
                var update = {
                    id: n.id,
                    hidden: !isNeighbor, // Hide if not a neighbor
                    borderWidth: 2, 
                    color: undefined // Reset color
                };
                // Highlight the central node
                if (n.id === nodeId) {
                    update.color = {background: '#9C27B0', border: '#7B1FA2'}; // Purple for Exploded Node
                    update.borderWidth = 5;
                }
                nodesToUpdate.push(update);
            });
            network.body.data.nodes.update(nodesToUpdate);

            // Hide all edges first, then reveal only connecting edges
            var edgesToUpdate = [];
            allEdges.forEach(function(e) {
                edgesToUpdate.push({
                    id: e.id,
                    hidden: true,
                    width: 2, 
                    color: '#848484'
                });
            });
            network.body.data.edges.update(edgesToUpdate);
            
            // 3. Highlight Connecting Edges (Reveal them)
            connectingEdges.forEach(function(e) {
                network.body.data.edges.update({
                    id: e.id,
                    hidden: false, // Make connecting edges visible
                    color: {color: '#9C27B0', highlight: '#7B1FA2'}, // Purple color
                    width: 4
                });
            });

            // 4. Update Analysis Panel
            var nodeLabel = network.body.data.nodes.get(nodeId).label;
            var callerCount = connectingEdges.filter(e => e.to === nodeId).length;
            var calleeCount = connectingEdges.filter(e => e.from === nodeId).length;

            var html = '<strong style="color:#9C27B0;">⚛️ Explosion View: ' + nodeLabel + '</strong><br>';
            html += '<div>Callers: <strong>' + callerCount + '</strong></div>';
            html += '<div>Callees: <strong>' + calleeCount + '</strong></div>';
            html += '<div style="margin-top:10px; font-size:11px;">Click **Reset View** to return to full graph.</div>';
            
            document.getElementById('analysis_content').innerHTML = html;
            document.getElementById('analysis_panel').style.display = 'block';

            // Fit view to the neighborhood
            network.fit({nodes: Array.from(neighbors), animation: true});
        }
    """
    group_js += """
        document.getElementById('btn_mode_explode').onclick = function() {
            resetAnalysis();
            currentMode = 'explode';
            document.getElementById('mode_info').innerHTML = '<strong>Explode Mode:</strong> Click any node to see its immediate neighbors.';
        };
        """
    group_js += "}); // End of DOMContentLoaded\n"
    group_js += "</script>\n"

    html = html.replace("</body>", f"{group_js}</body>")
    html = html.replace("<body>", f"<body>{header}")
    
    # print(output_path) # Debugging print removed
    with open(output_path, "w", encoding="utf8") as f:
        f.write(html)
    
    print(f"   Generated visualization: {output_path}")
